package com.elitesland.tw.tw5.server.prd.cal.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitesland.tw.tw5.api.prd.cal.payload.*;
import com.elitesland.tw.tw5.api.prd.cal.query.*;
import com.elitesland.tw.tw5.api.prd.cal.service.*;
import com.elitesland.tw.tw5.api.prd.cal.vo.*;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsDistributeVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalAccTurTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.CalAccTypeEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.PmsReasonTypeEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 核算-数据处理中间层
 *
 * @author carl
 * @date 2023-11-13
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CalResourceServiceImpl extends BaseServiceImpl implements CalResourceService {
    private final CalSettlePriceService calSettlePriceService;
    private final CalEqvaIncomeService calEqvaIncomeService;
    private final CalEqvaCostService calEqvaCostService;
    private final CacheUtil cacheUtil;
    private final CalAccountService calAccountService;
    private final CalAccountFreezeService calAccountFreezeService;
    private final CalAccountTurnoverService calAccountTurnoverService;


    @Override
    public BigDecimal querySettlePrice(CalResourcePriceQuery query) {
//        LocalDate finDate = query.getFinDate();
//        if (finDate == null) {
//            finDate = LocalDate.now();
//        }

//        int year = finDate.getYear();
//        int monthValue = finDate.getMonthValue();
        CalEqvaCostQuery costQuery = new CalEqvaCostQuery();
        costQuery.setBuId(query.getToBuId());
//        costQuery.setFinYear(year);
//        costQuery.setFinPeriod(monthValue);
        costQuery.setJobType1(query.getJobType1());
        costQuery.setJobType2(query.getJobType2());
        CalEqvaCostVO costView = calEqvaCostService.getEqvaCostPlus(costQuery);
        if (costView == null) {
            throw TwException.error("", "未找到符合条件的当量成本");
        }
        CalSettlePriceQuery settlePriceQuery = new CalSettlePriceQuery();
//        settlePriceQuery.setFinYear(year);
//        settlePriceQuery.setFinPeriod(monthValue);
        settlePriceQuery.setFromBuId(query.getFromBuId());
        settlePriceQuery.setToBuId(query.getToBuId());
        settlePriceQuery.setJobType1(query.getJobType1());
        settlePriceQuery.setJobType2(query.getJobType2());
        settlePriceQuery.setResId(query.getResId());
        settlePriceQuery.setProjId(query.getProjId());
        CalSettlePriceVO settlePricePlus = calSettlePriceService.getSettlePricePlus(settlePriceQuery);
        if (settlePricePlus == null) {
            throw TwException.error("", "未找到符合条件的结算定价");
        }
        BigDecimal eqvaCost = costView.getEqvaCost();//当量成本
        BigDecimal markupPrice = settlePricePlus.getMarkupPrice();//调整价格
        BigDecimal price = BigDecimal.valueOf(2000);
        if (settlePricePlus.getPriceStrategy() != null) {
            if (settlePricePlus.getPriceStrategy().equals("1")) {
                //mark up百分比
                price = eqvaCost.add(eqvaCost.multiply(markupPrice).divide(BigDecimal.valueOf(100), 4, RoundingMode.HALF_DOWN));
            }
            if (settlePricePlus.getPriceStrategy().equals("2")) {
                //mark up金额
                price = eqvaCost.add(markupPrice);
            }
            if (settlePricePlus.getPriceStrategy().equals("3")) {
                //绝对结算金额
                price = markupPrice;
            }
        }
        return price;
    }

    @Override
    public BigDecimal queryIncomePrice(CalResourcePriceQuery query) {
//        LocalDate finDate = query.getFinDate();
//        if (finDate == null) {
//            finDate = LocalDate.now();
//        }
        if (query.getResId() == null) {
            throw TwException.error("", "接收资源不可为空");
        }
//        int year = finDate.getYear();
//        int monthValue = finDate.getMonthValue();
        CalEqvaIncomeQuery eqvaIncomeQuery = new CalEqvaIncomeQuery();
//        eqvaIncomeQuery.setFinYear(year);
//        eqvaIncomeQuery.setFinPeriod(monthValue);
        eqvaIncomeQuery.setJobType1(query.getJobType1());
        eqvaIncomeQuery.setJobType2(query.getJobType2());
        eqvaIncomeQuery.setResId(query.getResId());
        eqvaIncomeQuery.setProjId(query.getProjId());

        //获取资源的组织信息
        PrdOrgEmployeeRefVO userDefaultOrg = cacheUtil.getUserDefaultOrg(query.getResId());
        if (userDefaultOrg == null) {
            throw TwException.error("", "接收资源不在组织体系内");
        }
        String cityLevel = "3";
        String extString5 = userDefaultOrg.getExtString5();
        if (StringUtils.hasText(extString5)) {
            PrdSystemSelectionVO prdSystemSelectionVO = cacheUtil.querySystemSelection("org:employee:serviceaddr", extString5);
            if (prdSystemSelectionVO != null) {
                cityLevel = prdSystemSelectionVO.getExtString1();
            }
        }
        eqvaIncomeQuery.setBuId(userDefaultOrg.getOrgId());
        eqvaIncomeQuery.setCoopType(userDefaultOrg.getCooperationMode());
        eqvaIncomeQuery.setCityLevel(cityLevel);

        CalEqvaIncomeVO eqvaSalary = calEqvaIncomeService.getEqvaSalary(eqvaIncomeQuery);
        if (eqvaSalary == null) {
            throw TwException.error("", "未找到符合条件的资源收入配置");
        }
        return eqvaSalary.getPreeqvaAmt();
    }


    @Override
    @Transactional(rollbackFor = Exception.class)
    public void taskSettleTurnover(List<CalTaskSettleVO> settleVOs) {
        log.info("开始处理结算单数据: {}", settleVOs.toString());
        List<CalAccountPayload> accountPayloads = new ArrayList<>();
        //查询相关账户
        settleVOs.forEach(settleVO -> {
            String auType = settleVO.getReasonType().equals(PmsReasonTypeEnum.PROJ_CONTRACT.getCode()) ? CalAccTypeEnum.PROJ.getCode() : settleVO.getReasonType().equals(PmsReasonTypeEnum.PROJ_OPPO.getCode()) ? CalAccTypeEnum.OPP.getCode() : CalAccTypeEnum.BU_PROJ.getCode();
            CalAccountPayload p1 = new CalAccountPayload();
            p1.setAuType(auType);
            p1.setAuId(settleVO.getProjId());
            accountPayloads.add(p1);
            CalAccountPayload p2 = new CalAccountPayload();
            p2.setAuType(CalAccTypeEnum.BU.getCode());
            p2.setAuId(settleVO.getResBuId());
            accountPayloads.add(p2);
            CalAccountPayload p3 = new CalAccountPayload();
            p3.setAuType(CalAccTypeEnum.RES.getCode());
            p3.setAuId(settleVO.getIncomeResId());
            accountPayloads.add(p3);
        });
        //获取所有相关账户数据
        List<CalAccountVO> calAccountVOS = queryAccounts(accountPayloads);
        CalResourcePayload resourcePayload = new CalResourcePayload();
        resourcePayload.setCalAccountVOS(calAccountVOS);
        settleVOs.forEach(settleVO -> {
            //数据处理
            operateTaskSettleTurnover(settleVO, resourcePayload);
        });
        //新增流水记录
        List<CalAccountTurnoverPayload> turnoverPayloads = resourcePayload.getTurnoverPayloads();
        //新增冻结记录
        List<CalAccountFreezePayload> freezePayloads = resourcePayload.getFreezePayloads();
        //待更新账户列表
        List<CalAccountVO> accountVOs = resourcePayload.getUpdateAccountVOs();

        log.info("变更后的账号数据: {}", accountVOs.toString());
        // 修改账户金额信息
        updateAccount(accountVOs);
        log.info("保存流水记录: {}", turnoverPayloads.toString());
        //保存流水数据
        calAccountTurnoverService.batchInsert(turnoverPayloads);
        //保存冻结记录数据
        if (freezePayloads.size() > 0) {
            calAccountFreezeService.batchInsert(freezePayloads);
        }

    }

    /**
     * 数据处理
     *
     * @param settleVO
     * @param resourcePayload
     */
    void operateTaskSettleTurnover(CalTaskSettleVO settleVO, CalResourcePayload resourcePayload) {
        /**
         * 流水记录
         * 1.项目-BU：项目扣减流水
         * 2.项目-BU：BU增加流水
         * 3.BU-资源：BU扣减流水
         * 4.BU-资源：资源增加流水
         *
         * 冻结记录
         * 5.生成资源冻结记录
         */
        //账户数据源
        List<CalAccountVO> calAccountVOS = resourcePayload.getCalAccountVOS();
        //新增流水记录
        List<CalAccountTurnoverPayload> turnoverPayloads = resourcePayload.getTurnoverPayloads();
        //新增冻结记录
        List<CalAccountFreezePayload> freezePayloads = resourcePayload.getFreezePayloads();
        //待更新账户列表
        List<CalAccountVO> accountVOs = resourcePayload.getUpdateAccountVOs();
        String auType = settleVO.getReasonType().equals(PmsReasonTypeEnum.PROJ_CONTRACT.getCode()) ? CalAccTypeEnum.PROJ.getCode() : settleVO.getReasonType().equals(PmsReasonTypeEnum.PROJ_OPPO.getCode()) ? CalAccTypeEnum.OPP.getCode() : CalAccTypeEnum.BU_PROJ.getCode();
        //项目账户
        CalAccountVO projAccountVO = calAccountVOS.stream().filter(vo -> vo.getAuType().equals(auType) && vo.getAuId().equals(settleVO.getProjId())).findFirst().get();
        accountVOs.add(projAccountVO);
        //1.项目-BU：项目扣减流水
        CalAccountTurnoverPayload projTurnoverPayload = getTaskTurnoverPayload(settleVO, projAccountVO, true, false);
        turnoverPayloads.add(projTurnoverPayload);
        //BU账户
        CalAccountVO buAccountVO = calAccountVOS.stream().filter(vo -> vo.getAuType().equals(CalAccTypeEnum.BU.getCode()) && vo.getAuId().equals(settleVO.getResBuId())).findFirst().get();
        accountVOs.add(buAccountVO);
        // 2.项目-BU：BU增加流水
        CalAccountTurnoverPayload buTurnoverPayload0 = getTaskTurnoverPayload(settleVO, buAccountVO, true, true);
        turnoverPayloads.add(buTurnoverPayload0);
        // 3.BU-资源：BU扣减流水
        CalAccountTurnoverPayload buTurnoverPayload1 = getTaskTurnoverPayload(settleVO, buAccountVO, false, false);
        turnoverPayloads.add(buTurnoverPayload1);
        //资源账户
        CalAccountVO resAccountVO = calAccountVOS.stream().filter(vo -> vo.getAuType().equals(CalAccTypeEnum.RES.getCode()) && vo.getAuId().equals(settleVO.getIncomeResId())).findFirst().get();
        accountVOs.add(resAccountVO);
        // 4.BU-资源：资源增加流水
        CalAccountTurnoverPayload resTurnoverPayload = getTaskTurnoverPayload(settleVO, resAccountVO, false, true);
        turnoverPayloads.add(resTurnoverPayload);
        //5.生成资源冻结记录
        CalAccountFreezePayload taskFreezePayload = getTaskFreezePayload(settleVO, resAccountVO, true);
        if (taskFreezePayload != null) {
            freezePayloads.add(taskFreezePayload);
        }

    }

    @Override
    public void unFreezeTaskSettle(List<CalTaskSettleVO> settleVOs) {
        //新增冻结记录
        List<CalAccountFreezePayload> freezePayloads = new ArrayList<>();
        //获取结算单收益人id，
        List<Long> resIds = settleVOs.stream().map(CalTaskSettleVO::getIncomeResId).distinct().collect(Collectors.toList());
        //获取收益人账户
        List<CalAccountVO> calAccountVOS = queryAccounts(CalAccTypeEnum.RES.getCode(), resIds);
        settleVOs.forEach(vo -> {
            Optional<CalAccountVO> first = calAccountVOS.stream().filter(calAccountVO -> calAccountVO.getAuId().equals(vo.getIncomeResId())).findFirst();
            if (first.isPresent()) {
                CalAccountFreezePayload taskFreezePayload = getTaskFreezePayload(vo, first.get(), false);
                if (taskFreezePayload != null) {
                    freezePayloads.add(taskFreezePayload);
                }

            } else {
                throw TwException.error("", "账户不存在");
            }
        });
        // 修改账户金额信息
        updateAccount(calAccountVOS);
        //保存冻结记录数据
        if (freezePayloads.size() > 0) {
            calAccountFreezeService.batchInsert(freezePayloads);
        }
    }


    @Override
    public void taskSettleWithdrawTurnover(CalTaskSettleWithdrawPayload payload) {
        log.info("开始处理结算单提现数据: {}", payload.toString());
        CalAccountVO calAccountVO = queryAccount(CalAccTypeEnum.RES.getCode(), payload.getWithdrawResId());
        log.info("提现前账户数据: {}", calAccountVO.toString());


        //账户总金额和账户总当量
        BigDecimal totalQty = calAccountVO.getTotalQty() == null ? BigDecimal.ZERO : calAccountVO.getTotalQty();
        BigDecimal totalAmt = calAccountVO.getTotalAmt() == null ? BigDecimal.ZERO : calAccountVO.getTotalAmt();
        //提现当量和金额
        BigDecimal withdrawQty = payload.getWithdrawQty().negate();
        BigDecimal withdrawAmt = payload.getWithdrawAmt().negate();
        //提现后数据
        BigDecimal avalQtyNew = totalQty.add(withdrawQty);
        BigDecimal avalAmtNew = totalAmt.add(withdrawAmt);

        LocalDate localDate = LocalDate.now();
        CalAccountTurnoverPayload turnoverPayload = new CalAccountTurnoverPayload();
        turnoverPayload.setAccId(calAccountVO.getId());
        turnoverPayload.setAccName(calAccountVO.getLedgerName());
        turnoverPayload.setFinYear(localDate.getYear());
        turnoverPayload.setFinPeriod(localDate.getMonthValue());
        turnoverPayload.setIoDate(localDate);
        turnoverPayload.setIoTime(LocalDateTime.now());
        turnoverPayload.setSourceType(CalAccTurTypeEnum.EQVA_WITHDRAW.getCode());
        turnoverPayload.setSourceId(payload.getWithdrawId());
        turnoverPayload.setSourceName(payload.getWithdrawNo());
        turnoverPayload.setTurnoverQty(withdrawQty);
        turnoverPayload.setTurnoverAmt(withdrawAmt);
        turnoverPayload.setBeforeQty(totalQty);
        turnoverPayload.setBeforeAmt(totalAmt);
        turnoverPayload.setAfterQty(avalQtyNew);
        turnoverPayload.setAfterAmt(avalAmtNew);

        //账户重新赋值
        //账户重新赋值
        calAccountVO.setTotalQty(avalQtyNew);
        calAccountVO.setTotalAmt(avalAmtNew);

        calAccountVO.setAvalQty(avalQtyNew.subtract(calAccountVO.getFrozenQty() == null ? BigDecimal.ZERO : calAccountVO.getFrozenQty()));
        calAccountVO.setAvalAmt(avalAmtNew.subtract(calAccountVO.getFrozenAmt() == null ? BigDecimal.ZERO : calAccountVO.getFrozenAmt()));
        log.info("提现后账户数据: {}", calAccountVO.toString());
        // 修改账户金额信息
        updateAccount(List.of(calAccountVO));
        //保存流水
        calAccountTurnoverService.batchInsert(List.of(turnoverPayload));
    }


    /**
     * 生成资源冻结或解冻记录
     *
     * @param settleVO     任务结算单
     * @param calAccountVO 结算账户
     * @param isAdd        冻结：true,解冻：false
     * @return
     */
    // @Override
    CalAccountFreezePayload getTaskFreezePayload(CalTaskSettleVO settleVO, CalAccountVO calAccountVO, Boolean isAdd) {

        BigDecimal graranteeEqva = isAdd ? settleVO.getGraranteeEqva() : settleVO.getGraranteeEqva().negate();
        if (graranteeEqva.compareTo(BigDecimal.ZERO) == 0) {
            //表示没有冻结或解冻操作
            return null;
        }
        BigDecimal graranteeAmt = isAdd ? settleVO.getGraranteeAmt() : settleVO.getGraranteeAmt().negate();
        BigDecimal avalQty = calAccountVO.getAvalQty() == null ? BigDecimal.ZERO : calAccountVO.getAvalQty();
        BigDecimal avalAmt = calAccountVO.getAvalAmt() == null ? BigDecimal.ZERO : calAccountVO.getAvalAmt();

        BigDecimal avalQtyNew = avalQty.subtract(graranteeEqva);
        BigDecimal avalAmtNew = avalAmt.subtract(graranteeAmt);
        CalAccountFreezePayload freezePayload = new CalAccountFreezePayload();
        freezePayload.setAccId(calAccountVO.getId());
        freezePayload.setAccName(calAccountVO.getLedgerName());
        freezePayload.setFinYear(settleVO.getSettleDate().getYear());
        freezePayload.setFinPeriod(settleVO.getSettleDate().getMonthValue());
        freezePayload.setIoDate(settleVO.getSettleDate());
        freezePayload.setIoTime(LocalDateTime.now());
        freezePayload.setSourceType(isAdd ? "1" : "2");
        freezePayload.setSourceId(settleVO.getId());
        freezePayload.setSourceName(settleVO.getSettleNo());
        freezePayload.setFreezeQty(graranteeEqva);
        freezePayload.setFreezeAmt(graranteeAmt);
        freezePayload.setEqvaPrice(settleVO.getEqvaSalary());
        freezePayload.setBeforeQty(avalQty);
        freezePayload.setBeforeAmt(avalAmt);
        freezePayload.setAfterQty(avalQtyNew);
        freezePayload.setAfterAmt(avalAmtNew);


        //账户重新赋值
        calAccountVO.setAvalQty(avalQtyNew);
        calAccountVO.setAvalAmt(avalAmtNew);

        calAccountVO.setFrozenQty(calAccountVO.getTotalQty().subtract(avalQtyNew));
        calAccountVO.setFrozenAmt(calAccountVO.getTotalAmt().subtract(avalAmtNew));
        return freezePayload;
    }

    /**
     * 生成任务流水payload
     *
     * @param settleVO     任务结算单
     * @param calAccountVO 结算账户
     * @param isSettle     计算结算相关数据：true,计算收入相关数据：false
     * @param isAdd        收入：true,支出：false
     * @return
     */
    CalAccountTurnoverPayload getTaskTurnoverPayload(CalTaskSettleVO settleVO, CalAccountVO calAccountVO, Boolean isSettle, Boolean isAdd) {
        LocalDate settleDate = settleVO.getSettleDate();
        //实际结算当量(.negate()判断加减)
        BigDecimal approveSettleEqva = isAdd ? settleVO.getApproveSettleEqva() : settleVO.getApproveSettleEqva().negate();
        //默认实际结算金额
        BigDecimal amt = isAdd ? settleVO.getApproveSettleAmt() : settleVO.getApproveSettleAmt().negate();
        //默认结算单价
        BigDecimal price = settleVO.getSettlePrice();
        if (!isSettle) {
            //收入金额
            amt = isAdd ? settleVO.getApproveIncomeAmt() : settleVO.getApproveIncomeAmt().negate();
            //支出及收入价格，使用从场景：1，bu和资源之间产生支出用的价格，2,资源收入价格
            price = settleVO.getEqvaSalary();
        }
        BigDecimal totalQty = calAccountVO.getTotalQty() == null ? BigDecimal.ZERO : calAccountVO.getTotalQty();
        BigDecimal totalAmt = calAccountVO.getTotalAmt() == null ? BigDecimal.ZERO : calAccountVO.getTotalAmt();
        BigDecimal totalQtyNew = totalQty.add(approveSettleEqva);
        BigDecimal totalAmtNew = totalAmt.add(amt);
        CalAccountTurnoverPayload turnoverPayload = new CalAccountTurnoverPayload();
        turnoverPayload.setAccId(calAccountVO.getId());
        turnoverPayload.setAccName(calAccountVO.getLedgerName());
        turnoverPayload.setFinYear(settleDate.getYear());
        turnoverPayload.setFinPeriod(settleDate.getMonthValue());
        turnoverPayload.setIoDate(settleDate);
        turnoverPayload.setIoTime(LocalDateTime.now());
        turnoverPayload.setSourceType(CalAccTurTypeEnum.TASK_SETT.getCode());
        turnoverPayload.setSourceId(settleVO.getId());
        turnoverPayload.setSourceName(settleVO.getSettleNo());
        turnoverPayload.setTurnoverQty(approveSettleEqva);
        turnoverPayload.setTurnoverAmt(amt);
        turnoverPayload.setEqvaPrice(price);
        turnoverPayload.setBeforeQty(totalQty);
        turnoverPayload.setBeforeAmt(totalAmt);
        turnoverPayload.setAfterQty(totalQtyNew);
        turnoverPayload.setAfterAmt(totalAmtNew);
        turnoverPayload.setRemark(settleVO.getRemark());
        //账户重新赋值
        calAccountVO.setTotalQty(totalQtyNew);
        calAccountVO.setTotalAmt(totalAmtNew);
        calAccountVO.setAvalQty(totalQtyNew.subtract(calAccountVO.getFrozenQty() == null ? BigDecimal.ZERO : calAccountVO.getFrozenQty()));
        calAccountVO.setAvalAmt(totalAmtNew.subtract(calAccountVO.getFrozenAmt() == null ? BigDecimal.ZERO : calAccountVO.getFrozenAmt()));
        return turnoverPayload;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void wideSettleTurnover(CalWideSettleVO calWideSettleVO) {
        /**
         * 流水记录
         * 1.支出核单元：扣减流水
         * 2.收入核单元：增加流水
         */
        //新增流水记录
        List<CalAccountTurnoverPayload> turnoverPayloads = new ArrayList<>();
        //账户列表
        List<CalAccountVO> accountVOs = new ArrayList<>();
        //支出核单元账户
        CalAccountVO fromAccountVO = queryAccount(calWideSettleVO.getFromSourceType(), calWideSettleVO.getFromSourceId());
        accountVOs.add(fromAccountVO);
        //1.支出核单元：扣减流水
        CalAccountTurnoverPayload fromTurnoverPayload = getWideTurnoverPayload(calWideSettleVO, fromAccountVO, false);
        turnoverPayloads.add(fromTurnoverPayload);
        //收入核单元
        CalAccountVO toAccountVO = queryAccount(calWideSettleVO.getToSourceType(), calWideSettleVO.getToSourceId());
        accountVOs.add(toAccountVO);
        //2.收入核单元：增加流水
        CalAccountTurnoverPayload toTurnoverPayload = getWideTurnoverPayload(calWideSettleVO, toAccountVO, true);
        turnoverPayloads.add(toTurnoverPayload);
        // 修改账户金额信息
        updateAccount(accountVOs);
        //保存流水数据
        calAccountTurnoverService.batchInsert(turnoverPayloads);
    }

    /**
     * 生成泛用结算单流水payload
     *
     * @param calWideSettleVO 泛用结算单
     * @param calAccountVO    结算账户
     * @param isAdd           收入：true,支出：false
     * @return
     */
    CalAccountTurnoverPayload getWideTurnoverPayload(CalWideSettleVO calWideSettleVO, CalAccountVO calAccountVO, Boolean isAdd) {
        LocalDate settleDate = calWideSettleVO.getSettleDate();
        //结算当量(.negate()判断加减)
        BigDecimal applySettleEqva = isAdd ? calWideSettleVO.getApplySettleEqva() : calWideSettleVO.getApplySettleEqva().negate();
        //结算金额
        BigDecimal applySettleAmt = isAdd ? calWideSettleVO.getApplySettleAmt() : calWideSettleVO.getApplySettleAmt().negate();
        //默认结算单价
        BigDecimal price = calWideSettleVO.getSettlePrice();

        BigDecimal totalQty = calAccountVO.getTotalQty() == null ? BigDecimal.ZERO : calAccountVO.getTotalQty();
        BigDecimal totalAmt = calAccountVO.getTotalAmt() == null ? BigDecimal.ZERO : calAccountVO.getTotalAmt();
        BigDecimal totalQtyNew = totalQty.add(applySettleEqva);
        BigDecimal totalAmtNew = totalAmt.add(applySettleAmt);

        CalAccountTurnoverPayload turnoverPayload = new CalAccountTurnoverPayload();
        turnoverPayload.setAccId(calAccountVO.getId());
        turnoverPayload.setAccName(calAccountVO.getLedgerName());
        turnoverPayload.setFinYear(settleDate.getYear());
        turnoverPayload.setFinPeriod(settleDate.getMonthValue());
        turnoverPayload.setIoDate(settleDate);
        turnoverPayload.setIoTime(LocalDateTime.now());
        turnoverPayload.setSourceType(calWideSettleVO.getSettleType());
        turnoverPayload.setSourceId(calWideSettleVO.getId());
        turnoverPayload.setSourceName(calWideSettleVO.getSettleNo());
        turnoverPayload.setTurnoverQty(applySettleEqva);
        turnoverPayload.setTurnoverAmt(applySettleAmt);
        turnoverPayload.setEqvaPrice(price);
        turnoverPayload.setBeforeQty(totalQty);
        turnoverPayload.setBeforeAmt(totalAmt);
        turnoverPayload.setAfterQty(totalQtyNew);
        turnoverPayload.setAfterAmt(totalAmtNew);

        //账户重新赋值
        calAccountVO.setTotalQty(totalQtyNew);
        calAccountVO.setTotalAmt(totalAmtNew);
        calAccountVO.setAvalQty(totalQtyNew.subtract(calAccountVO.getFrozenQty() == null ? BigDecimal.ZERO : calAccountVO.getFrozenQty()));
        calAccountVO.setAvalAmt(totalAmtNew.subtract(calAccountVO.getFrozenAmt() == null ? BigDecimal.ZERO : calAccountVO.getFrozenAmt()));
        return turnoverPayload;
    }

    /**
     * 泛用金额流水
     *
     * @param calNormSettleVO
     * @param normType        1:过账，0：取消过账
     */
    @Override
    public void amountSettleTurnover(CalNormSettleVO calNormSettleVO, Integer normType) {
        /**
         * 流水记录
         * 1.支出核单元：扣减流水
         * 2.收入核单元：增加流水
         */
        //新增流水记录
        List<CalAccountTurnoverPayload> turnoverPayloads = new ArrayList<>();

        //相关账户数据
        List<Long> accIds = Arrays.asList(calNormSettleVO.getOutAccountId(), calNormSettleVO.getInAccountId());
        List<CalAccountVO> calAccountVOS = calAccountService.queryByKeys(accIds);
        if (calAccountVOS.size() < accIds.size()) {
            throw TwException.error("", "账户数据不存在");
        }
        calAccountVOS.forEach(vo -> {
            if (vo.getLedgerStatus().equals("0")) {
                throw TwException.error("", vo.getLedgerName() + "账户已禁用");
            }
        });
        CalAccountVO outAccountVO = calAccountVOS.stream().filter(vo -> vo.getId().equals(calNormSettleVO.getOutAccountId())).findFirst().get();
        //1.支出核单元：扣减流水
        CalAccountTurnoverPayload outTurnoverPayload = getAmountTurnoverPayload(calNormSettleVO, outAccountVO, normType == 1 ? false : true);
        turnoverPayloads.add(outTurnoverPayload);
        //收入核单元
        CalAccountVO inAccountVO = calAccountVOS.stream().filter(vo -> vo.getId().equals(calNormSettleVO.getInAccountId())).findFirst().get();
        //2.收入核单元：增加流水
        CalAccountTurnoverPayload inTurnoverPayload = getAmountTurnoverPayload(calNormSettleVO, inAccountVO, normType == 1 ? true : false);
        turnoverPayloads.add(inTurnoverPayload);
        // 修改账户金额信息
        updateAccount(calAccountVOS);
        //保存流水数据
        calAccountTurnoverService.batchInsert(turnoverPayloads);
    }

    @Override
    public void amountBatchSettleTurnover(List<CalNormSettleVO> calNormSettleVOS, List<CalAccountVO> calAccountVOS) {
        //新增流水记录
        List<CalAccountTurnoverPayload> turnoverPayloads = new ArrayList<>();
        //修改账户
        List<CalAccountVO> updateAccountVOS = new ArrayList<>();
        calNormSettleVOS.forEach(calNormSettleVO -> {
            CalAccountVO outAccountVO = calAccountVOS.stream().filter(vo -> vo.getId().equals(calNormSettleVO.getOutAccountId())).findFirst().get();
            updateAccountVOS.add(outAccountVO);
            //1.支出核单元：扣减流水
            CalAccountTurnoverPayload outTurnoverPayload = getAmountTurnoverPayload(calNormSettleVO, outAccountVO, false);
            turnoverPayloads.add(outTurnoverPayload);
            //收入核单元
            CalAccountVO inAccountVO = calAccountVOS.stream().filter(vo -> vo.getId().equals(calNormSettleVO.getInAccountId())).findFirst().get();
            updateAccountVOS.add(inAccountVO);
            //2.收入核单元：增加流水
            CalAccountTurnoverPayload inTurnoverPayload = getAmountTurnoverPayload(calNormSettleVO, inAccountVO, true);
            turnoverPayloads.add(inTurnoverPayload);
        });
        //保存流水数据
        calAccountTurnoverService.batchInsert(turnoverPayloads);
        // 修改账户金额信息
        updateAccount(updateAccountVOS);
    }

    /**
     * 生成泛用金额结算单流水payload
     *
     * @param calNormSettleVO 泛用金额结算单
     * @param calAccountVO    结算账户
     * @param isAdd           收入：true,支出：false
     * @return
     */
    CalAccountTurnoverPayload getAmountTurnoverPayload(CalNormSettleVO calNormSettleVO, CalAccountVO calAccountVO, Boolean isAdd) {
        LocalDate settleDate = calNormSettleVO.getSettleDate();
        //(.negate()判断加减)
        //结算金额
        BigDecimal approveSettleAmt = isAdd ? calNormSettleVO.getApproveSettleAmt() : calNormSettleVO.getApproveSettleAmt().negate();

        BigDecimal totalAmt = calAccountVO.getTotalAmt() == null ? BigDecimal.ZERO : calAccountVO.getTotalAmt();
        BigDecimal totalQty = calAccountVO.getTotalQty() == null ? BigDecimal.ZERO : calAccountVO.getTotalQty();
        BigDecimal totalAmtNew = totalAmt.add(approveSettleAmt);

        CalAccountTurnoverPayload turnoverPayload = new CalAccountTurnoverPayload();
        turnoverPayload.setAccId(calAccountVO.getId());
        turnoverPayload.setAccName(calAccountVO.getLedgerName());
        turnoverPayload.setFinYear(settleDate.getYear());
        turnoverPayload.setFinPeriod(settleDate.getMonthValue());
        turnoverPayload.setIoDate(settleDate);
        turnoverPayload.setIoTime(LocalDateTime.now());
        turnoverPayload.setSourceType(CalAccTurTypeEnum.WIDE_AMT.getCode());
        turnoverPayload.setSourceId(calNormSettleVO.getId());
        turnoverPayload.setSourceName(calNormSettleVO.getSettleNo());
        turnoverPayload.setTurnoverQty(BigDecimal.ZERO);
        turnoverPayload.setTurnoverAmt(approveSettleAmt);
        turnoverPayload.setEqvaPrice(BigDecimal.ZERO);
        turnoverPayload.setBeforeQty(totalQty);
        turnoverPayload.setBeforeAmt(totalAmt);
        turnoverPayload.setAfterQty(totalQty);
        turnoverPayload.setAfterAmt(totalAmtNew);
        turnoverPayload.setRemark(calNormSettleVO.getRemark());
        //账户重新赋值
        calAccountVO.setTotalAmt(totalAmtNew);
        calAccountVO.setAvalAmt(totalAmtNew.subtract(calAccountVO.getFrozenAmt() == null ? BigDecimal.ZERO : calAccountVO.getFrozenAmt()));
        return turnoverPayload;


    }

    /**
     * 修改账户金额信息
     *
     * @param accountVOs
     */
    void updateAccount(List<CalAccountVO> accountVOs) {
        accountVOs.forEach(vo -> {
            //校验账户余额
            checkAccount(vo);

            CalAccountPayload payload = new CalAccountPayload();
            payload.setId(vo.getId());
            payload.setTotalQty(vo.getTotalQty());
            payload.setTotalAmt(vo.getTotalAmt());
            payload.setAvalQty(vo.getAvalQty());
            payload.setAvalAmt(vo.getAvalAmt());
            payload.setFrozenQty(vo.getFrozenQty());
            payload.setFrozenAmt(vo.getFrozenAmt());

            calAccountService.updateByKeyDynamic(payload);
        });
    }

    /**
     * 获取账户
     *
     * @param auType
     * @param auId
     * @return
     */
    @Override
    public CalAccountVO queryAccount(String auType, Long auId) {
        return calAccountService.queryByAuTypeAndAuId(auType, auId);
    }

    @Override
    public void checkAccount(CalAccountVO vo) {
        /**
         * 不校验账户余额
         * 1.平台账户
         * 2.商机账户
         */
        Boolean isCheck = true;
        if (StringUtils.hasText(vo.getLedgerType()) && "1".equals(vo.getLedgerType())) {
            //1.平台账户
            isCheck = false;
        }
        if (vo.getAuType().equals(CalAccTypeEnum.OPP.getCode())) {
            //2.商机账户
            isCheck = false;
        }
        if (isCheck) {
            //暂时不校验余额
            //可用当量
//            BigDecimal avalQty = vo.getAvalQty() == null ? BigDecimal.ZERO : vo.getAvalQty();
//            if (avalQty.compareTo(BigDecimal.ZERO) < 0) {
//                throw TwException.error("", "【" + vo.getLedgerName() + "】账户当量不足，操作后:" + avalQty + "");
//            }
//            //可用金额
//            BigDecimal avalAmt = vo.getAvalAmt() == null ? BigDecimal.ZERO : vo.getAvalAmt();
//            if (avalAmt.compareTo(BigDecimal.ZERO) < 0) {
//                log.error("【" + vo.getLedgerName() + "】账户余额不足:{}");
//                throw TwException.error("", "【" + vo.getLedgerName() + "】账户余额不足，操作后:" + avalAmt + "");
//            }
        }
    }

    @Override
    public void checkDistributeAccount(PmsDistributeVO distributeVO) {
//        if (DistributeEnum.distType.PROJECT.getCode().equals(distributeVO.getReasonType())) {
//            //校验项目派发
//            PmsProjectVO pmsProjectVO = pmsProjectService.queryByKey(distributeVO.getReasonId());
//            if (pmsProjectVO == null) {
//                throw TwException.error("", "派发项目存在");
//            }
//            if (!pmsProjectVO.getProjStatus().equals(ProjectStatusEnum.CREATE.getCode())) {
//                throw TwException.error("", "仅支持新建状态派发");
//            }
//            CalAccountVO calAccountVO = queryAccount(CalAccTypeEnum.BU.getCode(), pmsProjectVO.getDeliBuId());
        //可用当量
//            BigDecimal avalQty = calAccountVO.getAvalQty() == null ? BigDecimal.ZERO : calAccountVO.getAvalQty();
//            //可用金额
//            BigDecimal avalAmt = calAccountVO.getAvalAmt() == null ? BigDecimal.ZERO : calAccountVO.getAvalAmt();
//            calAccountVO.setAvalQty(avalQty.subtract(pmsProjectVO.getTotalEqva()));
//            calAccountVO.setAvalAmt(avalAmt.subtract(pmsProjectVO.getTotalCost()));
//            //核验账户余额
//            checkAccount(calAccountVO);
//        }
//        if (DistributeEnum.distType.TASK.getCode().equals(distributeVO.getReasonType()) || DistributeEnum.distType.TASK_PACKAGE.getCode().equals(distributeVO.getReasonType())) {
//            //事由类型
//            String reasonType = "";
//            //事由id
//            Long reasonId = 0L;
//            if (DistributeEnum.distType.TASK.getCode().equals(distributeVO.getReasonType())) {
//                TaskInfoVO taskInfoVO = taskInfoService.queryByKey(distributeVO.getReasonId(), true);
//                if (taskInfoVO == null) {
//                    throw TwException.error("", "派发任务存在");
//                }
//                reasonType = taskInfoVO.getReasonType();
//                reasonId = taskInfoVO.getReasonId();
//            }
//            //任务包不涉及当量
////            else {
////                TaskPackageVO taskPackageVO = taskPackageService.queryByKey(distributeVO.getReasonId());
////                if (taskPackageVO == null) {
////                    throw TwException.error("", "派发任务包存在");
////                }
////                reasonType = taskPackageVO.getReasonType();
////                reasonId = taskPackageVO.getReasonId();
////            }
//            String auType = reasonType.equals(PmsReasonTypeEnum.PROJ_CONTRACT.getCode()) ? CalAccTypeEnum.PROJ.getCode() : reasonType.equals(PmsReasonTypeEnum.PROJ_OPPO.getCode()) ? CalAccTypeEnum.OPP.getCode() : CalAccTypeEnum.BU_PROJ.getCode();
//            CalAccountVO calAccountVO = queryAccount(auType, reasonId);
//            //可用当量
//            BigDecimal avalQty = calAccountVO.getAvalQty() == null ? BigDecimal.ZERO : calAccountVO.getAvalQty();
//            //可用金额
//            BigDecimal avalAmt = calAccountVO.getAvalAmt() == null ? BigDecimal.ZERO : calAccountVO.getAvalAmt();
//            //TODO 不知道任务派发当量
////            calAccountVO.setAvalQty(avalQty.subtract(pmsProjectVO.getTotalEqva()));
////            calAccountVO.setAvalAmt(avalAmt.subtract(pmsProjectVO.getTotalCost()));
//            //核验账户余额
//            checkAccount(calAccountVO);
//        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void resetAccount(String auType, String operType, String remark) {
        Boolean rolePermission = cacheUtil.hasSystemRolePermission(Arrays.asList(RoleEnum.SYS.getCode()));
        if (!rolePermission) {
            throw TwException.error("", "无操作重置账户数据的权限！");
        }
        CalAccountQuery query = new CalAccountQuery();
        query.setAuType(auType);
        query.setOperType(operType);
        List<CalAccountVO> calAccountVOS = calAccountService.queryListDynamic(query);

        List<CalAccountTurnoverPayload> turnoverPayloads = new ArrayList<>();
        LocalDate now = LocalDate.now();
        calAccountVOS.forEach(calAccountVO -> {
            //账户历史数据
            BigDecimal totalAmt = calAccountVO.getTotalAmt() == null ? BigDecimal.ZERO : calAccountVO.getTotalAmt();
            BigDecimal totalQty = calAccountVO.getTotalQty() == null ? BigDecimal.ZERO : calAccountVO.getTotalQty();
            //账户变更数据
            BigDecimal amt = totalAmt.negate();
            BigDecimal eqva = totalQty.negate();

            BigDecimal totalAmtNew = BigDecimal.ZERO;
            BigDecimal totalQtyNew = BigDecimal.ZERO;

            CalAccountTurnoverPayload turnoverPayload = new CalAccountTurnoverPayload();
            turnoverPayload.setAccId(calAccountVO.getId());
            turnoverPayload.setAccName(calAccountVO.getLedgerName());
            turnoverPayload.setFinYear(now.getYear());
            turnoverPayload.setFinPeriod(now.getMonthValue());
            turnoverPayload.setIoDate(now);
            turnoverPayload.setIoTime(LocalDateTime.now());
            turnoverPayload.setSourceType(CalAccTurTypeEnum.HANDWORK.getCode());
            turnoverPayload.setSourceId(0L);
            turnoverPayload.setSourceName("");
            turnoverPayload.setTurnoverQty(eqva);
            turnoverPayload.setTurnoverAmt(amt);
            turnoverPayload.setEqvaPrice(BigDecimal.ZERO);
            turnoverPayload.setBeforeQty(totalQty);
            turnoverPayload.setBeforeAmt(totalAmt);
            turnoverPayload.setAfterQty(totalQtyNew);
            turnoverPayload.setAfterAmt(totalAmtNew);
            turnoverPayload.setRemark(remark);

            turnoverPayloads.add(turnoverPayload);
            //账户重新赋值
            calAccountVO.setTotalAmt(totalAmtNew);
            calAccountVO.setTotalQty(totalQtyNew);


            calAccountVO.setFrozenQty(BigDecimal.ZERO);
            calAccountVO.setFrozenAmt(BigDecimal.ZERO);
            calAccountVO.setAvalQty(BigDecimal.ZERO);
            calAccountVO.setAvalAmt(BigDecimal.ZERO);


        });
        if (turnoverPayloads.size() > 0) {
            // 修改账户金额信息
            updateAccount(calAccountVOS);
            //保存流水数据
            calAccountTurnoverService.batchInsert(turnoverPayloads);

        }

    }

    /**
     * 批量获取账户数据
     *
     * @param auType
     * @param auIds
     * @return
     */
    List<CalAccountVO> queryAccounts(String auType, List<Long> auIds) {
        List<CalAccountVO> calAccountVOS = calAccountService.queryByAuTypeAndAuIds(auType, auIds);
        if (ObjectUtils.isEmpty(calAccountVOS)) {
            throw TwException.error("", "账户不存在");
        }
        if (calAccountVOS.size() != auIds.size()) {
            throw TwException.error("", "账户数据存在异常或已禁用");
        }

        return calAccountVOS;
    }

    /**
     * 批量获取账户数据
     *
     * @param payloads
     * @return
     */
    List<CalAccountVO> queryAccounts(List<CalAccountPayload> payloads) {
        Set<String> auTypes = new HashSet<>();
        Set<Long> auIds = new HashSet<>();
        payloads.forEach(payload -> {
            auTypes.add(payload.getAuType());
            auIds.add(payload.getAuId());
        });
        List<CalAccountVO> calAccountVOS = calAccountService.queryByAuTypesAndAuIds(new ArrayList<>(auTypes), new ArrayList<>(auIds));
        payloads.forEach(payload -> {
            String auType = payload.getAuType();
            Optional<CalAccountVO> first = calAccountVOS.stream().filter(vo -> vo.getAuType().equals(auType) && vo.getAuId().equals(payload.getAuId())).findFirst();
            if (!first.isPresent()) {
                String name = "";
                if (auType.equals(CalAccTypeEnum.PROJ.getCode())) {
                    name = CalAccTypeEnum.PROJ.getDesc();
                }
                if (auType.equals(CalAccTypeEnum.BU.getCode())) {
                    name = CalAccTypeEnum.BU.getDesc();
                }
                if (auType.equals(CalAccTypeEnum.RES.getCode())) {
                    name = CalAccTypeEnum.RES.getDesc();
                }
                if (auType.equals(CalAccTypeEnum.OPP.getCode())) {
                    name = CalAccTypeEnum.OPP.getDesc();
                }
                if (auType.equals(CalAccTypeEnum.BU_PROJ.getCode())) {
                    name = CalAccTypeEnum.BU_PROJ.getDesc();
                }
                log.error(name + "账户不存在,auId:" + payload.getAuId());
                throw TwException.error("", name + "账户不存在或已禁用,auId:" + payload.getAuId());
            }
        });


        return calAccountVOS;
    }
}
