package com.elitesland.tw.tw5.server.prd.crm.service;


import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmGiftInfoPayload;
import com.elitesland.tw.tw5.api.prd.crm.payload.CrmGiftListPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmGiftListQuery;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmGiftInfoService;
import com.elitesland.tw.tw5.api.prd.crm.service.CrmGiftListService;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmGiftListVO;
import com.elitesland.tw.tw5.api.prd.org.query.PrdOrgEmployeeQuery;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgEmployeeVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.RoleEnum;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.WorkFlowStatusEnum;
import com.elitesland.tw.tw5.server.prd.crm.convert.CrmGiftListConvert;
import com.elitesland.tw.tw5.server.prd.crm.dao.CrmGiftListDAO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmGiftListDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmGiftListRepo;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgEmployeeDAO;
import com.elitesland.tw.tw5.server.prd.org.dao.PrdOrgOrganizationDAO;
import com.elitesland.tw.tw5.server.prd.system.dao.PrdSystemRoleDAO;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;


/**
 * 礼品单
 *
 * @author xxb
 * @date 2023-08-31
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CrmGiftListServiceImpl extends BaseServiceImpl implements CrmGiftListService {

    private final CrmGiftListRepo crmGiftListRepo;
    private final CrmGiftListDAO crmGiftListDAO;

    private final PrdOrgEmployeeDAO employeeDAO;

    private final PrdOrgOrganizationDAO daoOrg;

    private final WorkflowUtil workflowUtil;

    private final PrdSystemRoleDAO systemRoleDAO;

    private final CacheUtil cacheUtil;

    private final CrmGiftInfoService crmGiftInfoService;


    @Override
    public PagingVO<CrmGiftListVO> queryPaging(CrmGiftListQuery query) {
        return crmGiftListDAO.queryPaging(query);
    }

    @Override
    public List<CrmGiftListVO> queryListDynamic(CrmGiftListQuery query) {
        return crmGiftListDAO.queryListDynamic(query);
    }

    @Override
    public CrmGiftListVO queryByKey(Long key) {
        CrmGiftListDO entity = crmGiftListRepo.findById(key).orElseGet(CrmGiftListDO::new);
        Assert.notNull(entity.getId(), "不存在");
        CrmGiftListVO vo = CrmGiftListConvert.INSTANCE.toVo(entity);
        transferDatas(vo);
        return vo;
    }

    @Override
    public List<PrdOrgEmployeeVO> queryUserByGiftRole() {
        List<Long> userIdsByRole = systemRoleDAO.queryUserIdByRoleCodes(Arrays.asList(RoleEnum.GIFT_REQUEST_SPECIAL_USER.getCode(), RoleEnum.GIFT_APPLY_QUERY_ALL.getCode(), RoleEnum.SALE_RES.getCode()));
        if (ObjectUtils.isEmpty(userIdsByRole)) {
            return new ArrayList<>();
        }
        PrdOrgEmployeeQuery query = new PrdOrgEmployeeQuery();
        query.setUserIds(userIdsByRole);
        return employeeDAO.queryList(query);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmGiftListVO insert(CrmGiftListPayload payload) {
        //验证数据
        checkData(payload);

        //权限判断
        int specialUsersFlag = checkRole(payload.getApplyUserId());
        payload.setSpecialUsersFlag(specialUsersFlag);

        // 保存 礼品单基本信息
        CrmGiftListDO entityDo = CrmGiftListConvert.INSTANCE.toDo(payload);
        CrmGiftListVO vo = CrmGiftListConvert.INSTANCE.toVo(crmGiftListRepo.save(entityDo));
        // 保存 礼品明细
        List<CrmGiftInfoPayload> crmGiftInfoPayloads = payload.getCrmGiftInfoPayloads();
        crmGiftInfoPayloads.forEach(v -> {
            v.setGiftListId(vo.getId());
            v.setSpecialUsersFlag(payload.getSpecialUsersFlag());
        });
        crmGiftInfoService.batchInsert(crmGiftInfoPayloads);

        //开启审批流程
        if (payload.getSubmitFlag()) {
            submitProc(vo, payload.getApplyUserBuId());
        }
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CrmGiftListVO create() {
        Long loginUserId = GlobalUtil.getLoginUserId();
        Long createUserBuId = cacheUtil.getDefaultOrgIdByUserId(loginUserId);
        CrmGiftListVO vo = new CrmGiftListVO();
        // 生成规则：固定值+年份后两位+月份+四位序列   比如2023年7月的礼品申请，则序列号为固定值+23070001；依次增加
        vo.setGiftNo(generateSeqNum("GIFT_ACTIVITY"));
        vo.setCreateUserId(loginUserId);
        vo.setCreator(cacheUtil.getUserName(loginUserId));
        vo.setCreateUserBuId(createUserBuId);
        vo.setCreateTime(LocalDateTime.now());
        vo.setApplyStatus(WorkFlowStatusEnum.CREATE_WORK.getCode());
        transferDatas(vo);
        return vo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(CrmGiftListPayload payload) {
        if (ObjectUtils.isEmpty(payload.getId())) {
            throw TwException.error("", "礼品单ID不可为空");
        }
        //验证数据
        checkData(payload);

        //权限判断
        int specialUsersFlag = checkRole(payload.getApplyUserId());
        payload.setSpecialUsersFlag(specialUsersFlag);

        // 保存 礼品单基本信息
        CrmGiftListDO entity = crmGiftListRepo.findById(payload.getId()).orElseGet(CrmGiftListDO::new);
        Assert.notNull(entity.getId(), "不存在");

        Long loginUserId = GlobalUtil.getLoginUserId();
        if (!loginUserId.equals(entity.getCreateUserId()) && !loginUserId.equals(entity.getApplyUserId())) {
            throw TwException.error("", "非创建人和提交人 不能更改礼品单");
        }

        long result = crmGiftListDAO.updateByKeyDynamic(payload);

        // 保存 礼品明细
        List<CrmGiftInfoPayload> crmGiftInfoPayloads = payload.getCrmGiftInfoPayloads();
        crmGiftInfoPayloads.forEach(v -> {
            v.setGiftListId(payload.getId());
            v.setSpecialUsersFlag(payload.getSpecialUsersFlag());
        });
        crmGiftInfoService.batchUpdate(crmGiftInfoPayloads);

        // 开启审批流程
        if (payload.getSubmitFlag()) {
            if (ObjectUtils.isEmpty(entity.getProcInstId())) {
                // 开启审批流程
                submitProc(CrmGiftListConvert.INSTANCE.toVo(entity), payload.getApplyUserBuId());
            } else {
                payload.setApplyStatus(WorkFlowStatusEnum.APPROVING_WORK.getCode());
                //开启事务执行修改，主要是修改审批状态
                crmGiftListDAO.updateByKeyDynamic(payload);
            }
        }
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            crmGiftListDAO.deleteSoft(keys);
        }
    }


    /**
     * 发起新增审批流程
     *
     * @param vo
     */
    private void submitProc(CrmGiftListVO vo, Long applyUserBuId) {
        HashMap<String, Object> variables = new HashMap<>();
        //直属上级
        PrdOrgEmployeeRefVO prdOrgEmployeeRefVO = employeeDAO.queryUserOrgData(vo.getApplyUserId());
        if (prdOrgEmployeeRefVO == null) {
            throw TwException.error("", "员工信息查不到");
        }
        Long parentId = prdOrgEmployeeRefVO.getParentId();
        if (ObjectUtils.isEmpty(parentId)) {
            throw TwException.error("", "直属上级 不能为空");
        }
        variables.put("Activity_0i6loqz", CollUtil.newArrayList(parentId.toString()));
        //BU负责人
        Long buManageUserId = daoOrg.queryManageIdById(applyUserBuId);
        if (ObjectUtils.isEmpty(buManageUserId)) {
            throw TwException.error("", "BU负责人 不能为空");
        }
        variables.put("Activity_1dg4bha", CollUtil.newArrayList(buManageUserId.toString()));
        //发起流程审批
        ProcessInfo processInfo = workflowUtil.startProcess(StartProcessPayload.of(
                ProcDefKey.GIFT_SUBMIT.name(),
                "礼品申请流程-" + vo.getGiftNo(),
                vo.getId() + "",
                variables)
        );

        //流程启动成功后，回写业务表数据
        CrmGiftListPayload payload = new CrmGiftListPayload();
        payload.setProcInstId(processInfo.getProcInstId());
        payload.setId(vo.getId());
        payload.setProcInstStatus(processInfo.getProcInstStatus());
        payload.setApplyDate(LocalDate.now());
        payload.setApplyStatus(WorkFlowStatusEnum.APPROVING_WORK.getCode());
        //开启事务执行修改，主要是修改审批状态
        crmGiftListDAO.updateByKeyDynamic(payload);
    }

    /**
     * 全局翻译
     *
     * @param vo
     */
    void transferDatas(CrmGiftListVO vo) {
        vo.setGiftTypeName1(cacheUtil.transferSystemSelection("GIFT_TYPE", vo.getGiftType1()));
        vo.setGiftTypeName2(cacheUtil.transferSystemSelection("GIFT_TYPE:Festival", vo.getGiftType2()));
        vo.setApplyStatusName(cacheUtil.transferSystemSelection("gift:status", vo.getApplyStatus()));
        vo.setCreator(cacheUtil.getUserName(vo.getCreateUserId()));
    }


    /**
     * 验证数据为空
     *
     * @param payload
     */
    private void checkData(CrmGiftListPayload payload) {
        if (ObjectUtils.isEmpty(payload.getGiftType1())) {
            throw TwException.error("", "礼品类型不可为空");
        }
//        if (ObjectUtils.isEmpty(payload.getApplyDate())) {
//            throw TwException.error("", "申请日期不可为空");
//        }
        if (ObjectUtils.isEmpty(payload.getApplyUserId())) {
            throw TwException.error("", "申请人不可为空");
        }
        payload.setApplyUserBuId(cacheUtil.getDefaultOrgIdByUserId(payload.getApplyUserId()));
        List<CrmGiftInfoPayload> crmGiftInfoPayloads = payload.getCrmGiftInfoPayloads();
        if (ObjectUtils.isEmpty(crmGiftInfoPayloads)) {
            throw TwException.error("", "请先维护礼品信息");
        }
    }

    /**
     * 判断是否有权限执行
     *
     * @param loginUserId
     */
    private int checkRole(Long loginUserId) {
        List<String> roleCodes = systemRoleDAO.queryUserRoleCodes(loginUserId);
        if (!roleCodes.contains(RoleEnum.GIFT_REQUEST_SPECIAL_USER.getCode())
                && !roleCodes.contains(RoleEnum.GIFT_APPLY_QUERY_ALL.getCode())
                && !roleCodes.contains(RoleEnum.SALE_RES.getCode())) {
            throw TwException.error("", "你暂无权限");
        }
        // 判断当前用户 是否有 特殊角色 :盖莉珊，傅晓峰，梁锋，刘志军，肖屹松，周裕光
        return roleCodes.contains(RoleEnum.GIFT_REQUEST_SPECIAL_USER.getCode()) ? 1 : 0;
    }
}
