package com.elitesland.tw.tw5.server.prd.inv.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.inv.payload.InvInvoiceDocRefPayload;
import com.elitesland.tw.tw5.api.prd.inv.query.InvInvoiceDocRefQuery;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvInvoiceDocRefVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceDocRefDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvInvoiceDocRefDO;
import com.elitesland.tw.tw5.server.prd.inv.repo.InvInvoiceDocRefRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 发票单据关联管理
 *
 * @author sunxw
 * @date 2023-12-06
 */
@Repository
@RequiredArgsConstructor
public class InvInvoiceDocRefDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final InvInvoiceDocRefRepo repo;
    private final QInvInvoiceDocRefDO qdo = QInvInvoiceDocRefDO.invInvoiceDocRefDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<InvInvoiceDocRefVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(InvInvoiceDocRefVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 发票ID
                qdo.invId,
                // 关联单据类型 EXPENSE-报销
                qdo.docType,
                // 关联单据ID
                qdo.docId,
                // 关联单据明细ID
                qdo.docDtlId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<InvInvoiceDocRefVO> getJpaQueryWhere(InvInvoiceDocRefQuery query) {
        JPAQuery<InvInvoiceDocRefVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(InvInvoiceDocRefQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(InvInvoiceDocRefQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 发票ID 精确 */
        if (!ObjectUtils.isEmpty(query.getInvId())) {
            list.add(qdo.invId.eq(query.getInvId()));
        }
        /** 关联单据类型 EXPENSE-报销 精确 */
        if (!ObjectUtils.isEmpty(query.getDocType())) {
            list.add(qdo.docType.eq(query.getDocType()));
        }
        /** 关联单据ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDocId())) {
            list.add(qdo.docId.eq(query.getDocId()));
        }
        /** 关联单据明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getDocDtlId())) {
            list.add(qdo.docDtlId.eq(query.getDocDtlId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public InvInvoiceDocRefVO queryByKey(Long id) {
        JPAQuery<InvInvoiceDocRefVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<InvInvoiceDocRefVO> queryListDynamic(InvInvoiceDocRefQuery query) {
        JPAQuery<InvInvoiceDocRefVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<InvInvoiceDocRefVO> queryPaging(InvInvoiceDocRefQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<InvInvoiceDocRefVO> jpaQuery = getJpaQueryWhere(query);
        List<InvInvoiceDocRefVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<InvInvoiceDocRefVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public InvInvoiceDocRefDO save(InvInvoiceDocRefDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<InvInvoiceDocRefDO> saveAll(List<InvInvoiceDocRefDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(InvInvoiceDocRefPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 发票ID
        if (payload.getInvId() != null) {
            update.set(qdo.invId, payload.getInvId());
        }
        // 关联单据类型 EXPENSE-报销
        if (payload.getDocType() != null) {
            update.set(qdo.docType, payload.getDocType());
        }
        // 关联单据ID
        if (payload.getDocId() != null) {
            update.set(qdo.docId, payload.getDocId());
        }
        // 关联单据明细ID
        if (payload.getDocDtlId() != null) {
            update.set(qdo.docDtlId, payload.getDocDtlId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 发票ID
            if (nullFields.contains("invId")) {
                update.setNull(qdo.invId);
            }
            // 关联单据类型 EXPENSE-报销
            if (nullFields.contains("docType")) {
                update.setNull(qdo.docType);
            }
            // 关联单据ID
            if (nullFields.contains("docId")) {
                update.setNull(qdo.docId);
            }
            // 关联单据明细ID
            if (nullFields.contains("docDtlId")) {
                update.setNull(qdo.docDtlId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据单据id和类型删除
     *
     * @param docId
     * @param docType
     * @return
     */
    public long delByDocIdAndType(Long docId, String docType) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.docId.eq(docId).and(qdo.docType.eq(docType)));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据单据明细ID和类型获取发票ID
     *
     * @param docDtlId
     * @param docType
     * @return
     */
    public List<Long> queryByDtlIdAndType(Long docDtlId, String docType) {
        JPAQuery<Long> query = jpaQueryFactory.select(
                qdo.id
        ).from(qdo)
                .where(qdo.docDtlId.eq(docDtlId).and(qdo.docType.eq(docType)));
        return query.fetch();
    }
}

