package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitesland.tw.tw5.api.prd.pms.payload.PmsResourcePlanDaysPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsResourcePlanDaysQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsResourcePlanDaysVO;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsResourcePlanDaysDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsResourcePlanDaysDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsResourcePlanDaysRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

/**
 * 资源计划人天明细
 *
 * @author xxb
 * @date 2023-08-11
 */
@Repository
@RequiredArgsConstructor
public class PmsResourcePlanDaysDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsResourcePlanDaysRepo repo;
    private final QPmsResourcePlanDaysDO qdo = QPmsResourcePlanDaysDO.pmsResourcePlanDaysDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsResourcePlanDaysVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsResourcePlanDaysVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 关联资源计划ID
                qdo.planId,
                // 规划角色明细ID
                qdo.roleDetailId,
                // 规划工作天数
                qdo.days,
                // 开始时间
                qdo.startDate
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsResourcePlanDaysVO> getJpaQueryWhere(PmsResourcePlanDaysQuery query) {
        JPAQuery<PmsResourcePlanDaysVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsResourcePlanDaysQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsResourcePlanDaysQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 关联资源计划ID 精确 */
        if (!ObjectUtils.isEmpty(query.getPlanId())) {
            list.add(qdo.planId.eq(query.getPlanId()));
        }
        /** 规划角色明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRoleDetailId())) {
            list.add(qdo.roleDetailId.eq(query.getRoleDetailId()));
        }
        /** 规划角色明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRoleDetailIds())) {
            list.add(qdo.roleDetailId.in(query.getRoleDetailIds()));
        }
        /** 年周 精确 */
        if (!ObjectUtils.isEmpty(query.getYearWeek())) {
            list.add(qdo.yearWeek.eq(query.getYearWeek()));
        }
        /** 规划工作天数 精确 */
        if (!ObjectUtils.isEmpty(query.getDays())) {
            list.add(qdo.days.eq(query.getDays()));
        }
        /** 超过规划工作天数 精确 */
        if (!ObjectUtils.isEmpty(query.getLargeDays())) {
            list.add(qdo.days.gt(query.getLargeDays()));
        }
        /** 超过规划工作天数 精确 */
        if (!ObjectUtils.isEmpty(query.getNotNullPlanIdFlag())) {
            list.add(qdo.planId.isNotNull());
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsResourcePlanDaysVO queryByKey(Long id) {
        JPAQuery<PmsResourcePlanDaysVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsResourcePlanDaysVO> queryListDynamic(PmsResourcePlanDaysQuery query) {
        JPAQuery<PmsResourcePlanDaysVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsResourcePlanDaysVO> queryPaging(PmsResourcePlanDaysQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PmsResourcePlanDaysVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsResourcePlanDaysVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsResourcePlanDaysVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsResourcePlanDaysDO save(PmsResourcePlanDaysDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsResourcePlanDaysDO> saveAll(List<PmsResourcePlanDaysDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsResourcePlanDaysPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 关联资源计划ID
        if (payload.getPlanId() != null) {
            update.set(qdo.planId, payload.getPlanId());
        }
        // 规划角色明细ID
        if (payload.getRoleDetailId() != null) {
            update.set(qdo.roleDetailId, payload.getRoleDetailId());
        }
        // 年周
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 规划工作天数
        if (payload.getDays() != null) {
            update.set(qdo.days, payload.getDays());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 关联资源计划ID
            if (nullFields.contains("planId")) {
                update.setNull(qdo.planId);
            }
            // 规划角色明细ID
            if (nullFields.contains("roleDetailId")) {
                update.setNull(qdo.roleDetailId);
            }
            // 年周
            if (nullFields.contains("yearWeek")) {
                update.setNull(qdo.yearWeek);
            }
            // 规划工作天数
            if (nullFields.contains("days")) {
                update.setNull(qdo.days);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param startDates 主集合
     * @return 删除的行数
     */
    public long deleteSoftByPlanIdAndStartDate(Long planId, Set<LocalDate> startDates) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.planId.eq(planId))
                .where(qdo.startDate.in(startDates));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 根据 规划角色明细ID 逻辑删除
     *
     * @param roleDetailId 规划角色明细ID
     * @return 删除的行数
     */
    public long deleteSoftByRoleDetailId(Long roleDetailId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.roleDetailId.eq(roleDetailId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

