package com.elitesland.tw.tw5.server.prd.product.dao;

import com.elitesland.tw.tw5.api.prd.product.payload.PrdProductCasePayload;
import com.elitesland.tw.tw5.api.prd.product.query.PrdProductCaseQuery;
import com.elitesland.tw.tw5.api.prd.product.vo.PrdProductCaseVO;
import com.elitesland.tw.tw5.server.prd.product.entity.PrdProductCaseDO;
import com.elitesland.tw.tw5.server.prd.product.entity.QPrdProductCaseDO;
import com.elitesland.tw.tw5.server.prd.product.repo.PrdProductCaseRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 产品成功案例
 *
 * @author likunpeng
 * @date 2023-05-22
 */
@Repository
@RequiredArgsConstructor
public class PrdProductCaseDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PrdProductCaseRepo repo;
    private final QPrdProductCaseDO qdo = QPrdProductCaseDO.prdProductCaseDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdProductCaseVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdProductCaseVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 记录唯一ID
                qdo.prodId,
                // 案例名称
                qdo.caseName,
                // 案例描述
                qdo.caseDesc,
                // 联系信息
                qdo.contactDesc,
                // 案例图片路径
                qdo.casePicPath
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdProductCaseVO> getJpaQueryWhere(PrdProductCaseQuery query) {
        JPAQuery<PrdProductCaseVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PrdProductCaseQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PrdProductCaseQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProdId())) {
            list.add(qdo.prodId.eq(query.getProdId()));
        }
        /** 案例名称 精确 */
        if (!ObjectUtils.isEmpty(query.getCaseName())) {
            list.add(qdo.caseName.eq(query.getCaseName()));
        }
        /** 案例描述 精确 */
        if (!ObjectUtils.isEmpty(query.getCaseDesc())) {
            list.add(qdo.caseDesc.eq(query.getCaseDesc()));
        }
        /** 联系信息 精确 */
        if (!ObjectUtils.isEmpty(query.getContactDesc())) {
            list.add(qdo.contactDesc.eq(query.getContactDesc()));
        }
        /** 案例图片路径 精确 */
        if (!ObjectUtils.isEmpty(query.getCasePicPath())) {
            list.add(qdo.casePicPath.eq(query.getCasePicPath()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdProductCaseVO queryByKey(Long id) {
        JPAQuery<PrdProductCaseVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdProductCaseVO> queryListDynamic(PrdProductCaseQuery query) {
        JPAQuery<PrdProductCaseVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdProductCaseVO> queryPaging(PrdProductCaseQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PrdProductCaseVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdProductCaseVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdProductCaseVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdProductCaseDO save(PrdProductCaseDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdProductCaseDO> saveAll(List<PrdProductCaseDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PrdProductCasePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 记录唯一ID
        if (payload.getProdId() != null) {
            update.set(qdo.prodId, payload.getProdId());
        }
        // 案例名称
        if (payload.getCaseName() != null) {
            update.set(qdo.caseName, payload.getCaseName());
        }
        // 案例描述
        if (payload.getCaseDesc() != null) {
            update.set(qdo.caseDesc, payload.getCaseDesc());
        }
        // 联系信息
        if (payload.getContactDesc() != null) {
            update.set(qdo.contactDesc, payload.getContactDesc());
        }
        // 案例图片路径
        if (payload.getCasePicPath() != null) {
            update.set(qdo.casePicPath, payload.getCasePicPath());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 记录唯一ID
            if (nullFields.contains("prodId")) {
                update.setNull(qdo.prodId);
            }
            // 案例名称
            if (nullFields.contains("caseName")) {
                update.setNull(qdo.caseName);
            }
            // 案例描述
            if (nullFields.contains("caseDesc")) {
                update.setNull(qdo.caseDesc);
            }
            // 联系信息
            if (nullFields.contains("contactDesc")) {
                update.setNull(qdo.contactDesc);
            }
            // 案例图片路径
            if (nullFields.contains("casePicPath")) {
                update.setNull(qdo.casePicPath);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

