package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchasePaymentPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchasePaymentQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchasePaymentVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.WriteOffPaymentApplyVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccFinancialSubjectDO;
import com.elitesland.tw.tw5.server.prd.prj.entity.QPrjProjectDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchasePaymentDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 付款申请单
 *
 * @author likunpeng
 * @date 2023-11-07
 */
@Repository
@RequiredArgsConstructor
public class PurchasePaymentDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final QPurchasePaymentDO qdo = QPurchasePaymentDO.purchasePaymentDO;
    private final QPrjProjectDO qPrjProjectDO = QPrjProjectDO.prjProjectDO;
    private final QAccFinancialSubjectDO qAccFinancialSubjectDO = QAccFinancialSubjectDO.accFinancialSubjectDO;
    private final QPurchasePaymentDO qPrePaymentDO = new QPurchasePaymentDO("qPrePaymentDO");

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchasePaymentVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchasePaymentVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        qdo.updater,
                        qdo.modifyTime,
                        // 付款申请单编号
                        qdo.paymentNo,
                        // 付款申请单类型
                        qdo.paymentApplicationType,
                        // 申请日期
                        qdo.applicationDate,
                        // 供应商
                        qdo.supplierLegalBookId,
                        // 验收方式
                        qdo.acceptanceType,
                        // 付款金额
                        qdo.paymentAmt,
                        // 本次付款金额
                        qdo.currPaymentAmt,
                        // 币种
                        qdo.currCode,
                        // 付款申请人
                        qdo.purchaseInchargeResId,
                        // 付款申请单名称
                        qdo.purchaseName,
                        // 发票核销状态
                        qdo.invoiceState,
                        // 需求编号
                        qdo.demandNo,
                        // 关联单据类型
                        qdo.docType,
                        // 关联单据号
                        qdo.docNo,
                        // 关联销售合同
                        qdo.relatedSalesContract,
                        // 付款依据
                        qdo.basisFileCodes,
                        // 紧急付款凭证
                        qdo.urgentPaymentCodes,
                        // 关联项目号
                        qdo.relatedProjectNo,
                        // 归属付款申请
                        qdo.attributionPayApply,
                        // 发票编号
                        qdo.invoiceNo,
                        // 发票金额
                        qdo.invoiceAmt,
                        // 税率
                        qdo.rate,
                        // 税额
                        qdo.taxAmount,
                        // 付款方式
                        qdo.payMethod,
                        // 付款日期
                        qdo.payDate,
                        // 付款账期(天)
                        qdo.relatedDays,
                        // 付款账期到期时间
                        qdo.expRelatedDate,
                        // 收款人/单位BookId
                        qdo.receivingUnitBookId,
                        // 收款账号
                        qdo.receivingId,
                        // 收款银行
                        qdo.receivingBank,
                        // 预计核销日期
                        qdo.expWriteOffDate,
                        // 付款公司
                        qdo.payCompanyBookId,
                        // 付款银行
                        qdo.paymentBank,
                        // 付款账号
                        qdo.paymentId,
                        // 商机编号
                        qdo.opportunity,
                        // 商机名称
                        qPrjProjectDO.projectName.as("opportunityName"),
                        // 记账科目
                        qdo.accountingSubject,
                        // 财务记账付款公司
                        qdo.finalPaymentCompanyBookId,
                        // 财务记账付款银行
                        qdo.finalPaymentBank,
                        // 财务记账付款账号
                        qdo.finalPaymentId,
                        // 财务记账记账科目
                        qdo.finalAccountingSubject,
                        // 记账科目Desc
                        qAccFinancialSubjectDO.accName.as("finalAccountingSubjectDesc"),
                        // 财务记账付款方式
                        qdo.finalPayMethod,
                        // 财务记账付款日期
                        qdo.finalPayDate,
                        // 状态
                        qdo.state,
                        // 流程场景
                        qdo.scene,
                        // 是否无发票核销：0：否；1：是
                        qdo.noInvoiceVerification,
                        // 是否无单据核销：0：否；1：是
                        qdo.noDocVerification,
                        // 预付款申请单编号
                        qdo.prePaymentNo,
                        // 财务记账付款核销日期
                        qdo.finalWriteOffDate,
                        // 任务包
                        qdo.relatedTask,
                        // 实际付款日期
                        qdo.actualPayDate,
                        // 判断该付款申请单是否需要上传凭证: 0：否；1：是
                        qdo.isNeedUpload,
                        // 支付标识：1.已支付 ；2.待支付
                        qdo.payFlag,
                        // 申请单事由描述
                        qdo.reasonDesc,
                        // 备注
                        qdo.accountingNote,
                        // 财务记账备注
                        qdo.finalAccountingNote,
                        // 流程实例ID
                        qdo.procInstId,
                        // 流程审批状态
                        qdo.procInstStatus,
                        // 提交时间
                        qdo.submitTime,
                        // 审批时间
                        qdo.approvedTime,
                        // 是否为紧急付款
                        qdo.urgentPaymentFlag,
                        // 是否为驳回后再提交
                        qdo.isResubmit,
                        // 现金流量码
                        qdo.flowCode,
                        // 入账模块类型
                        qdo.accountModuleType,
                        // 生成资产卡片
                        qdo.assetFlag,
                        qdo.jdePayNo,
                        qdo.jdePaymentItem,
                        qdo.jdeDocumentType,
                        qdo.jdeDocumentNo,
                        qdo.jdeCompany,
                        qdo.jdeVoucher,
                        qdo.subAccountFlag,
                        qdo.expenseBuId,
                        // 预付款申请单id
                        qPrePaymentDO.id.as("prePaymentId")
                )).from(qdo)
                .leftJoin(qPrjProjectDO).on(qPrjProjectDO.projectNo.eq(qdo.opportunity))
                .leftJoin(qAccFinancialSubjectDO).on(qdo.finalAccountingSubject.eq(qAccFinancialSubjectDO.id))
                .leftJoin(qPrePaymentDO).on(qdo.prePaymentNo.eq(qPrePaymentDO.paymentNo));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchasePaymentVO> getJpaQueryWhere(PurchasePaymentQuery query) {
        JPAQuery<PurchasePaymentVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchasePaymentQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .leftJoin(qPrjProjectDO).on(qPrjProjectDO.projectNo.eq(qdo.opportunity))
                .leftJoin(qAccFinancialSubjectDO).on(qdo.accountingSubject.eq(qAccFinancialSubjectDO.id));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        Long result = jpaQuery.fetchOne();
        return result != null ? result : 0;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchasePaymentQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 付款申请单名称/编号 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentNameOrNo())) {
            list.add(qdo.paymentNo.like("%" + query.getPaymentNameOrNo() + "%")
                    .or(qdo.purchaseName.like("%" + query.getPaymentNameOrNo() + "%")));
        }
        /** 付款申请单类型 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplicationType())) {
            list.add(qdo.paymentApplicationType.eq(query.getPaymentApplicationType()));
        }
        /** 付款公司 精确 */
        if (!ObjectUtils.isEmpty(query.getPayCompanyBookId())) {
            list.add(qdo.payCompanyBookId.eq(query.getPayCompanyBookId()));
        }
        /** 供应商 精确 */
        if (!ObjectUtils.isEmpty(query.getSupplierLegalBookId())) {
            list.add(qdo.supplierLegalBookId.eq(query.getSupplierLegalBookId()));
        }
        /** 验收方式 精确 */
        if (!ObjectUtils.isEmpty(query.getAcceptanceType())) {
            list.add(qdo.acceptanceType.eq(query.getAcceptanceType()));
        }
        /** 关联销售合同 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedSalesContract())) {
            list.add(qdo.relatedSalesContract.eq(query.getRelatedSalesContract()));
        }
        /** 关联项目号 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedProjectNo())) {
            list.add(qdo.relatedProjectNo.eq(query.getRelatedProjectNo()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getState())) {
            list.add(qdo.state.eq(query.getState()));
        }
        /** 创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        /** 创建时间 区间 */
        if (!ObjectUtils.isEmpty(query.getCreateTimeStart()) && !ObjectUtils.isEmpty(query.getCreateTimeEnd())) {
            list.add(qdo.createTime.between(query.getCreateTimeStart(), query.getCreateTimeEnd()));
        }
        /** ids */
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 状态 List */
        if (!ObjectUtils.isEmpty(query.getNotInStateList())) {
            list.add(qdo.state.notIn(query.getNotInStateList()));
        }

        /** 采购合同编号 List */
        if (!ObjectUtils.isEmpty(query.getPurchaseContractNoList())) {
            list.add(qdo.docNo.in(query.getPurchaseContractNoList()));
        }
        /** 预付款核销的预付款申请单编号 List */
        if (!ObjectUtils.isEmpty(query.getPrePaymentNoList())) {
            list.add(qdo.prePaymentNo.in(query.getPrePaymentNoList()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchasePaymentVO queryByKey(Long id) {
        JPAQuery<PurchasePaymentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchasePaymentVO> queryListDynamic(PurchasePaymentQuery query) {
        JPAQuery<PurchasePaymentVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchasePaymentVO> queryPaging(PurchasePaymentQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PurchasePaymentVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchasePaymentVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchasePaymentVO>builder().records(result).total(total).build();
    }


    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchasePaymentPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 付款申请单编号
        if (payload.getPaymentNo() != null) {
            update.set(qdo.paymentNo, payload.getPaymentNo());
        }
        // 付款申请单类型
        if (payload.getPaymentApplicationType() != null) {
            update.set(qdo.paymentApplicationType, payload.getPaymentApplicationType());
        }
        // 申请日期
        if (payload.getApplicationDate() != null) {
            update.set(qdo.applicationDate, payload.getApplicationDate());
        }
        // 供应商
        if (payload.getSupplierLegalBookId() != null) {
            update.set(qdo.supplierLegalBookId, payload.getSupplierLegalBookId());
        }
        // 验收方式
        if (payload.getAcceptanceType() != null) {
            update.set(qdo.acceptanceType, payload.getAcceptanceType());
        }
        // 付款金额
        if (payload.getPaymentAmt() != null) {
            update.set(qdo.paymentAmt, payload.getPaymentAmt());
        }
        // 本次付款金额
        if (payload.getCurrPaymentAmt() != null) {
            update.set(qdo.currPaymentAmt, payload.getCurrPaymentAmt());
        }
        // 币种
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 付款申请人
        if (payload.getPurchaseInchargeResId() != null) {
            update.set(qdo.purchaseInchargeResId, payload.getPurchaseInchargeResId());
        }
        // 付款申请单名称
        if (payload.getPurchaseName() != null) {
            update.set(qdo.purchaseName, payload.getPurchaseName());
        }
        // 发票核销状态
        if (payload.getInvoiceState() != null) {
            update.set(qdo.invoiceState, payload.getInvoiceState());
        }
        // 需求编号
        if (payload.getDemandNo() != null) {
            update.set(qdo.demandNo, payload.getDemandNo());
        }
        // 关联单据类型
        if (payload.getDocType() != null) {
            update.set(qdo.docType, payload.getDocType());
        }
        // 关联单据号
        if (payload.getDocNo() != null) {
            update.set(qdo.docNo, payload.getDocNo());
        }
        // 关联销售合同
        if (payload.getRelatedSalesContract() != null) {
            update.set(qdo.relatedSalesContract, payload.getRelatedSalesContract());
        }
        // 关联项目号
        if (payload.getRelatedProjectNo() != null) {
            update.set(qdo.relatedProjectNo, payload.getRelatedProjectNo());
        }
        // 归属付款申请
        if (payload.getAttributionPayApply() != null) {
            update.set(qdo.attributionPayApply, payload.getAttributionPayApply());
        }
        // 发票编号
        if (payload.getInvoiceNo() != null) {
            update.set(qdo.invoiceNo, payload.getInvoiceNo());
        }
        // 发票金额
        if (payload.getInvoiceAmt() != null) {
            update.set(qdo.invoiceAmt, payload.getInvoiceAmt());
        }
        // 税率
        if (payload.getRate() != null) {
            update.set(qdo.rate, payload.getRate());
        }
        // 税额
        if (payload.getTaxAmount() != null) {
            update.set(qdo.taxAmount, payload.getTaxAmount());
        }
        // 付款方式
        if (payload.getPayMethod() != null) {
            update.set(qdo.payMethod, payload.getPayMethod());
        }
        // 付款日期
        if (payload.getPayDate() != null) {
            update.set(qdo.payDate, payload.getPayDate());
        }
        // 付款账期(天)
        if (payload.getRelatedDays() != null) {
            update.set(qdo.relatedDays, payload.getRelatedDays());
        }
        // 付款账期到期时间
        if (payload.getExpRelatedDate() != null) {
            update.set(qdo.expRelatedDate, payload.getExpRelatedDate());
        }
        // 收款人/单位BookId
        if (payload.getReceivingUnitBookId() != null) {
            update.set(qdo.receivingUnitBookId, payload.getReceivingUnitBookId());
        }
        // 收款账号
        if (payload.getReceivingId() != null) {
            update.set(qdo.receivingId, payload.getReceivingId());
        }
        // 收款银行
        if (payload.getReceivingBank() != null) {
            update.set(qdo.receivingBank, payload.getReceivingBank());
        }
        // 预计核销日期
        if (payload.getExpWriteOffDate() != null) {
            update.set(qdo.expWriteOffDate, payload.getExpWriteOffDate());
        }
        // 付款公司
        if (payload.getPayCompanyBookId() != null) {
            update.set(qdo.payCompanyBookId, payload.getPayCompanyBookId());
        }
        // 付款银行
        if (payload.getPaymentBank() != null) {
            update.set(qdo.paymentBank, payload.getPaymentBank());
        }
        // 付款账号
        if (payload.getPaymentId() != null) {
            update.set(qdo.paymentId, payload.getPaymentId());
        }
        // 商机
        if (payload.getOpportunity() != null) {
            update.set(qdo.opportunity, payload.getOpportunity());
        }
        // 记账科目
        if (payload.getAccountingSubject() != null) {
            update.set(qdo.accountingSubject, payload.getAccountingSubject());
        }
        // 财务记账付款公司
        if (payload.getFinalPaymentCompanyBookId() != null) {
            update.set(qdo.finalPaymentCompanyBookId, payload.getFinalPaymentCompanyBookId());
        }
        // 财务记账付款银行
        if (payload.getFinalPaymentBank() != null) {
            update.set(qdo.finalPaymentBank, payload.getFinalPaymentBank());
        }
        // 财务记账付款账号
        if (payload.getFinalPaymentId() != null) {
            update.set(qdo.finalPaymentId, payload.getFinalPaymentId());
        }
        // 财务记账记账科目
        if (payload.getFinalAccountingSubject() != null) {
            update.set(qdo.finalAccountingSubject, payload.getFinalAccountingSubject());
        }
        // 财务记账付款方式
        if (payload.getFinalPayMethod() != null) {
            update.set(qdo.finalPayMethod, payload.getFinalPayMethod());
        }

        // 生成资产卡片(是/否) 0：否；1：是
        if (payload.getAssetFlag() != null) {
            update.set(qdo.assetFlag, payload.getAssetFlag());
        }

        // 现金流量码
        if (payload.getFlowCode() != null) {
            update.set(qdo.flowCode, payload.getFlowCode());
        }

        // 入账模块类型
        if (payload.getAccountModuleType() != null) {
            update.set(qdo.accountModuleType, payload.getAccountModuleType());
        }
        // 财务记账付款日期
        if (payload.getFinalPayDate() != null) {
            update.set(qdo.finalPayDate, payload.getFinalPayDate());
        }
        // 状态
        if (payload.getState() != null) {
            update.set(qdo.state, payload.getState());
        }
        // 流程场景
        if (payload.getScene() != null) {
            update.set(qdo.scene, payload.getScene());
        }
        // 是否无发票核销：0：否；1：是
        if (payload.getNoInvoiceVerification() != null) {
            update.set(qdo.noInvoiceVerification, payload.getNoInvoiceVerification());
        }
        // 是否无单据核销：0：否；1：是
        if (payload.getNoDocVerification() != null) {
            update.set(qdo.noDocVerification, payload.getNoDocVerification());
        }
        // 预付款申请单编号
        if (payload.getPrePaymentNo() != null) {
            update.set(qdo.prePaymentNo, payload.getPrePaymentNo());
        }
        // 财务记账付款核销日期
        if (payload.getFinalWriteOffDate() != null) {
            update.set(qdo.finalWriteOffDate, payload.getFinalWriteOffDate());
        }
        // 任务包
        if (payload.getRelatedTask() != null) {
            update.set(qdo.relatedTask, payload.getRelatedTask());
        }
        // 实际付款日期
        if (payload.getActualPayDate() != null) {
            update.set(qdo.actualPayDate, payload.getActualPayDate());
        }
        // 判断该付款申请单是否需要上传凭证: 0：否；1：是
        if (payload.getIsNeedUpload() != null) {
            update.set(qdo.isNeedUpload, payload.getIsNeedUpload());
        }
        // 支付标识：1.已支付 ；2.待支付
        if (payload.getPayFlag() != null) {
            update.set(qdo.payFlag, payload.getPayFlag());
        }
        // 申请单事由描述
        if (payload.getReasonDesc() != null) {
            update.set(qdo.reasonDesc, payload.getReasonDesc());
        }
        // 备注
        if (payload.getAccountingNote() != null) {
            update.set(qdo.accountingNote, payload.getAccountingNote());
        }
        // 财务记账备注
        if (payload.getFinalAccountingNote() != null) {
            update.set(qdo.finalAccountingNote, payload.getFinalAccountingNote());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 提交时间
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 是否为紧急付款
        if (payload.getUrgentPaymentFlag() != null) {
            update.set(qdo.urgentPaymentFlag, payload.getUrgentPaymentFlag());
        }
        // 紧急付款凭证
        if (payload.getUrgentPaymentCodes() != null) {
            update.set(qdo.urgentPaymentCodes, payload.getUrgentPaymentCodes());
        }
        // 是是否为驳回后再提交
        if (payload.getIsResubmit() != null) {
            update.set(qdo.isResubmit, payload.getIsResubmit());
        }
        // 付款依据
        if (payload.getBasisFileCodes() != null) {
            update.set(qdo.basisFileCodes, payload.getBasisFileCodes());
        }
        // 付款依据
        if (payload.getBasisFileCodes() != null) {
            update.set(qdo.basisFileCodes, payload.getBasisFileCodes());
        }
        // 付款依据
        if (payload.getBasisFileCodes() != null) {
            update.set(qdo.basisFileCodes, payload.getBasisFileCodes());
        }
        // 付款依据
        if (payload.getBasisFileCodes() != null) {
            update.set(qdo.basisFileCodes, payload.getBasisFileCodes());
        }
        // 是否子帐
        if (payload.getSubAccountFlag() != null) {
            update.set(qdo.subAccountFlag, payload.getSubAccountFlag());
        }
        // 费用承担bu
        if (payload.getExpenseBuId() != null) {
            update.set(qdo.expenseBuId, payload.getExpenseBuId());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 付款依据
            if (nullFields.contains("basisFileCodes")) {
                update.setNull(qdo.basisFileCodes);
            }
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 付款申请单编号
            if (nullFields.contains("paymentNo")) {
                update.setNull(qdo.paymentNo);
            }
            // 付款申请单类型
            if (nullFields.contains("paymentApplicationType")) {
                update.setNull(qdo.paymentApplicationType);
            }
            // 申请日期
            if (nullFields.contains("applicationDate")) {
                update.setNull(qdo.applicationDate);
            }
            // 供应商
            if (nullFields.contains("supplierLegalBookId")) {
                update.setNull(qdo.supplierLegalBookId);
            }
            // 验收方式
            if (nullFields.contains("acceptanceType")) {
                update.setNull(qdo.acceptanceType);
            }
            // 付款金额
            if (nullFields.contains("paymentAmt")) {
                update.setNull(qdo.paymentAmt);
            }
            // 本次付款金额
            if (nullFields.contains("currPaymentAmt")) {
                update.setNull(qdo.currPaymentAmt);
            }
            // 币种
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 付款申请人
            if (nullFields.contains("purchaseInchargeResId")) {
                update.setNull(qdo.purchaseInchargeResId);
            }
            // 付款申请单名称
            if (nullFields.contains("purchaseName")) {
                update.setNull(qdo.purchaseName);
            }
            // 发票核销状态
            if (nullFields.contains("invoiceState")) {
                update.setNull(qdo.invoiceState);
            }
            // 需求编号
            if (nullFields.contains("demandNo")) {
                update.setNull(qdo.demandNo);
            }
            // 关联单据类型
            if (nullFields.contains("docType")) {
                update.setNull(qdo.docType);
            }
            // 关联单据号
            if (nullFields.contains("docNo")) {
                update.setNull(qdo.docNo);
            }
            // 关联销售合同
            if (nullFields.contains("relatedSalesContract")) {
                update.setNull(qdo.relatedSalesContract);
            }
            // 关联项目号
            if (nullFields.contains("relatedProjectNo")) {
                update.setNull(qdo.relatedProjectNo);
            }
            // 归属付款申请
            if (nullFields.contains("attributionPayApply")) {
                update.setNull(qdo.attributionPayApply);
            }
            // 发票编号
            if (nullFields.contains("invoiceNo")) {
                update.setNull(qdo.invoiceNo);
            }
            // 发票金额
            if (nullFields.contains("invoiceAmt")) {
                update.setNull(qdo.invoiceAmt);
            }
            // 税率
            if (nullFields.contains("rate")) {
                update.setNull(qdo.rate);
            }
            // 税额
            if (nullFields.contains("taxAmount")) {
                update.setNull(qdo.taxAmount);
            }
            // 付款方式
            if (nullFields.contains("payMethod")) {
                update.setNull(qdo.payMethod);
            }
            // 付款日期
            if (nullFields.contains("payDate")) {
                update.setNull(qdo.payDate);
            }
            // 付款账期(天)
            if (nullFields.contains("relatedDays")) {
                update.setNull(qdo.relatedDays);
            }
            // 付款账期到期时间
            if (nullFields.contains("expRelatedDate")) {
                update.setNull(qdo.expRelatedDate);
            }
            // 收款人/单位BookId
            if (nullFields.contains("receivingUnitBookId")) {
                update.setNull(qdo.receivingUnitBookId);
            }
            // 收款账号
            if (nullFields.contains("receivingId")) {
                update.setNull(qdo.receivingId);
            }
            // 收款银行
            if (nullFields.contains("receivingBank")) {
                update.setNull(qdo.receivingBank);
            }
            // 预计核销日期
            if (nullFields.contains("expWriteOffDate")) {
                update.setNull(qdo.expWriteOffDate);
            }
            // 付款公司
            if (nullFields.contains("payCompanyBookId")) {
                update.setNull(qdo.payCompanyBookId);
            }
            // 付款银行
            if (nullFields.contains("paymentBank")) {
                update.setNull(qdo.paymentBank);
            }
            // 付款账号
            if (nullFields.contains("paymentId")) {
                update.setNull(qdo.paymentId);
            }
            // 商机
            if (nullFields.contains("opportunity")) {
                update.setNull(qdo.opportunity);
            }
            // 记账科目
            if (nullFields.contains("accountingSubject")) {
                update.setNull(qdo.accountingSubject);
            }
            // 财务记账付款公司
            if (nullFields.contains("finalPaymentCompanyBookId")) {
                update.setNull(qdo.finalPaymentCompanyBookId);
            }
            // 财务记账付款银行
            if (nullFields.contains("finalPaymentBank")) {
                update.setNull(qdo.finalPaymentBank);
            }
            // 财务记账付款账号
            if (nullFields.contains("finalPaymentId")) {
                update.setNull(qdo.finalPaymentId);
            }
            // 财务记账记账科目
            if (nullFields.contains("finalAccountingSubject")) {
                update.setNull(qdo.finalAccountingSubject);
            }
            // 财务记账付款方式
            if (nullFields.contains("finalPayMethod")) {
                update.setNull(qdo.finalPayMethod);
            }
            // 财务记账付款日期
            if (nullFields.contains("finalPayDate")) {
                update.setNull(qdo.finalPayDate);
            }
            // 状态
            if (nullFields.contains("state")) {
                update.setNull(qdo.state);
            }
            // 流程场景
            if (nullFields.contains("scene")) {
                update.setNull(qdo.scene);
            }
            // 是否无发票核销：0：否；1：是
            if (nullFields.contains("noInvoiceVerification")) {
                update.setNull(qdo.noInvoiceVerification);
            }
            // 是否无单据核销：0：否；1：是
            if (nullFields.contains("noDocVerification")) {
                update.setNull(qdo.noDocVerification);
            }
            // 预付款申请单编号
            if (nullFields.contains("prePaymentNo")) {
                update.setNull(qdo.prePaymentNo);
            }
            // 财务记账付款核销日期
            if (nullFields.contains("finalWriteOffDate")) {
                update.setNull(qdo.finalWriteOffDate);
            }
            // 任务包
            if (nullFields.contains("relatedTask")) {
                update.setNull(qdo.relatedTask);
            }
            // 实际付款日期
            if (nullFields.contains("actualPayDate")) {
                update.setNull(qdo.actualPayDate);
            }
            // 判断该付款申请单是否需要上传凭证: 0：否；1：是
            if (nullFields.contains("isNeedUpload")) {
                update.setNull(qdo.isNeedUpload);
            }
            // 支付标识：1.已支付 ；2.待支付
            if (nullFields.contains("payFlag")) {
                update.setNull(qdo.payFlag);
            }
            // 申请单事由描述
            if (nullFields.contains("reasonDesc")) {
                update.setNull(qdo.reasonDesc);
            }
            // 备注
            if (nullFields.contains("accountingNote")) {
                update.setNull(qdo.accountingNote);
            }
            // 财务记账备注
            if (nullFields.contains("finalAccountingNote")) {
                update.setNull(qdo.finalAccountingNote);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 提交时间
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            // 是否为紧急付款
            if (nullFields.contains("urgentPaymentFlag")) {
                update.setNull(qdo.urgentPaymentFlag);
            }
            // 紧急付款凭证
            if (nullFields.contains("urgentPaymentCodes")) {
                update.setNull(qdo.urgentPaymentCodes);
            }
            // 是是否为驳回后再提交
            if (nullFields.contains("isResubmit")) {
                update.setNull(qdo.isResubmit);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param id 主键
     * @return 删除的行数
     */
    public long deleteSoft(Long id) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.eq(id));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 获取标识和创建人
     *
     * @param id 申请单Id
     * @return Pair<Long, String> key 创建人 value 标识
     */
    public Tuple getSceneAndCreatorAndState(Long id) {
        JPAQuery<Tuple> jpaQuery = jpaQueryFactory.select(
                        qdo.scene,
                        qdo.createUserId,
                        qdo.state,
                        // 付款申请单类型
                        qdo.paymentApplicationType,
                        // 关联单据类型
                        qdo.docType
                ).from(qdo)
                .where(qdo.id.eq(id));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据预付款核销申请单id查询预付款申请单Id、本次付款金额
     *
     * @param paymentApplyId 预付款核销申请单id
     * @return 预付款id
     */
    public Tuple findPrePaymentId(Long paymentApplyId) {
        JPAQuery<Tuple> jpaQuery = jpaQueryFactory.select(
                        qPrePaymentDO.id,
                        qPrePaymentDO.currPaymentAmt
                ).from(qdo)
                .leftJoin(qPrePaymentDO).on(qdo.prePaymentNo.eq(qPrePaymentDO.paymentNo))
                .where(qdo.id.eq(paymentApplyId));
        return jpaQuery.fetchFirst();
    }

    /**
     * 修改付款申请单状态
     *
     * @param paymentApplyNo 付款申请单编号
     * @param status         修改后的状态
     */
    public void updatePaymentStatus(String paymentApplyNo, PurchasePaymentEnum.PaymentStatus status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.state, status.getCode())
                .where(qdo.paymentNo.eq(paymentApplyNo));
        update.execute();
    }

    /**
     * 修改付款申请单状态
     *
     * @param paymentApplyId 付款申请单Id
     * @param status         修改后的状态
     */
    public void updatePaymentStatus(Long paymentApplyId, PurchasePaymentEnum.PaymentStatus status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.state, status.getCode())
                .where(qdo.id.eq(paymentApplyId));
        update.execute();
    }

    /**
     * 通过DocNo和状态查询付款申请单
     *
     * @param docNo
     * @param statusList 状态集合
     */
    public List<PurchasePaymentVO> queryByDocNo(String docNo, List<String> statusList) {
        return jpaQueryFactory.select(Projections.bean(PurchasePaymentVO.class,
                        qdo.id,
                        qdo.state,
                        qdo.currPaymentAmt)).
                from(qdo).
                where(qdo.deleteFlag.eq(0)).
                where(qdo.state.in(statusList)).
                where(qdo.docNo.eq(docNo)).
                fetch();
    }

    /**
     * 根据预付款申请单编号查询预付款核销单
     *
     * @param prePaymentApplyNoList 预付款申请单编号List
     * @return 预付款核销单
     */
    public List<WriteOffPaymentApplyVO> findWriteOffByPrePaymentNoList(List<String> prePaymentApplyNoList) {
        JPAQuery<WriteOffPaymentApplyVO> jpaQuery = jpaQueryFactory.select(Projections.bean(WriteOffPaymentApplyVO.class,
                qdo.id,
                qdo.paymentNo,
                qdo.prePaymentNo,
                qdo.currPaymentAmt,
                qdo.state
        )).from(qdo).where(qdo.prePaymentNo.in(prePaymentApplyNoList)
                .and(qdo.deleteFlag.eq(0)));
        return jpaQuery.fetch();
    }

    /**
     * 查询付款申请单本次付款金额
     *
     * @param paymentNo 付款申请单编号
     * @return 本次付款基恩
     */
    public BigDecimal findCurrPaymentAmtByNo(String paymentNo) {
        return jpaQueryFactory.select(
                        qdo.currPaymentAmt
                ).from(qdo)
                .where(qdo.deleteFlag.eq(0)
                        .and(qdo.paymentNo.eq(paymentNo))).fetchOne();
    }



    /**
     * 根据单号查询id
     * @param paymentNo
     * @return
     */
    public Long findCurrPaymentById(String paymentNo) {
        return jpaQueryFactory.select(
                        qdo.id
                ).from(qdo)
                .where(qdo.deleteFlag.eq(0)
                        .and(qdo.paymentNo.eq(paymentNo))).fetchOne();
    }
}

