package com.elitesland.tw.tw5.server.prd.schedule.dao;

import com.elitesland.tw.tw5.api.prd.schedule.payload.PrdSchedulePayload;
import com.elitesland.tw.tw5.api.prd.schedule.query.PrdScheduleQuery;
import com.elitesland.tw.tw5.api.prd.schedule.vo.PrdScheduleVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgEmployeeDO;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgPersonDO;
import com.elitesland.tw.tw5.server.prd.schedule.entity.PrdScheduleDO;
import com.elitesland.tw.tw5.server.prd.schedule.entity.QPrdCalendarDO;
import com.elitesland.tw.tw5.server.prd.schedule.entity.QPrdScheduleDO;
import com.elitesland.tw.tw5.server.prd.schedule.entity.QPrdScheduleRepeatDO;
import com.elitesland.tw.tw5.server.prd.schedule.repo.PrdScheduleRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author zoey
 * @Description:日程的dao层
 * @date 2022/4/19 - 23:26
 */
@Repository
@RequiredArgsConstructor
public class PrdScheduleDAO {
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdScheduleRepo repo;
    private final QPrdScheduleDO qdo = QPrdScheduleDO.prdScheduleDO;
    private final QPrdScheduleRepeatDO rqdo = QPrdScheduleRepeatDO.prdScheduleRepeatDO;
    private final QPrdCalendarDO rcdo= QPrdCalendarDO.prdCalendarDO;
    private final QPrdOrgEmployeeDO employeeDO = new QPrdOrgEmployeeDO("prdOrgEmployeeDO");
    private final QPrdOrgPersonDO personDO = QPrdOrgPersonDO.prdOrgPersonDO;

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdScheduleDO save(PrdScheduleDO ado) {
        return repo.save(ado);
    }


    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdScheduleDO> saveAll(List<PrdScheduleDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 根据主键查询
     * @param id 主键
     * @return 结果
     */
    public PrdScheduleVO queryByKey(Long id){
        JPAQuery<PrdScheduleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据企业微信日程id查询
     * @param id 主键
     * @return 结果
     */
    public PrdScheduleVO queryByQyWxScheduleId(String id){
        JPAQuery<PrdScheduleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.qyWxScheduleId.eq(id));
        return jpaQuery.fetchFirst();
    }




    /**
     * 拼装查询字段
     * @return  jpaQuery对象
     */
    private JPAQuery<PrdScheduleVO> getJpaQuerySelect() {
        JPAQuery<PrdScheduleVO> jpaQuery=jpaQueryFactory.select(Projections.bean(PrdScheduleVO.class,
                qdo.id,
                qdo.title,
                qdo.description,
                qdo.location,
                qdo.startTime,
                qdo.endTime,
                qdo.calId,
                qdo.allowActiveJoin,
                qdo.isRepeat,
                qdo.isRemind,
                qdo.allDay,
                qdo.remindBeforeEventSecs,
                qdo.createUserId,
                qdo.qyWxScheduleId,
                rqdo.repeatType,
                rqdo.repeatUntil,
                rqdo.isCustomRepeat,
                rqdo.repeatInterval,
                rqdo.repeatDayOfWeek,
                rqdo.repeatDayOfMonth,
                rqdo.exdate,
                employeeDO.employeeName,
                rcdo.title.as("calTitle"),
                rcdo.color
        )).from(qdo)
                .leftJoin(rqdo).on(qdo.id.longValue().eq(rqdo.secheduleId))
                .leftJoin(rcdo).on(rcdo.id.longValue().eq(qdo.calId))
                .leftJoin(employeeDO).on(qdo.createUserId.eq(employeeDO.userId));
//                .leftJoin(personDO).on(employeeDO.personId.eq(personDO.id));
        return jpaQuery;
    }


    /**
     * 拼装查询条件
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdScheduleVO> getJpaQueryWhere(PrdScheduleQuery query) {
        JPAQuery<PrdScheduleVO> jpaQuery = getJpaQuerySelect();

        if (!ObjectUtils.isEmpty(query.getTitle())) {
            jpaQuery.where(qdo.title.like(SqlUtil.toSqlLikeString(query.getTitle())));
        }
        if (!ObjectUtils.isEmpty(query.getDescription())) {
            jpaQuery.where(qdo.description.like(SqlUtil.toSqlLikeString(query.getDescription())));
        }
        if (!ObjectUtils.isEmpty(query.getCalId())) {
            jpaQuery.where(qdo.calId.eq(query.getCalId()));
        }
        if (!ObjectUtils.isEmpty(query.getCalIds())) {
            List<String> list = Arrays.asList(query.getCalIds().split(","));
            List<Long> collect = list.stream().map(item -> Long.parseLong(item)).collect(Collectors.toList());
            jpaQuery.where(qdo.calId.in(collect));
        } else{
            jpaQuery.where(qdo.calId.in(Collections.singletonList(-1L)));
        }
        //查询的时间区间（查询时间区间与日程时间区间有交集）

        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery,qdo._super,query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 动态查询集合(不分页查询)
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdScheduleVO> queryListDynamic(PrdScheduleQuery query) {
        JPAQuery<PrdScheduleVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }



    /**
     * 分页查询
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdScheduleVO> queryPaging(PrdScheduleQuery query){
        JPAQuery<PrdScheduleVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<PrdScheduleVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        System.out.println(result.getTotal());
        return PagingVO.<PrdScheduleVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PrdSchedulePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        if (payload.getTitle() != null) {
            update.set(qdo.title, payload.getTitle());
        }
        if (payload.getDescription() != null) {
            update.set(qdo.description, payload.getDescription());
        }
        if (payload.getStartTime() != null) {
            update.set(qdo.startTime, payload.getStartTime());
        }
        if (payload.getEndTime() != null) {
            update.set(qdo.endTime, payload.getEndTime());
        }
        if (payload.getAllowActiveJoin() != null) {
            update.set(qdo.allowActiveJoin, payload.getAllowActiveJoin());
        }
        if (payload.getIsRepeat() != null) {
            update.set(qdo.isRepeat, payload.getIsRepeat());
        }
        if (payload.getIsRemind() != null) {
            update.set(qdo.isRemind, payload.getIsRemind());
        }
        if (payload.getRemindBeforeEventSecs() != null) {
            update.set(qdo.remindBeforeEventSecs, payload.getRemindBeforeEventSecs());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("endTime")) {
                update.setNull(qdo.endTime);
            }
        }

        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        return update.execute();
    }


}
