package com.elitesland.tw.tw5.server.prd.budget.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.budget.payload.BudgetSubjectDetailPayload;
import com.elitesland.tw.tw5.api.prd.budget.query.BudgetSubjectDetailQuery;
import com.elitesland.tw.tw5.api.prd.budget.vo.BudgetSubjectDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccBudgetItemDO;
import com.elitesland.tw.tw5.server.prd.budget.entity.BudgetSubjectDetailDO;
import com.elitesland.tw.tw5.server.prd.budget.entity.QBudgetSubjectDetailDO;
import com.elitesland.tw.tw5.server.prd.budget.repo.BudgetSubjectDetailRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 预算科目明细
 *
 * @author xxb
 * @date 2023-11-01
 */
@Repository
@RequiredArgsConstructor
public class BudgetSubjectDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BudgetSubjectDetailRepo repo;
    private final QBudgetSubjectDetailDO qdo = QBudgetSubjectDetailDO.budgetSubjectDetailDO;


    private final QAccBudgetItemDO budgetItemQdo = new QAccBudgetItemDO("AccBudgetItemDO");

    private final QAccBudgetItemDO budgetItemQdo0 = new QAccBudgetItemDO("AccBudgetItemDO");

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BudgetSubjectDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BudgetSubjectDetailVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 预算ID
                qdo.budgetId,
                // 科目ID
                qdo.accId,
                // 科目父级ID
                qdo.accParentId,
                // 原始预算金额
                qdo.originalAmt,
                // 预算金额
                qdo.budgetAmt,
                // 已拨付金额
                qdo.feeReleasedAmt,
                // 已使用金额
                qdo.usedAmt,
                // 可用金额
                qdo.availableAmt,
                // 占用金额
                qdo.occupyAmt,
                // 可配置化字段1
                qdo.configurableField1,
                // 可配置化字段2
                qdo.configurableField2,
                // 可配置化字段3
                qdo.configurableField3,
                // 明细控制
                qdo.detailControlFlag,
                qdo.amtProportion,
                qdo.usedAmt,
                qdo.occupyAmt,
                qdo.usedEqvaProportion,
                qdo.usedAmtProportion,
                // 科目科目
                budgetItemQdo.budgetCode.as("accCode"),
                // 科目名称
                budgetItemQdo.budgetName.as("accName"),
                // 上级科目主键
                budgetItemQdo.parentId.as("accParentId")
        )).from(qdo).leftJoin(budgetItemQdo).on(qdo.accId.eq(budgetItemQdo.id)).leftJoin(budgetItemQdo0).on(budgetItemQdo.parentId.eq(budgetItemQdo0.id).and(budgetItemQdo0.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BudgetSubjectDetailVO> getJpaQueryWhere(BudgetSubjectDetailQuery query) {
        JPAQuery<BudgetSubjectDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BudgetSubjectDetailQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BudgetSubjectDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 预算ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetId())) {
            list.add(qdo.budgetId.eq(query.getBudgetId()));
        }
        /** 预算ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetIds())) {
            list.add(qdo.budgetId.in(query.getBudgetIds()));
        }
        /** 科目ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAccId())) {
            list.add(qdo.accId.eq(query.getAccId()));
        }
        /** 科目父级ID 精确 */
        if (!ObjectUtils.isEmpty(query.getAccParentId())) {
            list.add(qdo.accParentId.eq(query.getAccParentId()));
        }
        /** 原始预算金额 精确 */
        if (!ObjectUtils.isEmpty(query.getOriginalAmt())) {
            list.add(qdo.originalAmt.eq(query.getOriginalAmt()));
        }
        /** 预算金额 精确 */
        if (!ObjectUtils.isEmpty(query.getBudgetAmt())) {
            list.add(qdo.budgetAmt.eq(query.getBudgetAmt()));
        }
        /** 已拨付金额 精确 */
        if (!ObjectUtils.isEmpty(query.getFeeReleasedAmt())) {
            list.add(qdo.feeReleasedAmt.eq(query.getFeeReleasedAmt()));
        }
        /** 已使用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getUsedAmt())) {
            list.add(qdo.usedAmt.eq(query.getUsedAmt()));
        }
        /** 可用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAvailableAmt())) {
            list.add(qdo.availableAmt.eq(query.getAvailableAmt()));
        }

        /** 占用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getOccupyAmt())) {
            list.add(qdo.occupyAmt.eq(query.getOccupyAmt()));
        }
        /** 可配置化字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigurableField1())) {
            list.add(qdo.configurableField1.eq(query.getConfigurableField1()));
        }
        /** 可配置化字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigurableField2())) {
            list.add(qdo.configurableField2.eq(query.getConfigurableField2()));
        }
        /** 可配置化字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getConfigurableField3())) {
            list.add(qdo.configurableField3.eq(query.getConfigurableField3()));
        }
        /** 明细控制  精确 */
        if (!ObjectUtils.isEmpty(query.getDetailControlFlag())) {
            list.add(qdo.detailControlFlag.eq(query.getDetailControlFlag()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BudgetSubjectDetailVO queryByKey(Long id) {
        JPAQuery<BudgetSubjectDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据预算项目id查询
     *
     * @param budgetItemId 预算项目id
     * @return 结果
     */
    public BudgetSubjectDetailVO queryByBudgetItemId(Long budgetItemId) {
        JPAQuery<BudgetSubjectDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.accId.eq(budgetItemId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BudgetSubjectDetailVO> queryListDynamic(BudgetSubjectDetailQuery query) {
        JPAQuery<BudgetSubjectDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BudgetSubjectDetailVO> queryPaging(BudgetSubjectDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<BudgetSubjectDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<BudgetSubjectDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BudgetSubjectDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BudgetSubjectDetailDO save(BudgetSubjectDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BudgetSubjectDetailDO> saveAll(List<BudgetSubjectDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BudgetSubjectDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 预算ID
        if (payload.getBudgetId() != null) {
            update.set(qdo.budgetId, payload.getBudgetId());
        }
        // 科目ID
        if (payload.getAccId() != null) {
            update.set(qdo.accId, payload.getAccId());
        }
        // 科目父级ID
        if (payload.getAccParentId() != null) {
            update.set(qdo.accParentId, payload.getAccParentId());
        }
        // 原始预算金额
        if (payload.getOriginalAmt() != null) {
            update.set(qdo.originalAmt, payload.getOriginalAmt());
        }
        // 预算金额
        if (payload.getBudgetAmt() != null) {
            update.set(qdo.budgetAmt, payload.getBudgetAmt());
        }
        // 已拨付金额
        if (payload.getFeeReleasedAmt() != null) {
            update.set(qdo.feeReleasedAmt, payload.getFeeReleasedAmt());
        }
        // 费用占比
        if (payload.getAmtProportion() != null) {
            update.set(qdo.amtProportion, payload.getAmtProportion());
        }
        // 已使用金额
        if (payload.getUsedAmt() != null) {
            update.set(qdo.usedAmt, payload.getUsedAmt());
        }
        // 可用金额
        if (payload.getAvailableAmt() != null) {
            update.set(qdo.availableAmt, payload.getAvailableAmt());
        }
        // 占用金额
        if (payload.getOccupyAmt() != null) {
            update.set(qdo.occupyAmt, payload.getOccupyAmt());
        }
        // 可配置化字段1
        if (payload.getConfigurableField1() != null) {
            update.set(qdo.configurableField1, payload.getConfigurableField1());
        }
        // 可配置化字段2
        if (payload.getConfigurableField2() != null) {
            update.set(qdo.configurableField2, payload.getConfigurableField2());
        }
        // 可配置化字段3
        if (payload.getConfigurableField3() != null) {
            update.set(qdo.configurableField3, payload.getConfigurableField3());
        }
        // 明细控制
        if (payload.getDetailControlFlag() != null) {
            update.set(qdo.detailControlFlag, payload.getDetailControlFlag());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }

        if (payload.getUsedAmt() != null) {
            update.set(qdo.usedAmt, payload.getUsedAmt());
        }
        if (payload.getOccupyAmt() != null) {
            update.set(qdo.occupyAmt, payload.getOccupyAmt());
        }
        if (payload.getUsedEqvaProportion() != null) {
            update.set(qdo.usedEqvaProportion, payload.getUsedEqvaProportion());
        }
        if (payload.getUsedAmtProportion() != null) {
            update.set(qdo.usedAmtProportion, payload.getUsedAmtProportion());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 预算ID
            if (nullFields.contains("budgetId")) {
                update.setNull(qdo.budgetId);
            }
            // 科目ID
            if (nullFields.contains("accId")) {
                update.setNull(qdo.accId);
            }
            // 科目父级ID
            if (nullFields.contains("accParentId")) {
                update.setNull(qdo.accParentId);
            }
            // 原始预算金额
            if (nullFields.contains("originalAmt")) {
                update.setNull(qdo.originalAmt);
            }
            // 预算金额
            if (nullFields.contains("budgetAmt")) {
                update.setNull(qdo.budgetAmt);
            }
            // 已拨付金额
            if (nullFields.contains("feeReleasedAmt")) {
                update.setNull(qdo.feeReleasedAmt);
            }
            // 已使用金额
            if (nullFields.contains("usedAmt")) {
                update.setNull(qdo.usedAmt);
            }
            // 可用金额
            if (nullFields.contains("availableAmt")) {
                update.setNull(qdo.availableAmt);
            }
            // 占用金额
            if (nullFields.contains("occupyAmt")) {
                update.setNull(qdo.occupyAmt);
            }
            // 可配置化字段1
            if (nullFields.contains("configurableField1")) {
                update.setNull(qdo.configurableField1);
            }
            // 可配置化字段2
            if (nullFields.contains("configurableField2")) {
                update.setNull(qdo.configurableField2);
            }
            // 可配置化字段3
            if (nullFields.contains("configurableField3")) {
                update.setNull(qdo.configurableField3);
            }
            // 明细控制
            if (nullFields.contains("detailControlFlag")) {
                update.setNull(qdo.detailControlFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param budgetId 预算主键
     * @return 删除的行数
     */
    public long deleteSoftByBudgetId(Long budgetId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.budgetId.in(budgetId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

