package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalEqvaCostPayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalEqvaCostQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalEqvaCostVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalEqvaCostDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalEqvaCostDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalEqvaCostRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 当量成本配置管理
 *
 * @author carl
 * @date 2023-11-07
 */
@Repository
@RequiredArgsConstructor
public class CalEqvaCostDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalEqvaCostRepo repo;
    private final QCalEqvaCostDO qdo = QCalEqvaCostDO.calEqvaCostDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaCostVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalEqvaCostVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 平台编号
                qdo.busifieldType,
                // buid
                qdo.buId,
                // 当量名称
                qdo.eqvaName,
                // 工种
                qdo.jobType1,
                // 工种子类
                qdo.jobType2,
                // 核算年度
                qdo.finYear,

                // 核算期间
                qdo.finPeriod,
                // 当量成本
                qdo.eqvaCost
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaCostVO> getJpaQueryWhere(CalEqvaCostQuery query) {
        JPAQuery<CalEqvaCostVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalEqvaCostQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalEqvaCostQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 平台编号 精确 */
        if (!ObjectUtils.isEmpty(query.getBusifieldType())) {
            list.add(qdo.busifieldType.eq(query.getBusifieldType()));
        }
        /** buid 精确 */
        if (!ObjectUtils.isEmpty(query.getBuId())) {
            list.add(qdo.buId.eq(query.getBuId()));
        }
        /** 当量名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getEqvaName())) {
            list.add(qdo.eqvaName.like(SqlUtil.toSqlLikeString(query.getEqvaName())));
        }
        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        }
        /** 核算期间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        }
        /** 当量成本 精确 */
        if (!ObjectUtils.isEmpty(query.getEqvaCost())) {
            list.add(qdo.eqvaCost.eq(query.getEqvaCost()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalEqvaCostVO queryByKey(Long id) {
        JPAQuery<CalEqvaCostVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalEqvaCostVO> queryListDynamic(CalEqvaCostQuery query) {
        JPAQuery<CalEqvaCostVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalEqvaCostVO> queryPaging(CalEqvaCostQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalEqvaCostVO> jpaQuery = getJpaQueryWhere(query);
        List<CalEqvaCostVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalEqvaCostVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalEqvaCostDO save(CalEqvaCostDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalEqvaCostDO> saveAll(List<CalEqvaCostDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalEqvaCostPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 平台编号
        if (payload.getBusifieldType() != null) {
            update.set(qdo.busifieldType, payload.getBusifieldType());
        }
        // buid
        if (payload.getBuId() != null) {
            update.set(qdo.buId, payload.getBuId());
        }
        // 当量名称
        if (payload.getEqvaName() != null) {
            update.set(qdo.eqvaName, payload.getEqvaName());
        }
        // 工种
        if (payload.getJobType1() != null) {
            update.set(qdo.jobType1, payload.getJobType1());
        }
        // 工种子类
        if (payload.getJobType2() != null) {
            update.set(qdo.jobType2, payload.getJobType2());
        }
        // 核算年度
        if (payload.getFinYear() != null) {
            update.set(qdo.finYear, payload.getFinYear());
        }
        // 核算期间
        if (payload.getFinPeriod() != null) {
            update.set(qdo.finPeriod, payload.getFinPeriod());
        }
        // 当量成本
        if (payload.getEqvaCost() != null) {
            update.set(qdo.eqvaCost, payload.getEqvaCost());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 平台编号
            if (nullFields.contains("busifieldType")) {
                update.setNull(qdo.busifieldType);
            }
            // buid
            if (nullFields.contains("buId")) {
                update.setNull(qdo.buId);
            }
            // 当量名称
            if (nullFields.contains("eqvaName")) {
                update.setNull(qdo.eqvaName);
            }
            // 工种
            if (nullFields.contains("jobType1")) {
                update.setNull(qdo.jobType1);
            }
            // 工种子类
            if (nullFields.contains("jobType2")) {
                update.setNull(qdo.jobType2);
            }
            // 核算年度
            if (nullFields.contains("finYear")) {
                update.setNull(qdo.finYear);
            }
            // 核算期间
            if (nullFields.contains("finPeriod")) {
                update.setNull(qdo.finPeriod);
            }
            // 当量成本
            if (nullFields.contains("eqvaCost")) {
                update.setNull(qdo.eqvaCost);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate checkDataWhere(CalEqvaCostQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 平台编号 精确 */
        if (!ObjectUtils.isEmpty(query.getBusifieldType())) {
            list.add(qdo.busifieldType.eq(query.getBusifieldType()));
        } else {
            list.add(qdo.busifieldType.isNull());
        }
        /** buid 精确 */
        if (!ObjectUtils.isEmpty(query.getBuId())) {
            list.add(qdo.buId.eq(query.getBuId()));
        } else {
            list.add(qdo.buId.isNull());
        }

        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        } else {
            list.add(qdo.jobType1.isNull());
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        } else {
            list.add(qdo.jobType2.isNull());
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        } else {
            list.add(qdo.finYear.isNull());
        }
        /** 核算期间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        } else {
            list.add(qdo.finPeriod.isNull());
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<Long> getCheckDataWhere(CalEqvaCostQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdo.id).from(qdo);
        // 条件封装
        jpaQuery.where(checkDataWhere(query));
        return jpaQuery;
    }

    /**
     * 校验用
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<Long> queryCheckData(CalEqvaCostQuery query) {
        JPAQuery<Long> jpaQuery = getCheckDataWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate EqvaCostPlusWhere(CalEqvaCostQuery query) {
        List<Predicate> list = new ArrayList<>();


        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()).or(qdo.jobType1.isNull()));
        } else {
            list.add(qdo.jobType1.isNull());
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()).or(qdo.jobType2.isNull()));
        } else {
            list.add(qdo.jobType2.isNull());
        }
//        /** 核算年度 精确 */
//        if (!ObjectUtils.isEmpty(query.getFinYear())) {
//            list.add(qdo.finYear.eq(query.getFinYear()).or(qdo.finYear.isNull()));
//        } else {
//            list.add(qdo.finYear.isNull());
//        }
//        /** 核算期间 精确 */
//        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
//            list.add(qdo.finPeriod.eq(query.getFinPeriod()).or(qdo.finPeriod.isNull()));
//        } else {
//            list.add(qdo.finPeriod.isNull());
//        }
        /** buId 精确 */
        if (!ObjectUtils.isEmpty(query.getBuId())) {
            list.add(qdo.buId.eq(query.getBuId()).or(qdo.buId.isNull()));
        } else {
            list.add(qdo.buId.isNull());
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaCostVO> getEqvaCostPlusWhere(CalEqvaCostQuery query) {
        JPAQuery<CalEqvaCostVO> jpaQuery = jpaQueryFactory.select(Projections.bean(CalEqvaCostVO.class,
                qdo.id,
                // 工种
                qdo.jobType1,
                // 工种子类
                qdo.jobType2,
                // 核算年度
                qdo.finYear,
                // 核算期间id
                qdo.finPeriod,
                // 当量成本
                qdo.eqvaCost,
                // 当量名称
                qdo.eqvaName
        )).from(qdo);
        // 条件封装
        jpaQuery.where(EqvaCostPlusWhere(query));
        List<OrderItem> orderse = new ArrayList<>();

        orderse.add(OrderItem.desc("buId"));
        orderse.add(OrderItem.desc("jobType2"));
        orderse.add(OrderItem.desc("jobType1"));
        orderse.add(OrderItem.desc("finPeriod"));
        orderse.add(OrderItem.desc("finYear"));
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));

        return jpaQuery;
    }

    /**
     * 获取最精准数据匹配
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public CalEqvaCostVO getEqvaCostPlus(CalEqvaCostQuery query) {
        JPAQuery<CalEqvaCostVO> jpaQuery = getEqvaCostPlusWhere(query);
        return jpaQuery.fetchFirst();
    }
}

