package com.elitesland.tw.tw5.server.prd.cas.dao;


import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.cas.entity.PrdCasSettingDO;
import com.elitesland.tw.tw5.api.prd.cas.payload.PrdCasSettingPayload;
import com.elitesland.tw.tw5.api.prd.cas.query.PrdCasSettingQuery;
import com.elitesland.tw.tw5.server.prd.cas.entity.QPrdCasSettingDO;
import com.elitesland.tw.tw5.server.prd.cas.repo.PrdCasSettingRepo;
import com.elitesland.tw.tw5.api.prd.cas.vo.PrdCasSettingVO;
import com.elitesland.tw.tw5.server.prd.org.entity.PrdOrgEmployeeDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * cas_setting
 *
 * @author zoey
 * @date 2024-01-25
 */
@Repository
@RequiredArgsConstructor
public class PrdCasSettingDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PrdCasSettingRepo repo;
    private final QPrdCasSettingDO qdo = QPrdCasSettingDO.prdCasSettingDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdCasSettingVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdCasSettingVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 应用识别码
                qdo.appId,
                // JWT签名key
                qdo.secret,
                // 交叉认证接口
                qdo.validateUri,
                // 跳转页面
                qdo.redirectUri,
                // 请求头信息
                qdo.requestHeaders
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdCasSettingVO> getJpaQueryWhere(PrdCasSettingQuery query) {
        JPAQuery<PrdCasSettingVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PrdCasSettingQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PrdCasSettingQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 应用识别码 精确 */
        if (!ObjectUtils.isEmpty(query.getAppId())) {
            list.add(qdo.appId.eq(query.getAppId()));
        }
        /** JWT签名key 精确 */
        if (!ObjectUtils.isEmpty(query.getSecret())) {
            list.add(qdo.secret.eq(query.getSecret()));
        }
        /** 交叉认证接口 精确 */
        if (!ObjectUtils.isEmpty(query.getValidateUri())) {
            list.add(qdo.validateUri.eq(query.getValidateUri()));
        }
        /** 跳转页面 精确 */
        if (!ObjectUtils.isEmpty(query.getRedirectUri())) {
            list.add(qdo.redirectUri.eq(query.getRedirectUri()));
        }
        /** 请求头信息 精确 */
        if (!ObjectUtils.isEmpty(query.getRequestHeaders())) {
            list.add(qdo.requestHeaders.eq(query.getRequestHeaders()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdCasSettingVO queryByKey(Long id) {
        JPAQuery<PrdCasSettingVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdCasSettingVO> queryListDynamic(PrdCasSettingQuery query) {
        JPAQuery<PrdCasSettingVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdCasSettingVO> queryPaging(PrdCasSettingQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PrdCasSettingVO> jpaQuery = getJpaQueryWhere(query);
        List<PrdCasSettingVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PrdCasSettingVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdCasSettingDO save(PrdCasSettingDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PrdCasSettingDO> saveAll(List<PrdCasSettingDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PrdCasSettingPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 应用识别码
        if (payload.getAppId() != null) {
            update.set(qdo.appId, payload.getAppId());
        }
        // JWT签名key
        if (payload.getSecret() != null) {
            update.set(qdo.secret, payload.getSecret());
        }
        // 交叉认证接口
        if (payload.getValidateUri() != null) {
            update.set(qdo.validateUri, payload.getValidateUri());
        }
        // 跳转页面
        if (payload.getRedirectUri() != null) {
            update.set(qdo.redirectUri, payload.getRedirectUri());
        }
        // 请求头信息
        if (payload.getRequestHeaders() != null) {
            update.set(qdo.requestHeaders, payload.getRequestHeaders());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 应用识别码
            if (nullFields.contains("appId")) {
                update.setNull(qdo.appId);
            }
            // JWT签名key
            if (nullFields.contains("secret")) {
                update.setNull(qdo.secret);
            }
            // 交叉认证接口
            if (nullFields.contains("validateUri")) {
                update.setNull(qdo.validateUri);
            }
            // 跳转页面
            if (nullFields.contains("redirectUri")) {
                update.setNull(qdo.redirectUri);
            }
            // 请求头信息
            if (nullFields.contains("requestHeaders")) {
                update.setNull(qdo.requestHeaders);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public PrdCasSettingDO queryByAppId(String appId) {
        JPAQuery<PrdCasSettingDO> jpaQuery = jpaQueryFactory.select(qdo).from(qdo);
        jpaQuery.where(qdo.appId.eq(appId));
        return jpaQuery.fetchFirst();
    }
}

