package com.elitesland.tw.tw5.server.prd.my.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.OutWorkApplyPayload;
import com.elitesland.tw.tw5.api.prd.my.query.OutWorkApplyQuery;
import com.elitesland.tw.tw5.api.prd.my.vo.OutWorkApplyVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.OutWorkApplyDO;
import com.elitesland.tw.tw5.server.prd.my.entity.QOutWorkApplyDO;
import com.elitesland.tw.tw5.server.prd.my.repo.OutWorkApplyRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import io.swagger.annotations.ApiModelProperty;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

/**
 * 外勤办公申请
 *
 * @author carl
 * @date 2024-05-22
 */
@Repository
@RequiredArgsConstructor
public class OutWorkApplyDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final OutWorkApplyRepo repo;
    private final QOutWorkApplyDO qdo = QOutWorkApplyDO.outWorkApplyDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<OutWorkApplyVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(OutWorkApplyVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 外出申请人id
                qdo.applyUserId,
                // 归属bu_id
                qdo.orgId,
                // base地id
                qdo.baseAddrId,
                // 直属领导id
                qdo.parentUserId,
                // 岗位
                qdo.jobs,
                // 级别
                qdo.grade,
                // 申请日期
                qdo.applyDate,
                // 外勤办公日期
                qdo.outWorkDate,
                // 外勤办公时间开始
                qdo.outWorkTimeStart,
                // 外勤办公时间结束
                qdo.outWorkTimeEnd,
                // 外勤工作天数
                qdo.outWorkDays,
                // 0.5天区分上午下午 AM上午 PM下午
                qdo.outWorkInterval,
                // 外勤工作地
                qdo.outWorkAddr,
                // 编号
                qdo.applyNo,
                // 名称
                qdo.applyName,
                // 任务状态
                qdo.applyStatus,
                // 审批状态
                qdo.procInstStatus,
                // 流程id
                qdo.procInstId
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<OutWorkApplyVO> getJpaQueryWhere(OutWorkApplyQuery query) {
        JPAQuery<OutWorkApplyVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(OutWorkApplyQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(OutWorkApplyQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 璁板綍鍞?竴ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 外出申请人id 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyUserId())) {
            list.add(qdo.applyUserId.eq(query.getApplyUserId()));
        }
        /** 归属bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            list.add(qdo.orgId.eq(query.getOrgId()));
        }
        /** base地id 精确 */
        if (!ObjectUtils.isEmpty(query.getBaseAddrId())) {
            list.add(qdo.baseAddrId.eq(query.getBaseAddrId()));
        }
        /** 直属领导id 精确 */
        if (!ObjectUtils.isEmpty(query.getParentUserId())) {
            list.add(qdo.parentUserId.eq(query.getParentUserId()));
        }
        /** 岗位 精确 */
        if (!ObjectUtils.isEmpty(query.getJobs())) {
            list.add(qdo.jobs.eq(query.getJobs()));
        }
        /** 级别 精确 */
        if (!ObjectUtils.isEmpty(query.getGrade())) {
            list.add(qdo.grade.eq(query.getGrade()));
        }
        /** 申请日期开始 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyStartDate())) {
            list.add(qdo.applyDate.goe(query.getApplyStartDate()));
        }
        /** 申请日期结束 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyEndDate())) {
            list.add(qdo.applyDate.loe(query.getApplyEndDate()));
        }
        /** 外勤办公日期开始 精确 */
        if (!ObjectUtils.isEmpty(query.getOutWorkStartDate())) {
            list.add(qdo.outWorkDate.goe(query.getOutWorkStartDate()));
        }
        /** 外勤办公日期结束 精确 */
        if (!ObjectUtils.isEmpty(query.getOutWorkEndDate())) {
            list.add(qdo.outWorkDate.loe(query.getOutWorkEndDate()));
        }
        /** 外勤工作天数 精确 */
        if (!ObjectUtils.isEmpty(query.getOutWorkDays())) {
            list.add(qdo.outWorkDays.eq(query.getOutWorkDays()));
        }
        /** 0.5天区分上午下午 AM上午 PM下午 精确 */
        if (!ObjectUtils.isEmpty(query.getOutWorkInterval())) {
            list.add(qdo.outWorkInterval.eq(query.getOutWorkInterval()));
        }
        /** 外勤工作地 精确 */
        if (!ObjectUtils.isEmpty(query.getOutWorkAddr())) {
            list.add(qdo.outWorkAddr.eq(query.getOutWorkAddr()));
        }
        /** 编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getApplyNo())) {
            list.add(qdo.applyNo.like(SqlUtil.toSqlLikeString(query.getApplyNo())));
        }
        /** 名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getApplyName())) {
            list.add(qdo.applyName.like(SqlUtil.toSqlLikeString(query.getApplyName())));
        }
        /** 任务状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyStatus())) {
            list.add(qdo.applyStatus.eq(query.getApplyStatus()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 流程id 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public OutWorkApplyVO queryByKey(Long id) {
        JPAQuery<OutWorkApplyVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<OutWorkApplyVO> queryListDynamic(OutWorkApplyQuery query) {
        JPAQuery<OutWorkApplyVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<OutWorkApplyVO> queryPaging(OutWorkApplyQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<OutWorkApplyVO> jpaQuery = getJpaQueryWhere(query);
        List<OutWorkApplyVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<OutWorkApplyVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public OutWorkApplyDO save(OutWorkApplyDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<OutWorkApplyDO> saveAll(List<OutWorkApplyDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(OutWorkApplyPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 璁板綍鍞?竴ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 外出申请人id
        if (payload.getApplyUserId() != null) {
            update.set(qdo.applyUserId, payload.getApplyUserId());
        }
        // 归属bu_id
        if (payload.getOrgId() != null) {
            update.set(qdo.orgId, payload.getOrgId());
        }
        // base地id
        if (payload.getBaseAddrId() != null) {
            update.set(qdo.baseAddrId, payload.getBaseAddrId());
        }
        // 直属领导id
        if (payload.getParentUserId() != null) {
            update.set(qdo.parentUserId, payload.getParentUserId());
        }
        // 岗位
        if (payload.getJobs() != null) {
            update.set(qdo.jobs, payload.getJobs());
        }
        // 级别
        if (payload.getGrade() != null) {
            update.set(qdo.grade, payload.getGrade());
        }
        // 申请日期
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 外勤办公日期
        if (payload.getOutWorkDate() != null) {
            update.set(qdo.outWorkDate, payload.getOutWorkDate());
        }
        // 外勤办公时间开始
        if (payload.getOutWorkTimeStart() != null) {
            update.set(qdo.outWorkTimeStart, payload.getOutWorkTimeStart());
        }
        // 外勤办公时间结束
        if (payload.getOutWorkTimeEnd() != null) {
            update.set(qdo.outWorkTimeEnd, payload.getOutWorkTimeEnd());
        }
        // 外勤工作天数
        if (payload.getOutWorkDays() != null) {
            update.set(qdo.outWorkDays, payload.getOutWorkDays());
        }
        // 0.5天区分上午下午 AM上午 PM下午
        if (payload.getOutWorkInterval() != null) {
            update.set(qdo.outWorkInterval, payload.getOutWorkInterval());
        }
        // 外勤工作地
        if (payload.getOutWorkAddr() != null) {
            update.set(qdo.outWorkAddr, payload.getOutWorkAddr());
        }
        // 编号
        if (payload.getApplyNo() != null) {
            update.set(qdo.applyNo, payload.getApplyNo());
        }
        // 名称
        if (payload.getApplyName() != null) {
            update.set(qdo.applyName, payload.getApplyName());
        }
        // 任务状态
        if (payload.getApplyStatus() != null) {
            update.set(qdo.applyStatus, payload.getApplyStatus());
        }
        // 审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        // 流程id
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 璁板綍鍞?竴ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 外出申请人id
            if (nullFields.contains("applyUserId")) {
                update.setNull(qdo.applyUserId);
            }
            // 归属bu_id
            if (nullFields.contains("orgId")) {
                update.setNull(qdo.orgId);
            }
            // base地id
            if (nullFields.contains("baseAddrId")) {
                update.setNull(qdo.baseAddrId);
            }
            // 直属领导id
            if (nullFields.contains("parentUserId")) {
                update.setNull(qdo.parentUserId);
            }
            // 岗位
            if (nullFields.contains("jobs")) {
                update.setNull(qdo.jobs);
            }
            // 级别
            if (nullFields.contains("grade")) {
                update.setNull(qdo.grade);
            }
            // 申请日期
            if (nullFields.contains("applyDate")) {
                update.setNull(qdo.applyDate);
            }
            // 外勤办公日期
            if (nullFields.contains("outWorkDate")) {
                update.setNull(qdo.outWorkDate);
            }
            // 外勤办公时间开始
            if (nullFields.contains("outWorkTimeStart")) {
                update.setNull(qdo.outWorkTimeStart);
            }
            // 外勤办公时间结束
            if (nullFields.contains("outWorkTimeEnd")) {
                update.setNull(qdo.outWorkTimeEnd);
            }
            // 外勤工作天数
            if (nullFields.contains("outWorkDays")) {
                update.setNull(qdo.outWorkDays);
            }
            // 0.5天区分上午下午 AM上午 PM下午
            if (nullFields.contains("outWorkInterval")) {
                update.setNull(qdo.outWorkInterval);
            }
            // 外勤工作地
            if (nullFields.contains("outWorkAddr")) {
                update.setNull(qdo.outWorkAddr);
            }
            // 编号
            if (nullFields.contains("applyNo")) {
                update.setNull(qdo.applyNo);
            }
            // 名称
            if (nullFields.contains("applyName")) {
                update.setNull(qdo.applyName);
            }
            // 任务状态
            if (nullFields.contains("applyStatus")) {
                update.setNull(qdo.applyStatus);
            }
            // 审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 流程id
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

