package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectConclusionPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectConclusionQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectConclusionVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.budget.entity.QBudgetDO;
import com.elitesland.tw.tw5.server.prd.pms.common.functionEnum.PmsReasonTypeEnum;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectConclusionDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectConclusionDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectConclusionRepo;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目结项
 *
 * @author xxb
 * @date 2023-11-27
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectConclusionDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectConclusionRepo repo;
    private final QPmsProjectConclusionDO qdo = QPmsProjectConclusionDO.pmsProjectConclusionDO;

    private final QPmsProjectDO projectQdo = QPmsProjectDO.pmsProjectDO;

    private final QSaleConContractDO qdoSaleCon = QSaleConContractDO.saleConContractDO;
    private final QBudgetDO qdoBudget = QBudgetDO.budgetDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectConclusionVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectConclusionVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 异常结项标记
                qdo.abnormalFlag,
                // 结项状态
                qdo.conclusionStatus,
                // 审批状态
                qdo.apprStatus,
                // 流程实例ID
                qdo.procInstId,
                // 提交资源userid
                qdo.applyUserId,
                // 申请日期
                qdo.applyDate,
                // 结项检查事项
                qdo.conclusionCheckItems,
                // 项目id
                qdo.projId,
                // 结项编号
                qdo.conclusionCode,
                // 节点操作详情
                qdo.approvalNodeInfo
        )).from(qdo);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectConclusionVO> getJpaQuerySelectJoinProject() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectConclusionVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 异常结项标记
                qdo.abnormalFlag,
                // 结项状态
                qdo.conclusionStatus,
                // 审批状态
                qdo.apprStatus,
                // 流程实例ID
                qdo.procInstId,
                // 提交资源userid
                qdo.applyUserId,
                // 申请日期
                qdo.applyDate,
                // 结项检查事项
                qdo.conclusionCheckItems,
                // 项目id
                qdo.projId,
                // 结项编号
                qdo.conclusionCode,
                qdo.approvalNodeInfo,
                //    项目编号、项目名称、子合同编号、项目状态、工作类型、项目经理、交付BU、交付负责人、销售负责人
                projectQdo.projNo,
                projectQdo.projName,
                projectQdo.projStatus,
                projectQdo.pmEqvaRatio,
                projectQdo.eqvaPrice,
                qdoSaleCon.workType,
                projectQdo.pmResId,
                projectQdo.deliBuId,
                projectQdo.deliUserId,
                projectQdo.totalDays,
                projectQdo.totalEqva,
                projectQdo.totalCost,
                projectQdo.totalReimbursement,
                projectQdo.budgetFiles,
                projectQdo.contractId,
                qdoSaleCon.platType,
                qdoSaleCon.code.as("contractNo"),
                qdoSaleCon.saleManUserId,
                // 产品大类
                qdoSaleCon.productClass,
                // 产品小类
                qdoSaleCon.productSubClass
        )).from(qdo).leftJoin(projectQdo).on(qdo.projId.eq(projectQdo.id)).leftJoin(qdoSaleCon).on(projectQdo.contractId.eq(qdoSaleCon.id));
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectConclusionVO> getJpaQuerySelectJoinProject0() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectConclusionVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 异常结项标记
                qdo.abnormalFlag,
                // 结项状态
                qdo.conclusionStatus,
                // 审批状态
                qdo.apprStatus,
                // 流程实例ID
                qdo.procInstId,
                // 提交资源userid
                qdo.applyUserId,
                // 申请日期
                qdo.applyDate,
                // 结项检查事项
                qdo.conclusionCheckItems,
                // 项目id
                qdo.projId,
                // 结项编号
                qdo.conclusionCode,
                qdo.approvalNodeInfo,
                //    项目编号、项目名称、子合同编号、项目状态、工作类型、项目经理、交付BU、交付负责人、销售负责人
                projectQdo.projNo,
                projectQdo.projName,
                projectQdo.projStatus,
                projectQdo.pmEqvaRatio,
                projectQdo.eqvaPrice,
                qdoSaleCon.workType,
                projectQdo.pmResId,
                projectQdo.deliBuId,
                projectQdo.deliUserId,
                projectQdo.totalDays,
                projectQdo.totalEqva,
                projectQdo.totalCost,
                projectQdo.totalReimbursement,
                projectQdo.jdeConfirmTime,
                // projectQdo.budgetFiles, 项目中的预算字段灭用
                projectQdo.contractId,
                qdoSaleCon.platType,
                qdoSaleCon.code.as("contractNo"),
                qdoSaleCon.saleManUserId,
                // 产品大类
                qdoSaleCon.productClass,
                // 产品小类
                qdoSaleCon.productSubClass,
                qdoBudget.budgetFiles
        )).from(qdo).leftJoin(projectQdo).on(qdo.projId.eq(projectQdo.id)).leftJoin(qdoBudget).on(qdo.projId.eq(qdoBudget.sourceId).and(qdoBudget.sourceType.eq(PmsReasonTypeEnum.PROJ_CONTRACT.getCode()))).leftJoin(qdoSaleCon).on(projectQdo.contractId.eq(qdoSaleCon.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectConclusionVO> getJpaQueryWhere(PmsProjectConclusionQuery query) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectConclusionVO> getJpaQuerySelectJoinProject(PmsProjectConclusionQuery query) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelectJoinProject();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectConclusionQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count0(PmsProjectConclusionQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(projectQdo).on(qdo.projId.eq(projectQdo.id)).leftJoin(qdoSaleCon).on(projectQdo.contractId.eq(qdoSaleCon.id));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectConclusionQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 异常结项标记 精确 */
        if (!ObjectUtils.isEmpty(query.getAbnormalFlag())) {
            list.add(qdo.abnormalFlag.eq(query.getAbnormalFlag()));
        }
        /** 结项状态 精确 */
        if (!ObjectUtils.isEmpty(query.getConclusionStatus())) {
            list.add(qdo.conclusionStatus.eq(query.getConclusionStatus()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApprStatus())) {
            list.add(qdo.apprStatus.eq(query.getApprStatus()));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 提交资源userid 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyUserId())) {
            list.add(qdo.applyUserId.eq(query.getApplyUserId()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qdo.applyDate.eq(query.getApplyDate()));
        }
        if (!ObjectUtils.isEmpty(query.getStartDate()) && !ObjectUtils.isEmpty(query.getEndDate())) {
            list.add(qdo.applyDate.between(query.getStartDate(), query.getEndDate()));
        }
        /** 结项检查事项 精确 */
        if (!ObjectUtils.isEmpty(query.getConclusionCheckItems())) {
            list.add(qdo.conclusionCheckItems.eq(query.getConclusionCheckItems()));
        }
        /** 结项编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getConclusionCode())) {
            list.add(qdo.conclusionCode.like(SqlUtil.toSqlLikeString(query.getConclusionCode())));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }

        /** 合同编号/名称 */
        if (!ObjectUtils.isEmpty(query.getContractNo())) {
            list.add(qdoSaleCon.code.like(SqlUtil.toSqlLikeString(query.getContractNo())));
        }
        /** 项目编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjNo())) {
            list.add(projectQdo.projNo.like(SqlUtil.toSqlLikeString(query.getProjNo())));
        }
        /** 项目状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProjStatus())) {
            list.add(projectQdo.projStatus.eq(query.getProjStatus()));
        }
        /** 项目经理资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getPmResId())) {
            list.add(projectQdo.pmResId.eq(query.getPmResId()));
        }
        /** 交付BU_ID */
        if (!ObjectUtils.isEmpty(query.getDeliBuId())) {
            list.add(projectQdo.deliBuId.eq(query.getDeliBuId()));
        }
        /** 交付负责人id */
        if (!ObjectUtils.isEmpty(query.getDeliUserId())) {
            list.add(projectQdo.deliUserId.eq(query.getDeliUserId()));
        }
        /** 销售负责人ID */
        if (!ObjectUtils.isEmpty(query.getSaleManUserId())) {
            list.add(projectQdo.saleManUserId.eq(query.getSaleManUserId()));
        }
        //需要权限
        if (query.getPermissionFlag()) {
            //研发需求：#7514 项目结项申请列表 -> 除运维管理员外，所有用户仅看自己提交的数据
            list.add(qdo.applyUserId.eq(query.getLoginUserId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectConclusionVO queryByKeyJoinProject(Long id) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelectJoinProject0();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectConclusionVO queryByKey(Long id) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键集合查询
     *
     * @param ids
     * @return
     */
    public List<PmsProjectConclusionVO> queryByKeys(List<Long> ids) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


    /**
     * 根据项目ID查询项目结论信息
     *
     * @param projId 项目ID
     * @return 项目结论信息
     */
    public PmsProjectConclusionVO queryByProjId(Long projId) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelectJoinProject();
        jpaQuery.where(qdo.projId.eq(projId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectConclusionVO> queryListDynamic(PmsProjectConclusionQuery query) {
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectConclusionVO> queryPaging(PmsProjectConclusionQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectConclusionVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectConclusionVO>builder().records(result).total(total).build();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectConclusionVO> queryPagingJoinProject(PmsProjectConclusionQuery query) {
        long total = count0(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectConclusionVO> jpaQuery = getJpaQuerySelectJoinProject(query);
        List<PmsProjectConclusionVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectConclusionVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectConclusionDO save(PmsProjectConclusionDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectConclusionDO> saveAll(List<PmsProjectConclusionDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectConclusionPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 异常结项标记
        if (payload.getAbnormalFlag() != null) {
            update.set(qdo.abnormalFlag, payload.getAbnormalFlag());
        }
        // 结项状态
        if (payload.getConclusionStatus() != null) {
            update.set(qdo.conclusionStatus, payload.getConclusionStatus());
        }
        // 审批状态
        if (payload.getApprStatus() != null) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 提交资源userid
        if (payload.getApplyUserId() != null) {
            update.set(qdo.applyUserId, payload.getApplyUserId());
        }
        // 申请日期
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 结项检查事项
        if (payload.getConclusionCheckItems() != null) {
            update.set(qdo.conclusionCheckItems, payload.getConclusionCheckItems());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 结项编号
        if (payload.getConclusionCode() != null) {
            update.set(qdo.conclusionCode, payload.getConclusionCode());
        }
        // 节点操作详情
        if (payload.getApprovalNodeInfo() != null) {
            update.set(qdo.approvalNodeInfo, payload.getApprovalNodeInfo());
        }
        // 删除标志
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 异常结项标记
            if (nullFields.contains("abnormalFlag")) {
                update.setNull(qdo.abnormalFlag);
            }
            // 结项状态
            if (nullFields.contains("conclusionStatus")) {
                update.setNull(qdo.conclusionStatus);
            }
            // 审批状态
            if (nullFields.contains("apprStatus")) {
                update.setNull(qdo.apprStatus);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 提交资源userid
            if (nullFields.contains("applyUserId")) {
                update.setNull(qdo.applyUserId);
            }
            // 申请日期
            if (nullFields.contains("applyDate")) {
                update.setNull(qdo.applyDate);
            }
            // 结项检查事项
            if (nullFields.contains("conclusionCheckItems")) {
                update.setNull(qdo.conclusionCheckItems);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 结项编号
            if (nullFields.contains("conclusionCode")) {
                update.setNull(qdo.conclusionCode);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

