package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConReceivableDPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConReceivableDQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConReceivableDVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConReceivableDDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConReceivableDDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConReceivableDRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 合同收款明细
 *
 * @author lemon
 * @date 2023-09-05
 */
@Repository
@RequiredArgsConstructor
public class ConReceivableDDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConReceivableDRepo repo;
    private final QConReceivableDDO qdo = QConReceivableDDO.conReceivableDDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConReceivableDVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConReceivableDVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 收款主表ID
                qdo.recvId,
                // 收款计划ID
                qdo.recvplanId,
                qdo.recvAmt
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConReceivableDVO> getJpaQueryWhere(ConReceivableDQuery query) {
        JPAQuery<ConReceivableDVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConReceivableDQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConReceivableDQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 收款主表ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRecvId())) {
            list.add(qdo.recvId.eq(query.getRecvId()));
        }
        /** 收款计划ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRecvplanId())) {
            list.add(qdo.recvplanId.eq(query.getRecvplanId()));
        }
        if (!ObjectUtils.isEmpty(query.getRecvAmt())) {
            list.add(qdo.recvAmt.eq(query.getRecvAmt()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConReceivableDVO queryByKey(Long id) {
        JPAQuery<ConReceivableDVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConReceivableDVO> queryListDynamic(ConReceivableDQuery query) {
        JPAQuery<ConReceivableDVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConReceivableDVO> queryPaging(ConReceivableDQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConReceivableDVO> jpaQuery = getJpaQueryWhere(query);
        List<ConReceivableDVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConReceivableDVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConReceivableDDO save(ConReceivableDDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConReceivableDDO> saveAll(List<ConReceivableDDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConReceivableDPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 收款主表ID
        if (payload.getRecvId() != null) {
            update.set(qdo.recvId, payload.getRecvId());
        }
        // 收款计划ID
        if (payload.getRecvplanId() != null) {
            update.set(qdo.recvplanId, payload.getRecvplanId());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 收款主表ID
            if (nullFields.contains("recvId")) {
                update.setNull(qdo.recvId);
            }
            // 收款计划ID
            if (nullFields.contains("recvplanId")) {
                update.setNull(qdo.recvplanId);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

