package com.elitesland.tw.tw5.server.prd.acc.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimDetailMealPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimDetailMealQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimDetailMealVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimDetailMealDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimDetailMealDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimDetailMealRepo;
import com.elitesland.tw.tw5.server.prd.my.entity.QTAttendanceNormalDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 报销明细餐费记录
 *
 * @author zoey
 * @date 2024-06-12
 */
@Repository
@RequiredArgsConstructor
public class AccReimDetailMealDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimDetailMealRepo repo;
    private final QAccReimDetailMealDO qdo = QAccReimDetailMealDO.accReimDetailMealDO;
    private final QTAttendanceNormalDO qdoAttendanceNormal = QTAttendanceNormalDO.tAttendanceNormalDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimDetailMealVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimDetailMealVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 报销单明细ID
                qdo.reimDetailId,
                // 打卡记录id
                qdo.attendanceId,
//                // 法定假期id
//                qdo.vacationId,
                // 附件
                qdo.fileCode,
                // 餐费日期
                qdo.mealDate,
                // 餐费
                qdo.mealAmt,
                // 打卡城市
                qdo.attendanceCity,
                // 上班打卡
                qdoAttendanceNormal.attendanceTimeStart,
                // 下班打卡
                qdoAttendanceNormal.attendanceTimeEnd
        )).from(qdo).leftJoin(qdoAttendanceNormal).on(qdoAttendanceNormal.id.longValue().eq(qdo.attendanceId.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimDetailMealVO> getJpaQueryWhere(AccReimDetailMealQuery query) {
        JPAQuery<AccReimDetailMealVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimDetailMealQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimDetailMealQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 报销单明细ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimDetailId())) {
            list.add(qdo.reimDetailId.eq(query.getReimDetailId()));
        }
        /** 打卡记录id 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceId())) {
            list.add(qdo.attendanceId.eq(query.getAttendanceId()));
        }
        /** 打卡记录id 精确 */
        if (!ObjectUtils.isEmpty(query.getAttendanceIds())) {
            list.add(qdo.attendanceId.in(query.getAttendanceIds()));
        }
//        /** 法定假期id 精确 */
//        if (!ObjectUtils.isEmpty(query.getVacationId())) {
//            list.add(qdo.vacationId.eq(query.getVacationId()));
//        }
        /** 附件 精确 */
        if (!ObjectUtils.isEmpty(query.getFileCode())) {
            list.add(qdo.fileCode.eq(query.getFileCode()));
        }
        /** 餐费日期 精确 */
        if (!ObjectUtils.isEmpty(query.getMealDate())) {
            list.add(qdo.mealDate.between(query.getMealDate().get(0), query.getMealDate().get(1)));
        }
        /** 打卡城市 */
        if (!ObjectUtils.isEmpty(query.getAttendanceCity())) {
            list.add(qdo.attendanceCity.eq(query.getAttendanceCity()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimDetailMealVO queryByKey(Long id) {
        JPAQuery<AccReimDetailMealVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimDetailMealVO> queryListDynamic(AccReimDetailMealQuery query) {
        JPAQuery<AccReimDetailMealVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimDetailMealVO> queryPaging(AccReimDetailMealQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimDetailMealVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimDetailMealVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimDetailMealVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimDetailMealDO save(AccReimDetailMealDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimDetailMealDO> saveAll(List<AccReimDetailMealDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimDetailMealPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 报销单明细ID
        if (payload.getReimDetailId() != null) {
            update.set(qdo.reimDetailId, payload.getReimDetailId());
        }
        // 打卡记录id
        if (payload.getAttendanceId() != null) {
            update.set(qdo.attendanceId, payload.getAttendanceId());
        }
//        // 法定假期id
//        if (payload.getVacationId() != null) {
//            update.set(qdo.vacationId, payload.getVacationId());
//        }
        // 附件
        if (payload.getFileCode() != null) {
            update.set(qdo.fileCode, payload.getFileCode());
        }
        // 餐费日期
        if (payload.getMealDate() != null) {
            update.set(qdo.mealDate, payload.getMealDate());
        }
        // 餐费
        if (payload.getMealAmt() != null) {
            update.set(qdo.mealAmt, payload.getMealAmt());
        }
        // 打卡城市
        if (payload.getAttendanceCity() != null) {
            update.set(qdo.attendanceCity, payload.getAttendanceCity());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 报销单明细ID
            if (nullFields.contains("reimDetailId")) {
                update.setNull(qdo.reimDetailId);
            }
            // 打卡记录id
            if (nullFields.contains("attendanceId")) {
                update.setNull(qdo.attendanceId);
            }
//            // 法定假期id
//            if (nullFields.contains("vacationId")) {
//                update.setNull(qdo.vacationId);
//            }
            // 附件
            if (nullFields.contains("fileCode")) {
                update.setNull(qdo.fileCode);
            }
            // 餐费日期
            if (nullFields.contains("mealDate")) {
                update.setNull(qdo.mealDate);
            }
            // 餐费
            if (nullFields.contains("mealAmt")) {
                update.setNull(qdo.mealAmt);
            }
            if (nullFields.contains("attendanceCity")) {
                update.setNull(qdo.attendanceCity);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

