package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseContractQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseContractVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseContractDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 旧采购合同
 *
 * @author likunpeng
 * @date 2024-01-03
 */
@Repository
@RequiredArgsConstructor
public class PurchaseContractDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final QPurchaseContractDO qdo = QPurchaseContractDO.purchaseContractDO;
    private final QSaleConContractDO qSaleConContractDO = QSaleConContractDO.saleConContractDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseContractVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchaseContractVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 合同状态
                qdo.contractStatus,
                // 主合同id
                qdo.mainContractId,
                // 子合同id
                qdo.subContractId,
                // 关联子合同名称
                qSaleConContractDO.name.as("subContractName"),
                // 关联子合同编号
                qSaleConContractDO.code.as("subContractCode"),
                // 供应商id
                qdo.supplierId,
                // 供应商bu_id
                qdo.supplierBuId,
                // 签约日期
                qdo.signDate,
                // 平台合同类型
                qdo.platType,
                // 采购类型
                qdo.purchaseType,
                // 采购大类
                qdo.purchaseType1,
                // 采购小类
                qdo.purchaseType2,
                // 采购产品
                qdo.productName,
                // 采购内容简述
                qdo.briefDesc,
                // 合同金额
                qdo.amt,
                // 税率
                qdo.taxRate,
                // 采购主体bu_id
                qdo.purchaseBuId,
                // 签单bu_id
                qdo.signBuId,
                // 销售人员资源id
                qdo.salesmanResId,
                // 交付bu_id
                qdo.deliBuId,
                // 交付资源id
                qdo.deliResId,
                // 激活日期
                qdo.activateDate,
                // 关闭日期
                qdo.closeDate,
                // 关闭原因
                qdo.closeReason,
                // 币种
                qdo.currCode,
                // 是否第三方外包（是/否）
                qdo.thirdPartFlag,
                // 采购负责人
                qdo.purchaseInchargeResId,
                // 特殊码
                qdo.specCode,
                // 服务类型（用于区分采购类型和费用参与收益归属问题）
                qdo.serviceType,
                // 采购合同编号
                qdo.contractNo,
                // 采购合同名称
                qdo.contractName,
                // 供应商法人地址簿号
                qdo.supplierLegalBookId,
                // 采购法人地址簿号
                qdo.purchaseLegalBookId,
                // 审批状态
                qdo.apprStatus
        )).from(qdo)
                .leftJoin(qSaleConContractDO).on(qdo.subContractId.eq(qSaleConContractDO.id)
                        .and(qSaleConContractDO.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseContractVO> getJpaQueryWhere(PurchaseContractQuery query) {
        JPAQuery<PurchaseContractVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchaseContractQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchaseContractQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 合同名称/编号 精确*/
        if (!ObjectUtils.isEmpty(query.getContractNameOrNo())) {
            list.add(qdo.contractName.like("%" + query.getContractNameOrNo() + "%")
                    .or(qdo.contractNo.like("%" + query.getContractNameOrNo() + "%")));
        }
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 采购主体bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseBuId())) {
            list.add(qdo.purchaseBuId.eq(query.getPurchaseBuId()));
        }
        /** 创建人 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserId())) {
            list.add(qdo.createUserId.eq(query.getCreateUserId()));
        }
        /** 采购大类 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseType1())) {
            list.add(qdo.purchaseType1.eq(query.getPurchaseType1()));
        }
        /** 合同状态 精确 */
        if (!ObjectUtils.isEmpty(query.getContractStatus())) {
            list.add(qdo.contractStatus.eq(query.getContractStatus()));
        }
        /** 采购类型 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseType())) {
            list.add(qdo.purchaseType.eq(query.getPurchaseType()));
        }
        /** 子合同id 精确 */
        if (!ObjectUtils.isEmpty(query.getSubContractId())) {
            list.add(qdo.subContractId.eq(query.getSubContractId()));
        }
        /** 关闭原因 精确 */
        if (!ObjectUtils.isEmpty(query.getCloseReason())) {
            list.add(qdo.closeReason.eq(query.getCloseReason()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchaseContractVO queryByKey(Long id) {
        JPAQuery<PurchaseContractVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchaseContractVO> queryListDynamic(PurchaseContractQuery query) {
        JPAQuery<PurchaseContractVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchaseContractVO> queryPaging(PurchaseContractQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PurchaseContractVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchaseContractVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchaseContractVO>builder().records(result).total(total).build();
    }

}

