package com.elitesland.tw.tw5.server.prd.salecon.dao;


import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConInvSettingPayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConInvSettingQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConInvSettingVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.partner.common.entity.QBusinessPartnerDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConInvSettingDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConInvSettingDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConInvSettingRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * ConInvSettingController
 *
 * @author zoey
 * @date 2024-03-14
 */
@Repository
@RequiredArgsConstructor
public class ConInvSettingDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConInvSettingRepo repo;
    private final QConInvSettingDO qdo = QConInvSettingDO.conInvSettingDO;
    private final QBusinessPartnerDO qBusinessPartnerDO = new QBusinessPartnerDO("qBusinessPartnerDO");     // 合同表(子合同)

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConInvSettingVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConInvSettingVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 开票年度
                qdo.invYear,
                // 开票主体
                qdo.invOuId,
                // 开票主体公司名称
                //qPrdOrgCompanyDO.companyName.as("invOuName"),
                qBusinessPartnerDO.partnerName.as("invOuName"),
                // 状态:1 有效；0 无效
                qdo.effective,
                // 月最后开票日
                qdo.deadline
        )).from(qdo)
                .leftJoin(qBusinessPartnerDO).on(qBusinessPartnerDO.bookId.eq(qdo.invOuId).and(qBusinessPartnerDO.deleteFlag.eq(0)));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConInvSettingVO> getJpaQueryWhere(ConInvSettingQuery query) {
        JPAQuery<ConInvSettingVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            OrderItem orderItem = OrderItem.desc("invYear");
            List<OrderItem> orderItemList = new ArrayList<>();
            orderItemList.add(orderItem);
            query.setOrders(orderItemList);
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConInvSettingQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConInvSettingQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 开票年度 精确 */
        if (!ObjectUtils.isEmpty(query.getInvYear())) {
            list.add(qdo.invYear.eq(query.getInvYear()));
        }
        /** 开票主体 精确 */
        if (!ObjectUtils.isEmpty(query.getInvOuId())) {
            list.add(qdo.invOuId.eq(query.getInvOuId()));
        }
        /** 状态:1 有效；0 无效 精确 */
        if (!ObjectUtils.isEmpty(query.getEffective())) {
            list.add(qdo.effective.eq(query.getEffective()));
        }
        /** 月最后开票日 精确 */
        if (!ObjectUtils.isEmpty(query.getDeadline())) {
            list.add(qdo.deadline.eq(query.getDeadline()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConInvSettingVO queryByKey(Long id) {
        JPAQuery<ConInvSettingVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConInvSettingVO> queryListDynamic(ConInvSettingQuery query) {
        JPAQuery<ConInvSettingVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConInvSettingVO> queryPaging(ConInvSettingQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConInvSettingVO> jpaQuery = getJpaQueryWhere(query);
        List<ConInvSettingVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConInvSettingVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConInvSettingDO save(ConInvSettingDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConInvSettingDO> saveAll(List<ConInvSettingDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConInvSettingPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 开票年度
        if (payload.getInvYear() != null) {
            update.set(qdo.invYear, payload.getInvYear());
        }
        // 开票主体
        if (payload.getInvOuId() != null) {
            update.set(qdo.invOuId, payload.getInvOuId());
        }
        // 状态:1 有效；0 无效
        if (payload.getEffective() != null) {
            update.set(qdo.effective, payload.getEffective());
        }
        // 月最后开票日
        if (payload.getDeadline() != null) {
            update.set(qdo.deadline, payload.getDeadline());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 开票年度
            if (nullFields.contains("invYear")) {
                update.setNull(qdo.invYear);
            }
            // 开票主体
            if (nullFields.contains("invOuId")) {
                update.setNull(qdo.invOuId);
            }
            // 状态:1 有效；0 无效
            if (nullFields.contains("effective")) {
                update.setNull(qdo.effective);
            }
            // 月最后开票日
            if (nullFields.contains("deadline")) {
                update.setNull(qdo.deadline);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

