package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.salecon.payload.ConReceivablePayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConReceivableQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConReceivableVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConReceivableDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConReceivableDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConReceivableRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

/**
 * 合同收款
 *
 * @author lemon
 * @date 2023-08-31
 */
@Repository
@RequiredArgsConstructor
public class ConReceivableDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConReceivableRepo repo;
    private final QConReceivableDO qdo = QConReceivableDO.conReceivableDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConReceivableVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConReceivableVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 收款批次号
                qdo.recvNo,
                // 收款类别 发票收款/预收款
                qdo.recvClass,
                // 来源ID
                qdo.sourceId,
                // 收款金额
                qdo.recvAmt,
                // 收款日期
                qdo.recvDate,
                // 银行账号
                qdo.accountNo,
                // 总账日期
                qdo.ledgerDate
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConReceivableVO> getJpaQueryWhere(ConReceivableQuery query) {
        JPAQuery<ConReceivableVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConReceivableQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConReceivableQuery query){
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 收款批次号 精确 */
        if (!ObjectUtils.isEmpty(query.getRecvNo())) {
            list.add(qdo.recvNo.eq(query.getRecvNo()));
        }
        /** 收款类别 发票收款/预收款 精确 */
        if (!ObjectUtils.isEmpty(query.getRecvClass())) {
            list.add(qdo.recvClass.eq(query.getRecvClass()));
        }
        /** 来源ID 精确 */
        if (!ObjectUtils.isEmpty(query.getSourceId())) {
            list.add(qdo.sourceId.eq(query.getSourceId()));
        }
        /** 收款金额 精确 */
        if (!ObjectUtils.isEmpty(query.getRecvAmt())) {
            list.add(qdo.recvAmt.eq(query.getRecvAmt()));
        }
        /** 收款日期 精确 */
        if (!ObjectUtils.isEmpty(query.getRecvDate())) {
            list.add(qdo.recvDate.eq(query.getRecvDate()));
        }
        /** 银行账号 精确 */
        if (!ObjectUtils.isEmpty(query.getAccountNo())) {
            list.add(qdo.accountNo.eq(query.getAccountNo()));
        }
        /** 总账日期 精确 */
        if (!ObjectUtils.isEmpty(query.getLedgerDate())) {
            list.add(qdo.ledgerDate.eq(query.getLedgerDate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConReceivableVO queryByKey(Long id) {
        JPAQuery<ConReceivableVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConReceivableVO> queryListDynamic(ConReceivableQuery query) {
        JPAQuery<ConReceivableVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConReceivableVO> queryPaging(ConReceivableQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConReceivableVO> jpaQuery = getJpaQueryWhere(query);
        List<ConReceivableVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConReceivableVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConReceivableDO save(ConReceivableDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConReceivableDO> saveAll(List<ConReceivableDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConReceivablePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 收款批次号
        if (payload.getRecvNo() != null) {
            update.set(qdo.recvNo, payload.getRecvNo());
        }
        // 收款类别 发票收款/预收款
        if (payload.getRecvClass() != null) {
            update.set(qdo.recvClass, payload.getRecvClass());
        }
        // 来源ID
        if (payload.getSourceId() != null) {
            update.set(qdo.sourceId, payload.getSourceId());
        }
        // 收款金额
        if (payload.getRecvAmt() != null) {
            update.set(qdo.recvAmt, payload.getRecvAmt());
        }
        // 收款日期
        if (payload.getRecvDate() != null) {
            update.set(qdo.recvDate, payload.getRecvDate());
        }
        // 银行账号
        if (payload.getAccountNo() != null) {
            update.set(qdo.accountNo, payload.getAccountNo());
        }
        // 总账日期
        if (payload.getLedgerDate() != null) {
            update.set(qdo.ledgerDate, payload.getLedgerDate());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 收款批次号
            if (nullFields.contains("recvNo")) {
                update.setNull(qdo.recvNo);
            }
            // 收款类别 发票收款/预收款
            if (nullFields.contains("recvClass")) {
                update.setNull(qdo.recvClass);
            }
            // 来源ID
            if (nullFields.contains("sourceId")) {
                update.setNull(qdo.sourceId);
            }
            // 收款金额
            if (nullFields.contains("recvAmt")) {
                update.setNull(qdo.recvAmt);
            }
            // 收款日期
            if (nullFields.contains("recvDate")) {
                update.setNull(qdo.recvDate);
            }
            // 银行账号
            if (nullFields.contains("accountNo")) {
                update.setNull(qdo.accountNo);
            }
            // 总账日期
            if (nullFields.contains("ledgerDate")) {
                update.setNull(qdo.ledgerDate);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 更新收款表银行账号、总账日期
     *
     * @return
     */
    public long updateRecvAccount(Long invId, String accountNo, String ledgerDate) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.accountNo, accountNo)
                .set(qdo.ledgerDate, LocalDate.parse(ledgerDate, DateTimeFormatter.ofPattern("yyyy-MM-dd")))
                .where(qdo.sourceId.in(invId).and(qdo.recvClass.eq("1")));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

