package com.elitesland.tw.tw5.server.prd.system.service;

import com.el.coordinator.boot.fsm.model.vo.FileObjRespVO;
import com.el.coordinator.boot.fsm.service.FileService;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdFsmFileRefPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdFsmFileRefQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdFsmFileRefService;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdFsmFileRefVO;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdFsmFileRefConvert;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdFsmFileRefDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdFsmFileRefRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.util.*;

/**
 * 附件业务关系表
 *
 * @author duwh
 * @date 2022/11/23
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdFsmFileRefServiceImpl implements PrdFsmFileRefService {

    private final PrdFsmFileRefRepo repo;
    private final FileUtil fileUtil;
    private final FileService fileService;
    /**
     * 一级评论标识
     */
    private final static Long REPLY_ID_DEFAULT = 0L;

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdFsmFileRefVO insert(PrdFsmFileRefPayload payload) {
        // 校验
        check(payload);
        if (StringUtils.hasText(payload.getFileCode())) {
            PrdFsmFileRefDO entityDo = PrdFsmFileRefConvert.INSTANCE.toDo(payload);
            repo.save(entityDo);
            return PrdFsmFileRefConvert.INSTANCE.toVo(entityDo);
        }
        return PrdFsmFileRefConvert.INSTANCE.payloadToVo(payload);
    }

    /**
     * 检查
     *
     * @param payload 有效载荷
     */
    private void check(PrdFsmFileRefPayload payload) {
        if (!StringUtils.hasText(payload.getObjType())) {
            throw TwException.error("", "objType不能为空");
        }
        if (null == payload.getObjId()) {
            throw TwException.error("", "objId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdFsmFileRefVO update(PrdFsmFileRefPayload payload) {
        PrdFsmFileRefDO entity = repo.findById(payload.getId()).orElseGet(PrdFsmFileRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PrdFsmFileRefDO entityDo = PrdFsmFileRefConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        final PrdFsmFileRefDO save = repo.save(entity);

        return PrdFsmFileRefConvert.INSTANCE.toVo(save);
    }

    @Override
    public PrdFsmFileRefVO queryByKey(Long key) {
        PrdFsmFileRefDO entity = repo.findById(key).orElseGet(PrdFsmFileRefDO::new);
        Assert.notNull(entity.getId(), "不存在");
        final PrdFsmFileRefVO crmCustomerOperationVO = PrdFsmFileRefConvert.INSTANCE.toVo(entity);
        crmCustomerOperationVO.setFileData(fileUtil.getFileDatas(crmCustomerOperationVO.getFileCode()));
        return crmCustomerOperationVO;
    }

    @Override
    public List<PrdFsmFileRefVO> queryList(PrdFsmFileRefQuery query) {
        // 校验参数必传
        checkQuery(query);
        final List<PrdFsmFileRefDO> listDO = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
        final List<PrdFsmFileRefVO> list = PrdFsmFileRefConvert.INSTANCE.toVoList(listDO);
        if (!CollectionUtils.isEmpty(list)) {
            final Map<String, List<FileObjRespVO<T>>> fileDataMap = fileUtil.getFileDataMap(list);
            list.forEach(prdFsmFileRefVO -> {
                prdFsmFileRefVO.setFileData(fileDataMap.get(prdFsmFileRefVO.getFileCode()));
            });
        }
        return list;
    }

    @Override
    public long count(PrdFsmFileRefQuery query) {
        // 校验参数必传
        checkQuery(query);
        return repo.count((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder));
    }

    @Override
    public long count(String objTypeEnum, Long objId) {
        PrdFsmFileRefQuery query = new PrdFsmFileRefQuery();
        query.setObjType(objTypeEnum);
        query.setObjId(objId);
        return count(query);
    }

    @Override
    public PagingVO<PrdFsmFileRefVO> paging(PrdFsmFileRefQuery query) {
        // 校验参数必传
        checkQuery(query);
        Page<PrdFsmFileRefDO> page = repo.findAll((root, criteriaQuery, criteriaBuilder) -> QueryHelp.getPredicate(root, query, criteriaBuilder), query.getPageRequest());
        final Page<PrdFsmFileRefVO> fsmFileRefVOPage = page.map(PrdFsmFileRefConvert.INSTANCE::toVo);
        final List<PrdFsmFileRefVO> list = fsmFileRefVOPage.getContent();
        if (!CollectionUtils.isEmpty(list)) {
            final Map<String, List<FileObjRespVO<T>>> fileDataMap = fileUtil.getFileDataMap(list);
            list.forEach(prdFsmFileRefVO -> {
                prdFsmFileRefVO.setFileData(fileDataMap.get(prdFsmFileRefVO.getFileCode()));
            });
        }
        return PageUtil.toPageVo(fsmFileRefVOPage);
    }

    /**
     * 检查查询
     *
     * @param query 查询
     */
    private void checkQuery(PrdFsmFileRefQuery query) {
        if (!StringUtils.hasText(query.getObjType())) {
            throw TwException.error("", "objType不能为空");
        }
        if (null == query.getObjId()) {
            throw TwException.error("", "objId不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            List<String> fileCodes = new ArrayList<>();
            keys.stream().forEach(id -> {
                Optional<PrdFsmFileRefDO> optional = repo.findById(id);
                if (!optional.isEmpty()) {
                    PrdFsmFileRefDO entity = optional.get();
                    fileCodes.add(entity.getFileCode());
                    repo.deleteSoft(Collections.singletonList(id));
                }
            });
            // 附件服务
            fileService.delete(fileCodes);
        }
    }

    @Override
    public void deleteSoftByFileCode(String fileCode) {
        if (StringUtils.hasText(fileCode)) {
            List<String> fileCodes = new ArrayList<>();
            Optional<PrdFsmFileRefDO> optional = repo.findByFileCode(fileCode);
            if (!optional.isEmpty()) {
                PrdFsmFileRefDO entity = optional.get();
                fileCodes.add(entity.getFileCode());
                repo.deleteSoft(Collections.singletonList(entity.getId()));
            }
            // 附件服务
            fileService.delete(fileCodes);
        }
    }

    @Override
    public void deleteSoftByObjIdAndObjType(Long objId, String objType){
        var list = repo.findByObjIdAndObjType(objId, objType);
        list.forEach(prdFsmFileRefDO -> prdFsmFileRefDO.setDeleteFlag(1));
        repo.saveAll(list);
    }

}
