package com.elitesland.tw.tw5.server.prd.inv.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.inv.payload.InvInvoiceVerDetailPayload;
import com.elitesland.tw.tw5.api.prd.inv.query.InvInvoiceVerDetailQuery;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvInvoiceVerDetailVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvInvoiceVerDetailDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvInvoiceDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvInvoiceVerDetailDO;
import com.elitesland.tw.tw5.server.prd.inv.repo.InvInvoiceVerDetailRepo;
import com.elitesland.tw.tw5.server.prd.product.entity.QPrdProductDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseAgreementDetailsDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseContractDetailsDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 发票核销明细表
 *
 * @author zoey
 * @date 2023-12-05
 */
@Repository
@RequiredArgsConstructor
public class InvInvoiceVerDetailDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final InvInvoiceVerDetailRepo repo;
    private final QInvInvoiceVerDetailDO qdo = QInvInvoiceVerDetailDO.invInvoiceVerDetailDO;
    private final QInvInvoiceDO qInvInvoiceDO = QInvInvoiceDO.invInvoiceDO;
    private final QPrdProductDO qPrdProductDO = QPrdProductDO.prdProductDO;
    private final QPurchaseContractDetailsDO qPurchaseContractDetailsDO = QPurchaseContractDetailsDO.purchaseContractDetailsDO;
    private final QPurchaseAgreementDetailsDO qPurchaseAgreementDetailsDO = QPurchaseAgreementDetailsDO.purchaseAgreementDetailsDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<InvInvoiceVerDetailVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(InvInvoiceVerDetailVO.class,
                        qdo.id,
                        //qdo.remark,
                        //qdo.createUserId,
                        //qdo.creator,
                        //qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 租户ID
                        qdo.tenantId,
                        // 付款申请单ID
                        qdo.paymentApplyId,
                        // 发票号
                        qdo.invoiceNo,
                        // 本次核销金额
                        qdo.theAmt,
                        // 发票金额
                        qdo.invoiceAmt,
                        // 已核销金额
                        qInvInvoiceDO.writtenOffAmt,
                        // 税率
                        qdo.rate,
                        // 发票类型
                        qdo.invoiceType,
                        // 搜索类型
                        qdo.searchType,
                        // 发票查验状态
                        qdo.inspection,
                        // 发票日期
                        qdo.invoiceDate,
                        // 发票代码
                        qdo.invoiceCode,
                        // 本人标志
                        qdo.selfFlag,
                        // 关联采购明细ID
                        qdo.purConOrAgreementDetailId,
                        // 附件
                        qInvInvoiceDO.imgContent,
                        // 发票凭证字段
                        qInvInvoiceDO.invoiceVoucher,
                        qInvInvoiceDO.jdeCompany,
                        qInvInvoiceDO.jdeDocumentNo,
                        qInvInvoiceDO.jdeDocumentType,
                        qInvInvoiceDO.jdePaymentItem,
                        qInvInvoiceDO.jdeInvoiceFailReason

                )).from(qdo)
                .leftJoin(qInvInvoiceDO).on(qdo.invoiceNo.eq(qInvInvoiceDO.invoiceNo)
                        .and(qdo.invoiceCode.eq(qInvInvoiceDO.invoiceCode)
                                .and(qInvInvoiceDO.deleteFlag.eq(0)
                                        .and(qInvInvoiceDO.isDel.eq(0)))));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<InvInvoiceVerDetailVO> getJpaQueryWhere(InvInvoiceVerDetailQuery query) {
        JPAQuery<InvInvoiceVerDetailVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(InvInvoiceVerDetailQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(InvInvoiceVerDetailQuery query) {
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 付款申请单ID 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyId())) {
            list.add(qdo.paymentApplyId.eq(query.getPaymentApplyId()));
        }
        /** 发票号 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceNo())) {
            list.add(qdo.invoiceNo.eq(query.getInvoiceNo()));
        }
        /** 本次核销金额 精确 */
        if (!ObjectUtils.isEmpty(query.getTheAmt())) {
            list.add(qdo.theAmt.eq(query.getTheAmt()));
        }
        /** 发票金额 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceAmt())) {
            list.add(qdo.invoiceAmt.eq(query.getInvoiceAmt()));
        }
        /** 税率 精确 */
        if (!ObjectUtils.isEmpty(query.getRate())) {
            list.add(qdo.rate.eq(query.getRate()));
        }
        /** 发票类型 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceType())) {
            list.add(qdo.invoiceType.eq(query.getInvoiceType()));
        }
        /** 搜索类型 精确 */
        if (!ObjectUtils.isEmpty(query.getSearchType())) {
            list.add(qdo.searchType.eq(query.getSearchType()));
        }
        /** 发票查验状态 精确 */
        if (!ObjectUtils.isEmpty(query.getInspection())) {
            list.add(qdo.inspection.eq(query.getInspection()));
        }
        /** 发票日期 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceDate())) {
            list.add(qdo.invoiceDate.eq(query.getInvoiceDate()));
        }
        /** 发票代码 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoiceCode())) {
            list.add(qdo.invoiceCode.eq(query.getInvoiceCode()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public InvInvoiceVerDetailVO queryByKey(Long id) {
        JPAQuery<InvInvoiceVerDetailVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<InvInvoiceVerDetailVO> queryListDynamic(InvInvoiceVerDetailQuery query) {
        JPAQuery<InvInvoiceVerDetailVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<InvInvoiceVerDetailVO> queryPaging(InvInvoiceVerDetailQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<InvInvoiceVerDetailVO> jpaQuery = getJpaQueryWhere(query);
        List<InvInvoiceVerDetailVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<InvInvoiceVerDetailVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public InvInvoiceVerDetailDO save(InvInvoiceVerDetailDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<InvInvoiceVerDetailDO> saveAll(List<InvInvoiceVerDetailDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(InvInvoiceVerDetailPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 付款申请单ID
        if (payload.getPaymentApplyId() != null) {
            update.set(qdo.paymentApplyId, payload.getPaymentApplyId());
        }
        // 发票号
        if (payload.getInvoiceNo() != null) {
            update.set(qdo.invoiceNo, payload.getInvoiceNo());
        }
        // 本次核销金额
        if (payload.getTheAmt() != null) {
            update.set(qdo.theAmt, payload.getTheAmt());
        }
        // 发票金额
        if (payload.getInvoiceAmt() != null) {
            update.set(qdo.invoiceAmt, payload.getInvoiceAmt());
        }
        // 税率
        if (payload.getRate() != null) {
            update.set(qdo.rate, payload.getRate());
        }
        // 发票类型
        if (payload.getInvoiceType() != null) {
            update.set(qdo.invoiceType, payload.getInvoiceType());
        }
        // 搜索类型
        if (payload.getSearchType() != null) {
            update.set(qdo.searchType, payload.getSearchType());
        }
        // 发票查验状态
        if (payload.getInspection() != null) {
            update.set(qdo.inspection, payload.getInspection());
        }
        // 发票日期
        if (payload.getInvoiceDate() != null) {
            update.set(qdo.invoiceDate, payload.getInvoiceDate());
        }
        // 发票代码
        if (payload.getInvoiceCode() != null) {
            update.set(qdo.invoiceCode, payload.getInvoiceCode());
        }
        // 发票凭证字段
        if (payload.getInvoiceVoucher() != null) {
            update.set(qdo.invoiceVoucher, payload.getInvoiceVoucher());
        }
        // 发票凭证字段
        if (payload.getTaxAmount() != null) {
            update.set(qdo.taxAmount, payload.getTaxAmount());
        }

        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 租户ID
            if (nullFields.contains("tenantId")) {
                update.setNull(qdo.tenantId);
            }
            // 付款申请单ID
            if (nullFields.contains("paymentApplyId")) {
                update.setNull(qdo.paymentApplyId);
            }
            // 发票号
            if (nullFields.contains("invoiceNo")) {
                update.setNull(qdo.invoiceNo);
            }
            // 本次核销金额
            if (nullFields.contains("theAmt")) {
                update.setNull(qdo.theAmt);
            }
            // 发票金额
            if (nullFields.contains("invoiceAmt")) {
                update.setNull(qdo.invoiceAmt);
            }
            // 税率
            if (nullFields.contains("rate")) {
                update.setNull(qdo.rate);
            }
            // 发票类型
            if (nullFields.contains("invoiceType")) {
                update.setNull(qdo.invoiceType);
            }
            // 搜索类型
            if (nullFields.contains("searchType")) {
                update.setNull(qdo.searchType);
            }
            // 发票查验状态
            if (nullFields.contains("inspection")) {
                update.setNull(qdo.inspection);
            }
            // 发票日期
            if (nullFields.contains("invoiceDate")) {
                update.setNull(qdo.invoiceDate);
            }
            // 发票代码
            if (nullFields.contains("invoiceCode")) {
                update.setNull(qdo.invoiceCode);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 根据付款申请单id删除发票核销明细
     *
     * @param paymentApplyId 付款申请单id
     */
    public void deleteByPaymentApplyId(Long paymentApplyId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.paymentApplyId.eq(paymentApplyId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }

    /**
     * 根据付款申请单id查询发票核销明细列表
     *
     * @param docType        类型
     * @param paymentApplyId 付款申请单id
     */
    public List<InvInvoiceVerDetailVO> queryListByPaymentApplyId(String docType, Long paymentApplyId) {
        JPAQuery<InvInvoiceVerDetailVO> jpaQueryByDocType = getJpaQueryByDocType(docType);
        jpaQueryByDocType.where(qdo.paymentApplyId.eq(paymentApplyId));
        return jpaQueryByDocType.fetch();
    }

    /**
     * 根据付款申请单id查询发票核销明细列表
     *
     * @param docType        类型
     * @param paymentApplyId 付款申请单id
     */
    public List<InvInvoiceVerDetailVO> queryListByJdePaymentApplyId(String docType, Long paymentApplyId) {
        JPAQuery<InvInvoiceVerDetailVO> jpaQueryByDocType = getJpaQueryByDocType(docType);
        jpaQueryByDocType.where(qdo.paymentApplyId.eq(paymentApplyId));
        jpaQueryByDocType.where(qInvInvoiceDO.invoiceVoucher.isNull());
        jpaQueryByDocType.where(qdo.deleteFlag.eq(0));

        return jpaQueryByDocType.fetch();
    }

    /**
     * 根据关联明细获取jpaQuery
     *
     * @param docType 关联单据类型
     * @return JPAQuery<InvInvoiceVerDetailVO>
     */
    private JPAQuery<InvInvoiceVerDetailVO> getJpaQueryByDocType(String docType) {
        JPAQuery<InvInvoiceVerDetailVO> jpaQuery = jpaQueryFactory.select(Projections.bean(InvInvoiceVerDetailVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        qdo.modifyUserId,
                        qdo.updater,
                        qdo.modifyTime,
                        // 租户ID
                        qdo.tenantId,
                        // 付款申请单ID
                        qdo.paymentApplyId,
                        // 发票号
                        qdo.invoiceNo,
                        // 本次核销金额
                        qdo.theAmt,
                        // 发票金额
                        qdo.invoiceAmt,
                        // 已核销金额
                        qInvInvoiceDO.writtenOffAmt,
                        // 税率
                        qdo.rate,
                        // 发票类型
                        qdo.invoiceType,
                        // 搜索类型
                        qdo.searchType,
                        // 发票查验状态
                        qdo.inspection,
                        // 发票日期
                        qdo.invoiceDate,
                        // 发票代码
                        qdo.invoiceCode,
                        // 本人标志
                        qdo.selfFlag,
                        qdo.taxAmount,
                        // 关联采购明细ID
                        qdo.purConOrAgreementDetailId,
                        // 关联产品ID
                        qPrdProductDO.id.as("relatedProductId"),
                        // 关联产品Name
                        qPrdProductDO.prodName.as("relatedProductName"),
                        // 发票抵扣税额
                        qInvInvoiceDO.deductTax,
                        // 附件
                        qInvInvoiceDO.imgContent,
                        // 发票凭证字段
                        qInvInvoiceDO.invoiceVoucher,
                        qInvInvoiceDO.jdeCompany,
                        qInvInvoiceDO.jdeDocumentNo,
                        qInvInvoiceDO.jdeDocumentType,
                        qInvInvoiceDO.jdePaymentItem,
                        qInvInvoiceDO.jdeInvoiceFailReason

                )).from(qdo)
                .leftJoin(qInvInvoiceDO).on(qdo.invoiceNo.eq(qInvInvoiceDO.invoiceNo)
                        .and(qdo.invoiceCode.coalesce("").eq(qInvInvoiceDO.invoiceCode.coalesce(""))
                                .and(qInvInvoiceDO.deleteFlag.eq(0)
                                        .and(qInvInvoiceDO.isDel.eq(0)))));
        // 关联合同
        if (PurchasePaymentEnum.PaymentDocType.CONTRACT.getCode().equals(docType)) {
            jpaQuery.leftJoin(qPurchaseContractDetailsDO).on(qdo.purConOrAgreementDetailId.eq(qPurchaseContractDetailsDO.id)
                            .and(qPurchaseContractDetailsDO.deleteFlag.eq(0)))
                    .leftJoin(qPrdProductDO).on(qPurchaseContractDetailsDO.relatedProductId.eq(qPrdProductDO.id)
                            .and(qPrdProductDO.deleteFlag.eq(0)));
            // 采购协议
        } else if (PurchasePaymentEnum.PaymentDocType.AGREEMENT.getCode().equals(docType)) {
            jpaQuery.leftJoin(qPurchaseAgreementDetailsDO).on(qdo.purConOrAgreementDetailId.eq(qPurchaseAgreementDetailsDO.id)
                            .and(qPurchaseAgreementDetailsDO.deleteFlag.eq(0)))
                    .leftJoin(qPrdProductDO).on(qPurchaseAgreementDetailsDO.relatedProductId.eq(qPrdProductDO.id)
                            .and(qPrdProductDO.deleteFlag.eq(0)));
        } else {
            return getJpaQuerySelect();
        }
        return jpaQuery;
    }
}

