package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsMaConfigPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsMaConfigQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsMaConfigVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsMaConfigDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsMaConfigDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsMaConfigRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目经理任务书配置
 *
 * @author carl
 * @date 2023-09-19
 */
@Repository
@RequiredArgsConstructor
public class PmsMaConfigDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsMaConfigRepo repo;
    private final QPmsMaConfigDO qdo = QPmsMaConfigDO.pmsMaConfigDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsMaConfigVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsMaConfigVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 排序
                qdo.sortIndex,
                // 项目阶段
                qdo.projectStage,
                // 关键任务
                qdo.keyTask
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsMaConfigVO> getJpaQueryWhere(PmsMaConfigQuery query) {
        JPAQuery<PmsMaConfigVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.asc("sortIndex"));
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsMaConfigQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsMaConfigQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 排序 精确 */
        if (!ObjectUtils.isEmpty(query.getSortIndex())) {
            list.add(qdo.sortIndex.eq(query.getSortIndex()));
        }
        /** 项目阶段 模糊 */
        if (!ObjectUtils.isEmpty(query.getProjectStage())) {
            list.add(qdo.projectStage.like(SqlUtil.toSqlLikeString(query.getProjectStage())));
        }
        /** 关键任务 模糊 */
        if (!ObjectUtils.isEmpty(query.getKeyTask())) {
            list.add(qdo.keyTask.like(SqlUtil.toSqlLikeString(query.getKeyTask())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsMaConfigVO queryByKey(Long id) {
        JPAQuery<PmsMaConfigVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsMaConfigVO> queryListDynamic(PmsMaConfigQuery query) {
        JPAQuery<PmsMaConfigVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsMaConfigVO> queryPaging(PmsMaConfigQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsMaConfigVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsMaConfigVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsMaConfigVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsMaConfigDO save(PmsMaConfigDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsMaConfigDO> saveAll(List<PmsMaConfigDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsMaConfigPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 排序
        if (payload.getSortIndex() != null) {
            update.set(qdo.sortIndex, payload.getSortIndex());
        }
        // 项目阶段
        if (payload.getProjectStage() != null) {
            update.set(qdo.projectStage, payload.getProjectStage());
        }
        // 关键任务
        if (payload.getKeyTask() != null) {
            update.set(qdo.keyTask, payload.getKeyTask());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 排序
            if (nullFields.contains("sortIndex")) {
                update.setNull(qdo.sortIndex);
            }
            // 项目阶段
            if (nullFields.contains("projectStage")) {
                update.setNull(qdo.projectStage);
            }
            // 关键任务
            if (nullFields.contains("keyTask")) {
                update.setNull(qdo.keyTask);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

