package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectEventLogPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectEventLogQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectEventLogVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectEventLogDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectEventLogDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectEventLogRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目事件日志功能管理
 *
 * @author carl
 * @date 2023-09-18
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectEventLogDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectEventLogRepo repo;
    private final QPmsProjectEventLogDO qdo = QPmsProjectEventLogDO.pmsProjectEventLogDO;
    private final QPmsProjectDO qdoProject = QPmsProjectDO.pmsProjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectEventLogVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectEventLogVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 项目主键
                qdo.projectId,
                //项目名称
                qdoProject.projName,
                //项目编号
                qdoProject.projNo,
                //交付bu
                qdoProject.deliBuId,
                //项目经理
                qdoProject.pmResId,
                // 事件内容及来源
                qdo.eventContentSource,
                // 可能产生的影响
                qdo.possibleImpacts,
                // 所属类型
                qdo.type,
                // 严重程度
                qdo.severity,
                // 发生几率
                qdo.probabilityOfOccurrence,
                // 风险程度
                qdo.riskLevel,
                // 应对策略
                qdo.copingStrategy,
                // 预计结束时间
                qdo.expectedEndDate,
                // 负责人
                qdo.headResId,
                qdo.finishTime,
                // 状态
                qdo.status
        )).from(qdo).leftJoin(qdoProject).on(qdo.projectId.eq(qdoProject.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectEventLogVO> getJpaQueryWhere(PmsProjectEventLogQuery query) {
        JPAQuery<PmsProjectEventLogVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectEventLogQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoProject).on(qdo.projectId.eq(qdoProject.id));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectEventLogQuery query) {
        List<Predicate> list = new ArrayList<>();
        list.add(qdoProject.deleteFlag.eq(0));
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 搜索 精确 */
        if (!ObjectUtils.isEmpty(query.getProjSearch())) {
            list.add(qdoProject.projName.like(SqlUtil.toSqlLikeString(query.getProjSearch())).or(qdoProject.projNo.like(SqlUtil.toSqlLikeString(query.getProjSearch()))));
        }
        /** 项目主键 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 事件内容及来源 精确 */
        if (!ObjectUtils.isEmpty(query.getEventContentSource())) {
            list.add(qdo.eventContentSource.eq(query.getEventContentSource()));
        }
        /** 可能产生的影响 模糊 */
        if (!ObjectUtils.isEmpty(query.getPossibleImpacts())) {
            list.add(qdo.possibleImpacts.like(SqlUtil.toSqlLikeString(query.getPossibleImpacts())));
        }
        /** 所属类型 精确 */
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        /** 严重程度 精确 */
        if (!ObjectUtils.isEmpty(query.getSeverity())) {
            list.add(qdo.severity.eq(query.getSeverity()));
        }
        /** 发生几率 精确 */
        if (!ObjectUtils.isEmpty(query.getProbabilityOfOccurrence())) {
            list.add(qdo.probabilityOfOccurrence.eq(query.getProbabilityOfOccurrence()));
        }
        /** 风险程度 精确 */
        if (!ObjectUtils.isEmpty(query.getRiskLevel())) {
            list.add(qdo.riskLevel.eq(query.getRiskLevel()));
        }
        /** 应对策略 模糊 */
        if (!ObjectUtils.isEmpty(query.getCopingStrategy())) {
            list.add(qdo.copingStrategy.like(SqlUtil.toSqlLikeString(query.getCopingStrategy())));
        }
        /** 预计结束时间 精确 */
        if (!ObjectUtils.isEmpty(query.getExpectedEndDate())) {
            list.add(qdo.expectedEndDate.eq(query.getExpectedEndDate()));
        }
        /** 负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getHeadResId())) {
            list.add(qdo.headResId.eq(query.getHeadResId()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        /** 项目经理 */
        if (!ObjectUtils.isEmpty(query.getPmResId())) {
            list.add(qdoProject.pmResId.eq(query.getPmResId()));
        }
        /** 交付bu 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliBuId())) {
            list.add(qdoProject.deliBuId.eq(query.getDeliBuId()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectEventLogVO queryByKey(Long id) {
        JPAQuery<PmsProjectEventLogVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<PmsProjectEventLogVO> queryByKeys(List<Long> ids) {
        JPAQuery<PmsProjectEventLogVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectEventLogVO> queryListDynamic(PmsProjectEventLogQuery query) {
        JPAQuery<PmsProjectEventLogVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectEventLogVO> queryPaging(PmsProjectEventLogQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectEventLogVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectEventLogVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectEventLogVO>builder().records(result).total(total).build();
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectEventLogVO> getJpaQuerySelect0() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectEventLogVO.class,
                qdo.id,
                // 项目主键
                qdo.projectId,
                //项目名称
                qdoProject.projName,
                // 事件内容及来源
                qdo.eventContentSource,
                // 可能产生的影响
                qdo.possibleImpacts,
                // 所属类型
                qdo.type,
                // 严重程度
                qdo.severity,
                // 发生几率
                qdo.probabilityOfOccurrence,
                // 风险程度
                qdo.riskLevel,
                // 应对策略
                qdo.copingStrategy,
                // 预计结束时间
                qdo.expectedEndDate,
                // 负责人
                qdo.headResId,
                // 状态
                qdo.status,
                // 项目经理
                qdoProject.pmResId,
                // 交付bu
                qdoProject.deliBuId
        )).from(qdo).leftJoin(qdoProject).on(qdo.projectId.eq(qdoProject.id));
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where0(PmsProjectEventLogQuery query) {
        List<Predicate> list = new ArrayList<>();
        list.add(qdoProject.deleteFlag.eq(0));
        list.add(qdo.expectedEndDate.isNotNull());
        BooleanExpression jpaQueryOr1 = qdo.headResId.eq(query.getLoginUserId())
                .or(qdoProject.deliUserId.eq(query.getLoginUserId()));
        BooleanExpression jpaQueryOr2 = qdo.headResId.eq(query.getLoginUserId())
                .or(qdoProject.deliUserId.eq(query.getLoginUserId()))
                .or(qdoProject.pmResId.eq(query.getLoginUserId()));
        if (!ObjectUtils.isEmpty(query.getHeadOrgIds())) {
            jpaQueryOr1 = jpaQueryOr1.or(qdoProject.deliBuId.in(query.getHeadOrgIds()));
            jpaQueryOr2 = jpaQueryOr2.or(qdoProject.deliBuId.in(query.getHeadOrgIds()));
        }
        // 状态为“已识别”,推送项目的交付负责人、交付BU负责人、项目事件日志负责人
        // 状态为“已延期”,项目的项目经理、交付负责人、交付BU负责人、项目事件日志负责人
        list.add((qdo.status.eq("IDENTIFIED").and(jpaQueryOr1)).or(qdo.status.eq("EXTENDED").and(jpaQueryOr2)));
        if (!ObjectUtils.isEmpty(query.getType())) {
            list.add(qdo.type.eq(query.getType()));
        }
        if (!ObjectUtils.isEmpty(query.getSeverity())) {
            list.add(qdo.severity.eq(query.getSeverity()));
        }
        if (!ObjectUtils.isEmpty(query.getStatus())) {
            list.add(qdo.status.eq(query.getStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getProjectStatus())) {
            list.add(qdoProject.projStatus.eq(query.getProjectStatus()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectEventLogVO> queryPushPaging(PmsProjectEventLogQuery query) {

        JPAQuery<Long> jpaQuery0 = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoProject).on(qdo.projectId.eq(qdoProject.id));
        jpaQuery0.where(where0(query));

        long total = jpaQuery0.fetchOne();

        if (total == 0) {
            return PagingVO.empty();
        }

        JPAQuery<PmsProjectEventLogVO> jpaQuery = getJpaQuerySelect0();
        // 条件封装
        jpaQuery.where(where0(query));
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));

        List<PmsProjectEventLogVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectEventLogVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectEventLogDO save(PmsProjectEventLogDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectEventLogDO> saveAll(List<PmsProjectEventLogDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectEventLogPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 项目主键
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // 事件内容及来源
        if (payload.getEventContentSource() != null) {
            update.set(qdo.eventContentSource, payload.getEventContentSource());
        }
        // 可能产生的影响
        if (payload.getPossibleImpacts() != null) {
            update.set(qdo.possibleImpacts, payload.getPossibleImpacts());
        }
        // 所属类型
        if (payload.getType() != null) {
            update.set(qdo.type, payload.getType());
        }
        // 严重程度
        if (payload.getSeverity() != null) {
            update.set(qdo.severity, payload.getSeverity());
        }
        // 发生几率
        if (payload.getProbabilityOfOccurrence() != null) {
            update.set(qdo.probabilityOfOccurrence, payload.getProbabilityOfOccurrence());
        }
        // 风险程度
        if (payload.getRiskLevel() != null) {
            update.set(qdo.riskLevel, payload.getRiskLevel());
        }
        // 应对策略
        if (payload.getCopingStrategy() != null) {
            update.set(qdo.copingStrategy, payload.getCopingStrategy());
        }
        // 预计结束时间
        if (payload.getExpectedEndDate() != null) {
            update.set(qdo.expectedEndDate, payload.getExpectedEndDate());
        }
        // 负责人
        if (payload.getHeadResId() != null) {
            update.set(qdo.headResId, payload.getHeadResId());
        }
        // 状态
        if (payload.getStatus() != null) {
            update.set(qdo.status, payload.getStatus());
        }
        // 完成时间
        if (payload.getFinishTime() != null) {
            update.set(qdo.finishTime, payload.getFinishTime());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 项目主键
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // 事件内容及来源
            if (nullFields.contains("eventContentSource")) {
                update.setNull(qdo.eventContentSource);
            }
            // 可能产生的影响
            if (nullFields.contains("possibleImpacts")) {
                update.setNull(qdo.possibleImpacts);
            }
            // 所属类型
            if (nullFields.contains("type")) {
                update.setNull(qdo.type);
            }
            // 严重程度
            if (nullFields.contains("severity")) {
                update.setNull(qdo.severity);
            }
            // 发生几率
            if (nullFields.contains("probabilityOfOccurrence")) {
                update.setNull(qdo.probabilityOfOccurrence);
            }
            // 风险程度
            if (nullFields.contains("riskLevel")) {
                update.setNull(qdo.riskLevel);
            }
            // 应对策略
            if (nullFields.contains("copingStrategy")) {
                update.setNull(qdo.copingStrategy);
            }
            // 预计结束时间
            if (nullFields.contains("expectedEndDate")) {
                update.setNull(qdo.expectedEndDate);
            }
            // 负责人
            if (nullFields.contains("headResId")) {
                update.setNull(qdo.headResId);
            }
            // 状态
            if (nullFields.contains("status")) {
                update.setNull(qdo.status);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public long updateStatus(List<Long> ids, String status, String riskLevel) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.status, status)
                .set(qdo.riskLevel, riskLevel)
                .where(qdo.id.in(ids));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

