package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalAccountPayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalAccountQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalAccountVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalAccountDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalAccountDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalAccountRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;

/**
 * 核算-账号功能管理
 *
 * @author carl
 * @date 2023-11-13
 */
@Repository
@RequiredArgsConstructor
public class CalAccountDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalAccountRepo repo;
    private final QCalAccountDO qdo = QCalAccountDO.calAccountDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalAccountVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalAccountVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 账号年度
                qdo.ledgerYear,
                // 账户名称
                qdo.ledgerName,
                // 账户状态(0有效1无效)
                qdo.ledgerStatus,
                // （账户类型）核算主体类型
                qdo.auType,
                // 核算主体id
                qdo.auId,
                // 账号编号
                qdo.ledgerNo,
                // 1：平台账户，0：否
                qdo.ledgerType,
                // 币种
                qdo.currCode,
                // 总数量
                qdo.totalQty,
                // 冻结数量
                qdo.frozenQty,
                // 可用数量
                qdo.avalQty,
                // 总金额
                qdo.totalAmt,
                // 冻结金额
                qdo.frozenAmt,
                // 可用金额
                qdo.avalAmt
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalAccountVO> getJpaQueryWhere(CalAccountQuery query) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalAccountQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalAccountQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 账号年度 精确 */
        if (!ObjectUtils.isEmpty(query.getLedgerYear())) {
            if (query.getLedgerYearFlag() != null && query.getLedgerYearFlag().equals(0)) {
                list.add(qdo.ledgerYear.ne(query.getLedgerYear()));
            } else {
                list.add(qdo.ledgerYear.eq(query.getLedgerYear()));
            }
        }
        /** 账户名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getLedgerName())) {
            list.add(qdo.ledgerName.like(SqlUtil.toSqlLikeString(query.getLedgerName())));
        }
        /** 账户状态(0有效1无效) 精确 */
        if (!ObjectUtils.isEmpty(query.getLedgerStatus())) {
            list.add(qdo.ledgerStatus.eq(query.getLedgerStatus()));
        }
        /** （账户类型）核算主体类型 精确 */
        if (!ObjectUtils.isEmpty(query.getAuType())) {
            list.add(qdo.auType.eq(query.getAuType()));
        }
        /** （账户类型）核算主体类型 精确 */
        if (!ObjectUtils.isEmpty(query.getAuTypes())) {
            list.add(qdo.auType.in(query.getAuTypes()));
        }
        /** 核算主体id 精确 */
        if (!ObjectUtils.isEmpty(query.getAuId())) {
            list.add(qdo.auId.eq(query.getAuId()));
        }
        /** 账号编号 精确 */
        if (!ObjectUtils.isEmpty(query.getLedgerNo())) {
            list.add(qdo.ledgerNo.eq(query.getLedgerNo()));
        }
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getLedgerType())) {
            list.add(qdo.ledgerType.eq(query.getLedgerType()));
        }
        /** 币种 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrCode())) {
            list.add(qdo.currCode.eq(query.getCurrCode()));
        }
        /** 总数量 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalQty())) {
            list.add(qdo.totalQty.eq(query.getTotalQty()));
        }

        /** 冻结数量 精确 */
        if (!ObjectUtils.isEmpty(query.getFrozenQty())) {
            list.add(qdo.frozenQty.eq(query.getFrozenQty()));
        }
        /** 可用数量 精确 */
        if (!ObjectUtils.isEmpty(query.getAvalQty())) {
            list.add(qdo.avalQty.eq(query.getAvalQty()));
        }
        /** 总金额 精确 */
        if (!ObjectUtils.isEmpty(query.getTotalAmt())) {
            list.add(qdo.totalAmt.eq(query.getTotalAmt()));
        }
        if (StringUtils.hasText(query.getOperType())) {
            if ("1".equals(query.getOperType())) {
                list.add(qdo.totalAmt.gt(0));
            }
            if ("-1".equals(query.getOperType())) {
                list.add(qdo.totalAmt.lt(0));
            }

        }
        /** 冻结金额 精确 */
        if (!ObjectUtils.isEmpty(query.getFrozenAmt())) {
            list.add(qdo.frozenAmt.eq(query.getFrozenAmt()));
        }
        /** 可用金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAvalAmt())) {
            list.add(qdo.avalAmt.eq(query.getAvalAmt()));
        }
        /** 账户编号 精确 */
        if (!ObjectUtils.isEmpty(query.getLedgerNos())) {
            list.add(qdo.ledgerNo.in(query.getLedgerNos()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalAccountVO queryByKey(Long id) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param keys 主键
     * @return 结果
     */
    public List<CalAccountVO> queryByKeys(List<Long> keys) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(keys));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalAccountVO> queryListDynamic(CalAccountQuery query) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalAccountVO> queryPaging(CalAccountQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalAccountVO> jpaQuery = getJpaQueryWhere(query);
        List<CalAccountVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalAccountVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalAccountDO save(CalAccountDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalAccountDO> saveAll(List<CalAccountDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalAccountPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 账号年度
        if (payload.getLedgerYear() != null) {
            update.set(qdo.ledgerYear, payload.getLedgerYear());
        }
        // 账户名称
        if (payload.getLedgerName() != null) {
            update.set(qdo.ledgerName, payload.getLedgerName());
        }
        // 账户状态(0有效1无效)
        if (payload.getLedgerStatus() != null) {
            update.set(qdo.ledgerStatus, payload.getLedgerStatus());
        }
        // （账户类型）核算主体类型
        if (payload.getAuType() != null) {
            update.set(qdo.auType, payload.getAuType());
        }
        // 核算主体id
        if (payload.getAuId() != null) {
            update.set(qdo.auId, payload.getAuId());
        }
        // 账号编号
        if (payload.getLedgerNo() != null) {
            update.set(qdo.ledgerNo, payload.getLedgerNo());
        }
        if (payload.getLedgerType() != null) {
            update.set(qdo.ledgerType, payload.getLedgerType());
        }
        // 币种
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 总数量
        if (payload.getTotalQty() != null) {
            update.set(qdo.totalQty, payload.getTotalQty());
        }
        // 冻结数量
        if (payload.getFrozenQty() != null) {
            update.set(qdo.frozenQty, payload.getFrozenQty());
        }
        // 可用数量
        if (payload.getAvalQty() != null) {
            update.set(qdo.avalQty, payload.getAvalQty());
        }
        // 总金额
        if (payload.getTotalAmt() != null) {
            update.set(qdo.totalAmt, payload.getTotalAmt());
        }
        // 冻结金额
        if (payload.getFrozenAmt() != null) {
            update.set(qdo.frozenAmt, payload.getFrozenAmt());
        }
        // 可用金额
        if (payload.getAvalAmt() != null) {
            update.set(qdo.avalAmt, payload.getAvalAmt());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 账号年度
            if (nullFields.contains("ledgerYear")) {
                update.setNull(qdo.ledgerYear);
            }
            // 账户名称
            if (nullFields.contains("ledgerName")) {
                update.setNull(qdo.ledgerName);
            }
            // 账户状态(0有效1无效)
            if (nullFields.contains("ledgerStatus")) {
                update.setNull(qdo.ledgerStatus);
            }
            // （账户类型）核算主体类型
            if (nullFields.contains("auType")) {
                update.setNull(qdo.auType);
            }
            // 核算主体id
            if (nullFields.contains("auId")) {
                update.setNull(qdo.auId);
            }
            // 账号编号
            if (nullFields.contains("ledgerNo")) {
                update.setNull(qdo.ledgerNo);
            }
            if (nullFields.contains("ledgerType")) {
                update.setNull(qdo.ledgerType);
            }
            // 币种
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 总数量
            if (nullFields.contains("totalQty")) {
                update.setNull(qdo.totalQty);
            }
            // 冻结数量
            if (nullFields.contains("frozenQty")) {
                update.setNull(qdo.frozenQty);
            }
            // 可用数量
            if (nullFields.contains("avalQty")) {
                update.setNull(qdo.avalQty);
            }
            // 总金额
            if (nullFields.contains("totalAmt")) {
                update.setNull(qdo.totalAmt);
            }
            // 冻结金额
            if (nullFields.contains("frozenAmt")) {
                update.setNull(qdo.frozenAmt);
            }
            // 可用金额
            if (nullFields.contains("avalAmt")) {
                update.setNull(qdo.avalAmt);
            }
        }
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public List<Long> queryListIds(CalAccountQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdo.id).from(qdo);
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);

        return jpaQuery.fetch();
    }

    /**
     * 根据主题类型和主题id查询
     *
     * @param auType 主题类型
     * @param auId   主题id
     * @return 结果
     */
    public CalAccountVO queryByAuTypeAndAuId(String auType, Long auId) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.auType.eq(auType));
        jpaQuery.where(qdo.auId.eq(auId));
        jpaQuery.where(qdo.ledgerStatus.eq(1));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主题类型和主题id查询
     *
     * @param auType 主题类型
     * @param auIds  主题id
     * @return 结果
     */
    public List<CalAccountVO> queryByAuTypeAndAuIds(String auType, List<Long> auIds) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.auType.eq(auType));
        jpaQuery.where(qdo.auId.in(auIds));
        jpaQuery.where(qdo.ledgerStatus.eq(1));
        return jpaQuery.fetch();
    }

    /**
     * 根据主题类型和主题id查询
     *
     * @param auTypes 主题类型
     * @param auIds   主题id
     * @return 结果
     */
    public List<CalAccountVO> queryByAuTypesAndAuIds(List<String> auTypes, List<Long> auIds) {
        JPAQuery<CalAccountVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.auType.in(auTypes));
        jpaQuery.where(qdo.auId.in(auIds));
        jpaQuery.where(qdo.ledgerStatus.eq(1));
        return jpaQuery.fetch();
    }

    /**
     * 年度数据清0
     *
     * @param keys 主集合
     * @param year 新的年度
     * @return
     */
    public long initAccount(List<Long> keys, int year) {
        BigDecimal zero = BigDecimal.ZERO;
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.in(keys));

        update.set(qdo.ledgerYear, year);
        // 总数量
        update.set(qdo.totalQty, zero);
        // 冻结数量
        update.set(qdo.frozenQty, zero);
        // 可用数量
        update.set(qdo.avalQty, zero);
        // 总金额
        update.set(qdo.totalAmt, zero);
        // 冻结金额
        update.set(qdo.frozenAmt, zero);
        // 可用金额
        update.set(qdo.avalAmt, zero);

        return update.execute();
    }
}

