package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalWideSettlePayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalWideSettleQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalWideSettleVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalWideSettleDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalWideSettleDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalWideSettleRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 泛用当量结算单管理
 *
 * @author carl
 * @date 2023-11-15
 */
@Repository
@RequiredArgsConstructor
public class CalWideSettleDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalWideSettleRepo repo;
    private final QCalWideSettleDO qdo = QCalWideSettleDO.calWideSettleDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalWideSettleVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalWideSettleVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 结算单号
                qdo.settleNo,
                // 结算状态
                qdo.settleStatus,
                // 审批状态
                qdo.apprStatus,
                // 流程实例id
                qdo.procInstId,
                //申请人id
                qdo.applyResId,
                // 结算类型（项目立项，项目结项。。。）
                qdo.settleType,
                // 结算日期
                qdo.settleDate,
                // 财务期间id
                qdo.finPeriodId,
                // 支出核算主题类型
                qdo.fromSourceType,
                // 支出核算主题id
                qdo.fromSourceId,
                qdo.fromSourceName,
                // 收入核算主题类型
                qdo.toSourceType,
                // 收入核算主题id
                qdo.toSourceId,
                qdo.toSourceName,
                // 申请结算当量数
                qdo.applySettleEqva,
                // 申请结算总额
                qdo.applySettleAmt,
                // 结算单价
                qdo.settlePrice,
                // 交易凭证文件code
                qdo.fileCodes
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalWideSettleVO> getJpaQueryWhere(CalWideSettleQuery query) {
        JPAQuery<CalWideSettleVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalWideSettleQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalWideSettleQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 结算单号 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleNo())) {
            list.add(qdo.settleNo.eq(query.getSettleNo()));
        }
        /** 结算状态 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleStatus())) {
            list.add(qdo.settleStatus.eq(query.getSettleStatus()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApprStatus())) {
            list.add(qdo.apprStatus.eq(query.getApprStatus()));
        }
        /** 流程实例id 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 结算类型（项目立项，项目结项。。。） 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleType())) {
            list.add(qdo.settleType.eq(query.getSettleType()));
        }
        /** 结算日期 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleStartDate())) {
            list.add(qdo.settleDate.goe(query.getSettleStartDate()));
        }
        /** 结算日期 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleEndDate())) {
            list.add(qdo.settleDate.loe(query.getSettleEndDate()));
        }
        /** 结算日期 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleDate())) {
            list.add(qdo.settleDate.eq(query.getSettleDate()));
        }
        /** 财务期间id 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriodId())) {
            list.add(qdo.finPeriodId.eq(query.getFinPeriodId()));
        }
        /** 支出核算主题类型 精确 */
        if (!ObjectUtils.isEmpty(query.getFromSourceType())) {
            list.add(qdo.fromSourceType.eq(query.getFromSourceType()));
        }
        /** 支出核算主题id 精确 */
        if (!ObjectUtils.isEmpty(query.getFromSourceId())) {
            list.add(qdo.fromSourceId.eq(query.getFromSourceId()));
        }
        /** 收入核算主题类型 精确 */
        if (!ObjectUtils.isEmpty(query.getToSourceType())) {
            list.add(qdo.toSourceType.eq(query.getToSourceType()));
        }
        /** 收入核算主题id 精确 */
        if (!ObjectUtils.isEmpty(query.getToSourceId())) {
            list.add(qdo.toSourceId.eq(query.getToSourceId()));
        }
        /** 申请结算当量数 精确 */
        if (!ObjectUtils.isEmpty(query.getApplySettleEqva())) {
            list.add(qdo.applySettleEqva.eq(query.getApplySettleEqva()));
        }
        /** 申请结算总额 精确 */
        if (!ObjectUtils.isEmpty(query.getApplySettleAmt())) {
            list.add(qdo.applySettleAmt.eq(query.getApplySettleAmt()));
        }
        /** 结算单价 精确 */
        if (!ObjectUtils.isEmpty(query.getSettlePrice())) {
            list.add(qdo.settlePrice.eq(query.getSettlePrice()));
        }
        /** 交易凭证文件code 精确 */
        if (!ObjectUtils.isEmpty(query.getFileCodes())) {
            list.add(qdo.fileCodes.eq(query.getFileCodes()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalWideSettleVO queryByKey(Long id) {
        JPAQuery<CalWideSettleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param ids 主键
     * @return 结果
     */
    public List<CalWideSettleVO> queryByKeys(List<Long> ids) {
        JPAQuery<CalWideSettleVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalWideSettleVO> queryListDynamic(CalWideSettleQuery query) {
        JPAQuery<CalWideSettleVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalWideSettleVO> queryPaging(CalWideSettleQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalWideSettleVO> jpaQuery = getJpaQueryWhere(query);
        List<CalWideSettleVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalWideSettleVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalWideSettleDO save(CalWideSettleDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalWideSettleDO> saveAll(List<CalWideSettleDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalWideSettlePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 结算单号
        if (payload.getSettleNo() != null) {
            update.set(qdo.settleNo, payload.getSettleNo());
        }
        // 结算状态
        if (payload.getSettleStatus() != null) {
            update.set(qdo.settleStatus, payload.getSettleStatus());
        }
        // 审批状态
        if (payload.getApprStatus() != null) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        // 流程实例id
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 结算类型（项目立项，项目结项。。。）
        if (payload.getSettleType() != null) {
            update.set(qdo.settleType, payload.getSettleType());
        }
        // 结算日期
        if (payload.getSettleDate() != null) {
            update.set(qdo.settleDate, payload.getSettleDate());
        }
        // 财务期间id
        if (payload.getFinPeriodId() != null) {
            update.set(qdo.finPeriodId, payload.getFinPeriodId());
        }
        // 支出核算主题类型
        if (payload.getFromSourceType() != null) {
            update.set(qdo.fromSourceType, payload.getFromSourceType());
        }
        // 支出核算主题id
        if (payload.getFromSourceId() != null) {
            update.set(qdo.fromSourceId, payload.getFromSourceId());
        }
        // 收入核算主题类型
        if (payload.getToSourceType() != null) {
            update.set(qdo.toSourceType, payload.getToSourceType());
        }
        // 收入核算主题id
        if (payload.getToSourceId() != null) {
            update.set(qdo.toSourceId, payload.getToSourceId());
        }
        // 申请结算当量数
        if (payload.getApplySettleEqva() != null) {
            update.set(qdo.applySettleEqva, payload.getApplySettleEqva());
        }
        // 申请结算总额
        if (payload.getApplySettleAmt() != null) {
            update.set(qdo.applySettleAmt, payload.getApplySettleAmt());
        }
        // 结算单价
        if (payload.getSettlePrice() != null) {
            update.set(qdo.settlePrice, payload.getSettlePrice());
        }
        // 交易凭证文件code
        if (payload.getFileCodes() != null) {
            update.set(qdo.fileCodes, payload.getFileCodes());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 结算单号
            if (nullFields.contains("settleNo")) {
                update.setNull(qdo.settleNo);
            }
            // 结算状态
            if (nullFields.contains("settleStatus")) {
                update.setNull(qdo.settleStatus);
            }
            // 审批状态
            if (nullFields.contains("apprStatus")) {
                update.setNull(qdo.apprStatus);
            }
            // 流程实例id
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 结算类型（项目立项，项目结项。。。）
            if (nullFields.contains("settleType")) {
                update.setNull(qdo.settleType);
            }
            // 结算日期
            if (nullFields.contains("settleDate")) {
                update.setNull(qdo.settleDate);
            }
            // 财务期间id
            if (nullFields.contains("finPeriodId")) {
                update.setNull(qdo.finPeriodId);
            }
            // 支出核算主题类型
            if (nullFields.contains("fromSourceType")) {
                update.setNull(qdo.fromSourceType);
            }
            // 支出核算主题id
            if (nullFields.contains("fromSourceId")) {
                update.setNull(qdo.fromSourceId);
            }
            // 收入核算主题类型
            if (nullFields.contains("toSourceType")) {
                update.setNull(qdo.toSourceType);
            }
            // 收入核算主题id
            if (nullFields.contains("toSourceId")) {
                update.setNull(qdo.toSourceId);
            }
            // 申请结算当量数
            if (nullFields.contains("applySettleEqva")) {
                update.setNull(qdo.applySettleEqva);
            }
            // 申请结算总额
            if (nullFields.contains("applySettleAmt")) {
                update.setNull(qdo.applySettleAmt);
            }
            // 结算单价
            if (nullFields.contains("settlePrice")) {
                update.setNull(qdo.settlePrice);
            }
            // 交易凭证文件code
            if (nullFields.contains("fileCodes")) {
                update.setNull(qdo.fileCodes);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

