package com.elitesland.tw.tw5.server.prd.pms.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.pms.payload.PmsProjectSummaryPayload;
import com.elitesland.tw.tw5.api.prd.pms.query.PmsProjectSummaryQuery;
import com.elitesland.tw.tw5.api.prd.pms.vo.PmsProjectSummaryVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.PmsProjectSummaryDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectSummaryDO;
import com.elitesland.tw.tw5.server.prd.pms.repo.PmsProjectSummaryRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 项目会议纪要
 *
 * @author zoey
 * @date 2024-05-28
 */
@Repository
@RequiredArgsConstructor
public class PmsProjectSummaryDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PmsProjectSummaryRepo repo;
    private final QPmsProjectSummaryDO qdo = QPmsProjectSummaryDO.pmsProjectSummaryDO;
    private final QPmsProjectDO qdoProject = QPmsProjectDO.pmsProjectDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectSummaryVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PmsProjectSummaryVO.class,
                qdo.id,
                //qdo.remark,
                qdo.createUserId,
                //qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 纪要编号
                qdo.summaryNo,
                // 纪要名称
                qdo.summaryName,
                // 项目id
                qdo.projId,
                // 消息内容
                qdo.summaryContent,
                // 状态（0：已保存；1：已发布）
                qdo.summaryStatus,
                // 记录日期
                qdo.summaryDate,
                //项目名称
                qdoProject.projName,
                //项目编号
                qdoProject.projNo,
                //交付bu
                qdoProject.deliBuId,
                //项目经理
                qdoProject.pmResId,
                //交付负责人
                qdoProject.deliUserId
        )).from(qdo).leftJoin(qdoProject).on(qdo.projId.eq(qdoProject.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PmsProjectSummaryVO> getJpaQueryWhere(PmsProjectSummaryQuery query) {
        JPAQuery<PmsProjectSummaryVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PmsProjectSummaryQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo).leftJoin(qdoProject).on(qdo.projId.eq(qdoProject.id));
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PmsProjectSummaryQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 纪要编号 精确 */
        if (!ObjectUtils.isEmpty(query.getSummaryNo())) {
            list.add(qdo.summaryNo.eq(query.getSummaryNo()));
        }
        /** 纪要名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getSummaryName())) {
            list.add(qdo.summaryName.like(SqlUtil.toSqlLikeString(query.getSummaryName())));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }
        /** 消息内容 精确 */
        if (!ObjectUtils.isEmpty(query.getSummaryContent())) {
            list.add(qdo.summaryContent.eq(query.getSummaryContent()));
        }
        /** 状态（0：已保存；1：已发布） 精确 */
        if (!ObjectUtils.isEmpty(query.getSummaryStatus())) {
            list.add(qdo.summaryStatus.eq(query.getSummaryStatus()));
        }

        /** 项目名称或者编号 */
        if (!ObjectUtils.isEmpty(query.getProjSearch())) {
            list.add(qdoProject.projName.like(SqlUtil.toSqlLikeString(query.getProjSearch())).or(qdoProject.projNo.like(SqlUtil.toSqlLikeString(query.getProjSearch()))));
        }
        /** 项目经理 */
        if (!ObjectUtils.isEmpty(query.getPmResId())) {
            list.add(qdoProject.pmResId.eq(query.getPmResId()));
        }
        /** 交付bu 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliBuId())) {
            list.add(qdoProject.deliBuId.eq(query.getDeliBuId()));
        }
        /** 交付负责人 */
        if (!ObjectUtils.isEmpty(query.getDeliUserId())) {
            list.add(qdoProject.deliUserId.eq(query.getDeliUserId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PmsProjectSummaryVO queryByKey(Long id) {
        JPAQuery<PmsProjectSummaryVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PmsProjectSummaryVO> queryListDynamic(PmsProjectSummaryQuery query) {
        JPAQuery<PmsProjectSummaryVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PmsProjectSummaryVO> queryPaging(PmsProjectSummaryQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<PmsProjectSummaryVO> jpaQuery = getJpaQueryWhere(query);
        List<PmsProjectSummaryVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PmsProjectSummaryVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PmsProjectSummaryDO save(PmsProjectSummaryDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PmsProjectSummaryDO> saveAll(List<PmsProjectSummaryDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PmsProjectSummaryPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 纪要编号
        if (payload.getSummaryNo() != null) {
            update.set(qdo.summaryNo, payload.getSummaryNo());
        }
        // 纪要名称
        if (payload.getSummaryName() != null) {
            update.set(qdo.summaryName, payload.getSummaryName());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 消息内容
        if (payload.getSummaryContent() != null) {
            update.set(qdo.summaryContent, payload.getSummaryContent());
        }
        // 状态（0：已保存；1：已发布）
        if (payload.getSummaryStatus() != null) {
            update.set(qdo.summaryStatus, payload.getSummaryStatus());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 纪要编号
            if (nullFields.contains("summaryNo")) {
                update.setNull(qdo.summaryNo);
            }
            // 纪要名称
            if (nullFields.contains("summaryName")) {
                update.setNull(qdo.summaryName);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 消息内容
            if (nullFields.contains("summaryContent")) {
                update.setNull(qdo.summaryContent);
            }
            // 状态（0：已保存；1：已发布）
            if (nullFields.contains("summaryStatus")) {
                update.setNull(qdo.summaryStatus);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

