package com.elitesland.tw.tw5.server.common;

import com.elitesland.workflow.exception.WorkflowException;
import lombok.Data;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 统一返回类
 *
 * @author duwh
 * @date 2023/02/14
 */
@Data
public class TwOutputUtil<T> {

    /**
     * 是否成功
     */
    private boolean ok;

    /**
     * 业务数据
     */
    private T data;

    /**
     * 成功信息
     */
    private List<TwMessage> infos = new ArrayList<>();

    /**
     * 错误信息
     */
    private List<TwMessage> errors = new ArrayList<>();

    /**
     * 警告信息
     */
    private List<TwMessage> warns = new ArrayList<>();

    private TwOutputUtil(boolean ok) {
        this.ok = ok;
    }

    /**
     * 返回
     *
     * @return 结果对象
     */
    public static TwOutputUtil output(boolean ok) {
        return new TwOutputUtil(ok);
    }

    /**
     * 最简单形式成功返回
     *
     * @return 结果对象
     */
    public static TwOutputUtil ok() {
        TwOutputUtil outputUtil = new TwOutputUtil(true);
        outputUtil.infos.add(TwMessage.of("COM-SUCCESS", "操作成功", null));
        return outputUtil;
    }

    /**
     * 最简单形式成功返回
     *
     * @param data 返回数据
     * @return 结果对象
     */
    public static <T> TwOutputUtil ok(T data) {
        TwOutputUtil outputUtil = new TwOutputUtil(true);
        outputUtil.setData(data);
        outputUtil.infos.add(TwMessage.of("COM-SUCCESS", "操作成功", null));
        return outputUtil;
    }

    /**
     * 返回成功
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public static TwOutputUtil ok(String localeCode, String defaultMsg, Object data) {
        TwOutputUtil outputUtil = new TwOutputUtil(true);
        outputUtil.data = data;
        outputUtil.infos.add(TwMessage.of(localeCode, defaultMsg, data));
        return outputUtil;
    }

    /**
     * 最简单形式失败返回
     *
     * @return 结果对象
     */
    public static TwOutputUtil error() {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.errors.add(TwMessage.of("COM-ERROR", "操作失败", null));
        return outputUtil;
    }

    /**
     * 返回失败
     *
     * @param defaultMsg 默认信息
     * @return 结果对象
     */
    public static TwOutputUtil error(String defaultMsg) {
        return error("",defaultMsg,null);
    }

    /**
     * 返回失败
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public static TwOutputUtil error(String localeCode, String defaultMsg, Object data) {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.data = data;
        outputUtil.errors.add(TwMessage.of(localeCode, defaultMsg, data));
        return outputUtil;
    }

    /**
     * 添加Exception异常
     *
     * @param exception 国际化码
     * @return 结果对象
     */
    public static TwOutputUtil addException(Exception exception) {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.errors.add(TwMessage.of("EXCEPTION", exception.getMessage(), null));
        return outputUtil;
    }

    /**
     * 添加twException异常
     *
     * @param twException 国际化码
     * @return 结果对象
     */
    public static TwOutputUtil addTwException(TwException twException) {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.errors = twException.getErrors();
        outputUtil.warns = twException.getWarns();
        return outputUtil;
    }

    /**
     * 添加NotOnlineException异常
     *
     * @param notOnlineException 异常
     * @return 结果对象
     */
    public static TwOutputUtil addNotOnlineException(NotOnlineException notOnlineException) {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.errors = notOnlineException.getErrors();
        outputUtil.warns = notOnlineException.getWarns();
        return outputUtil;
    }

    /**
     * 工作流异常  添加WorkflowException异常
     *
     * @param workflowException 异常
     * @return 结果对象
     */
    public static TwOutputUtil addWorkflowException(WorkflowException workflowException) {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.addError(workflowException.getCode()+"",workflowException.getMessage(),null);
        return outputUtil;
    }

    /**
     * 添加失败
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public TwOutputUtil addError(String localeCode, String defaultMsg, Object data) {
        this.errors.add(TwMessage.of(localeCode, defaultMsg, data));
        return this;
    }

    /**
     * 最简单形式警告返回
     *
     * @return 结果对象
     */
    public static TwOutputUtil warn() {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.warns.add(TwMessage.of("COM-WARN", "是否确认？", null));
        return outputUtil;
    }

    /**
     * 返回警告
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public static TwOutputUtil warn(String localeCode, String defaultMsg, Object data) {
        TwOutputUtil outputUtil = new TwOutputUtil(false);
        outputUtil.data = data;
        outputUtil.warns.add(TwMessage.of(localeCode, defaultMsg, data));
        return outputUtil;
    }


    /**
     * 添加警告
     *
     * @param localeCode 国际化码
     * @param defaultMsg 默认信息
     * @param data       数据
     * @return 结果对象
     */
    public TwOutputUtil addWarn(String localeCode, String defaultMsg, Object data) {
        this.warns.add(TwMessage.of(localeCode, defaultMsg, data));
        return this;
    }

    /**
     * 为了对接工作流的异常信息提示，必须添加该方法
     * @return 返回是否成功
     */
    public boolean isSuccess() {
        return this.ok;
    }
    /**
     * 为了对接工作流的异常信息提示，必须添加该方法
     * @return 返回是否成功
     */
    public String getMsg() {
        return this.errors.stream().map(TwMessage::getMsg).collect(Collectors.joining(","));
    }

}
