package com.elitesland.tw.tw5.server.prd.borrow.dao;

import com.elitesland.tw.tw5.api.prd.borrow.payload.BorrowWriteOffPayload;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowWriteOffQuery;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowMoneyVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowWriteOffVO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.BorrowWriteOffDO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.QBorrowMoneyDO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.QBorrowWriteOffDO;
import com.elitesland.tw.tw5.server.prd.borrow.repo.BorrowWriteOffRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 借款核销
 *
 * @author wangly
 * @date 2024-07-15
 */
@Repository
@RequiredArgsConstructor
public class BorrowWriteOffDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final BorrowWriteOffRepo repo;
    private final QBorrowWriteOffDO qdo = QBorrowWriteOffDO.borrowWriteOffDO;
    private final QBorrowMoneyDO qBorrowMoneyDO = QBorrowMoneyDO.borrowMoneyDO;
    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<BorrowWriteOffVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(BorrowWriteOffVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 借款ID
                qdo.borrowId,
                // 核销关联单据ID
                qdo.reimId,
                // 核销单据号
                qdo.reimNo,
                // 核销类型 [BORROW_MONEY:WRITE_OFF_TYPE]
                qdo.writeOffType,
                // 核销单据名称
                qdo.writeOffName,
                // 核销人ID
                qdo.writeOffUserId,
                // 核销日期
                qdo.writeOffDate,
                // 核销金额
                qdo.writeOffAmt,
                // 扩展字段1
                qdo.ext1,
                // 扩展字段2
                qdo.ext2,
                // 扩展字段3
                qdo.ext3,
                // 扩展字4
                qdo.ext4,
                // 扩展字5
                qdo.ext5
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<BorrowWriteOffVO> getJpaQueryWhere(BorrowWriteOffQuery query) {
        JPAQuery<BorrowWriteOffVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(BorrowWriteOffQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(BorrowWriteOffQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 借款ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBorrowId())) {
            list.add(qdo.borrowId.eq(query.getBorrowId()));
        }
        /** 借款ID集合 精确 */
        if (!CollectionUtils.isEmpty(query.getBorrowIdList())) {
            list.add(qdo.borrowId.in(query.getBorrowIdList()));
        }
        /** 核销关联单据ID 精确 */
        if (!ObjectUtils.isEmpty(query.getReimId())) {
            list.add(qdo.reimId.eq(query.getReimId()));
        }
        /** 核销单据号 精确 */
        if (!ObjectUtils.isEmpty(query.getReimNo())) {
            list.add(qdo.reimNo.eq(query.getReimNo()));
        }
        /** 核销类型 [BORROW_MONEY:WRITE_OFF_TYPE] 精确 */
        if (!ObjectUtils.isEmpty(query.getWriteOffType())) {
            list.add(qdo.writeOffType.eq(query.getWriteOffType()));
        }
        /** 核销单据名称 精确 */
        if (!ObjectUtils.isEmpty(query.getWriteOffName())) {
            list.add(qdo.writeOffName.eq(query.getWriteOffName()));
        }
        /** 核销人ID 精确 */
        if (!ObjectUtils.isEmpty(query.getWriteOffUserId())) {
            list.add(qdo.writeOffUserId.eq(query.getWriteOffUserId()));
        }
        /** 核销日期 精确 */
        if (!ObjectUtils.isEmpty(query.getWriteOffDate())) {
            list.add(qdo.writeOffDate.eq(query.getWriteOffDate()));
        }
        /** 核销金额 精确 */
        if (!ObjectUtils.isEmpty(query.getWriteOffAmt())) {
            list.add(qdo.writeOffAmt.eq(query.getWriteOffAmt()));
        }
        /** 扩展字段1 精确 */
        if (!ObjectUtils.isEmpty(query.getExt1())) {
            list.add(qdo.ext1.eq(query.getExt1()));
        }
        /** 扩展字段2 精确 */
        if (!ObjectUtils.isEmpty(query.getExt2())) {
            list.add(qdo.ext2.eq(query.getExt2()));
        }
        /** 扩展字段3 精确 */
        if (!ObjectUtils.isEmpty(query.getExt3())) {
            list.add(qdo.ext3.eq(query.getExt3()));
        }
        /** 扩展字4 精确 */
        if (!ObjectUtils.isEmpty(query.getExt4())) {
            list.add(qdo.ext4.eq(query.getExt4()));
        }
        /** 扩展字5 精确 */
        if (!ObjectUtils.isEmpty(query.getExt5())) {
            list.add(qdo.ext5.eq(query.getExt5()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public BorrowWriteOffVO queryByKey(Long id) {
        JPAQuery<BorrowWriteOffVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<BorrowWriteOffVO> queryListDynamic(BorrowWriteOffQuery query) {
        JPAQuery<BorrowWriteOffVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<BorrowWriteOffVO> queryPaging(BorrowWriteOffQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<BorrowWriteOffVO> jpaQuery = getJpaQueryWhere(query);
        List<BorrowWriteOffVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<BorrowWriteOffVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public BorrowWriteOffDO save(BorrowWriteOffDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<BorrowWriteOffDO> saveAll(List<BorrowWriteOffDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(BorrowWriteOffPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 借款ID
        if (payload.getBorrowId() != null) {
            update.set(qdo.borrowId, payload.getBorrowId());
        }
        // 核销关联单据ID
        if (payload.getReimId() != null) {
            update.set(qdo.reimId, payload.getReimId());
        }
        // 核销单据号
        if (payload.getReimNo() != null) {
            update.set(qdo.reimNo, payload.getReimNo());
        }
        // 核销类型 [BORROW_MONEY:WRITE_OFF_TYPE]
        if (payload.getWriteOffType() != null) {
            update.set(qdo.writeOffType, payload.getWriteOffType());
        }
        // 核销单据名称
        if (payload.getWriteOffName() != null) {
            update.set(qdo.writeOffName, payload.getWriteOffName());
        }
        // 核销人ID
        if (payload.getWriteOffUserId() != null) {
            update.set(qdo.writeOffUserId, payload.getWriteOffUserId());
        }
        // 核销日期
        if (payload.getWriteOffDate() != null) {
            update.set(qdo.writeOffDate, payload.getWriteOffDate());
        }
        // 核销金额
        if (payload.getWriteOffAmt() != null) {
            update.set(qdo.writeOffAmt, payload.getWriteOffAmt());
        }
        // 扩展字段1
        if (payload.getExt1() != null) {
            update.set(qdo.ext1, payload.getExt1());
        }
        // 扩展字段2
        if (payload.getExt2() != null) {
            update.set(qdo.ext2, payload.getExt2());
        }
        // 扩展字段3
        if (payload.getExt3() != null) {
            update.set(qdo.ext3, payload.getExt3());
        }
        // 扩展字4
        if (payload.getExt4() != null) {
            update.set(qdo.ext4, payload.getExt4());
        }
        // 扩展字5
        if (payload.getExt5() != null) {
            update.set(qdo.ext5, payload.getExt5());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 借款ID
            if (nullFields.contains("borrowId")) {
                update.setNull(qdo.borrowId);
            }
            // 核销关联单据ID
            if (nullFields.contains("reimId")) {
                update.setNull(qdo.reimId);
            }
            // 核销单据号
            if (nullFields.contains("reimNo")) {
                update.setNull(qdo.reimNo);
            }
            // 核销类型 [BORROW_MONEY:WRITE_OFF_TYPE]
            if (nullFields.contains("writeOffType")) {
                update.setNull(qdo.writeOffType);
            }
            // 核销单据名称
            if (nullFields.contains("writeOffName")) {
                update.setNull(qdo.writeOffName);
            }
            // 核销人ID
            if (nullFields.contains("writeOffUserId")) {
                update.setNull(qdo.writeOffUserId);
            }
            // 核销日期
            if (nullFields.contains("writeOffDate")) {
                update.setNull(qdo.writeOffDate);
            }
            // 核销金额
            if (nullFields.contains("writeOffAmt")) {
                update.setNull(qdo.writeOffAmt);
            }
            // 扩展字段1
            if (nullFields.contains("ext1")) {
                update.setNull(qdo.ext1);
            }
            // 扩展字段2
            if (nullFields.contains("ext2")) {
                update.setNull(qdo.ext2);
            }
            // 扩展字段3
            if (nullFields.contains("ext3")) {
                update.setNull(qdo.ext3);
            }
            // 扩展字4
            if (nullFields.contains("ext4")) {
                update.setNull(qdo.ext4);
            }
            // 扩展字5
            if (nullFields.contains("ext5")) {
                update.setNull(qdo.ext5);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    public List<BorrowMoneyVO> queryByReimId(Long reimId) {
        return jpaQueryFactory.select(Projections.bean(BorrowMoneyVO.class,
                qBorrowMoneyDO.id,
                qdo.writeOffAmt,
                qBorrowMoneyDO.borrowNo,
                // 借款名称
                qBorrowMoneyDO.borrowName,
                // 业务类型 udc[BORROW_MONEY:BUSINESS_TYPE]
                qBorrowMoneyDO.businessType,
                // 借款金额
                qBorrowMoneyDO.borrowAmt,
                // 已核销金额
                qBorrowMoneyDO.alreadyWriteOffAmt,
                // 未核销金额
                qBorrowMoneyDO.unwriteOffAmt,
                // 预计核销日期
                qBorrowMoneyDO.preWriteOffDate
        )).from(qdo).leftJoin(qBorrowMoneyDO).on(qdo.borrowId.eq(qBorrowMoneyDO.id)).
                where(qdo.reimId.eq(reimId)).
                where(qBorrowMoneyDO.deleteFlag.eq(0)).
                fetch();
    }

    /**
     * 通过报销id删除核销数据
     *
     * @param reimId 报销id
     */
    public long deleteByReimId(Long reimId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.reimId.eq(reimId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
}

