package com.elitesland.tw.tw5.server.prd.borrow.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.borrow.payload.BorrowMoneyPayload;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowMoneyQuery;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowWriteOffQuery;
import com.elitesland.tw.tw5.api.prd.borrow.query.TransferApplyQuery;
import com.elitesland.tw.tw5.api.prd.borrow.service.BorrowMoneyService;
import com.elitesland.tw.tw5.api.prd.borrow.service.BorrowWriteOffService;
import com.elitesland.tw.tw5.api.prd.borrow.service.TransferApplyService;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowMoneyVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowWriteOffVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.TransferApplyVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationVO;
import com.elitesland.tw.tw5.server.common.GenerateSeqNumConstants;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.prd.borrow.constant.BorrowStatusEnum;
import com.elitesland.tw.tw5.server.prd.borrow.convert.BorrowMoneyConvert;
import com.elitesland.tw.tw5.server.prd.borrow.dao.BorrowMoneyDAO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.BorrowMoneyDO;
import com.elitesland.tw.tw5.server.prd.borrow.repo.BorrowMoneyRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.OrgEnum;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 借款管理
 *
 * @author wangly
 * @date 2024-07-15
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BorrowMoneyServiceImpl extends BaseServiceImpl implements BorrowMoneyService {

    private final BorrowMoneyRepo borrowMoneyRepo;
    private final BorrowMoneyDAO borrowMoneyDAO;
    private final WorkflowUtil workflowUtil;
    private final BorrowWriteOffService borrowWriteOffService;
    private final TransferApplyService transferApplyService;
    private final CacheUtil cacheUtil;
    private final TransactionUtilService transactionUtilService;

    @Override
    public PagingVO<BorrowMoneyVO> queryPaging(BorrowMoneyQuery query){
        return borrowMoneyDAO.queryPaging(query);
    }

    @Override
    public List<BorrowMoneyVO> queryListDynamic(BorrowMoneyQuery query){
        return borrowMoneyDAO.queryListDynamic(query);
    }

    @Override
    public BorrowMoneyVO queryByKey(Long key) {
        BorrowMoneyVO borrowMoneyVO = borrowMoneyDAO.queryByKey(key);
        if(borrowMoneyVO!=null){
            transfer(Collections.singletonList(borrowMoneyVO));
        }
        return borrowMoneyVO;
    }

    @Override
    public BorrowMoneyVO insert(BorrowMoneyPayload payload) {
        // 保存数据
        BorrowMoneyDO save = transactionUtilService.executeWithCallable(() -> borrowMoneyRepo.save(this.processData(payload)));
        // 发起流程
        BorrowMoneyVO borrowMoneyVO = BorrowMoneyConvert.INSTANCE.toVo(save);
        activeBorrowMoneyProc(borrowMoneyVO);
        return borrowMoneyVO;
    }

    /**
     * 处理数据
     * @param payload payload
     * @return
     */
    private BorrowMoneyDO processData(BorrowMoneyPayload payload) {
        BorrowMoneyDO entityDo = BorrowMoneyConvert.INSTANCE.toDo(payload);
        entityDo.setRepaymentUserId(entityDo.getApplicantUserId());
        entityDo.setBorrowNo(generateSeqNum(GenerateSeqNumConstants.BORROW_MONEY_NO));
        entityDo.setAlreadyWriteOffAmt(BigDecimal.ZERO);
        entityDo.setUnwriteOffAmt(entityDo.getBorrowAmt());
        return entityDo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BorrowMoneyVO update(BorrowMoneyPayload payload) {
        BorrowMoneyDO entity = borrowMoneyRepo.findById(payload.getId()).orElseGet(BorrowMoneyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        BorrowMoneyDO entityDo = BorrowMoneyConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return BorrowMoneyConvert.INSTANCE.toVo(borrowMoneyRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(BorrowMoneyPayload payload) {
        BorrowMoneyDO entity = borrowMoneyRepo.findById(payload.getId()).orElseGet(BorrowMoneyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        return borrowMoneyDAO.updateByKeyDynamic(payload);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            borrowMoneyDAO.deleteSoft(keys);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateBorrowStatus(List<Long> keys,String status){
       if (!keys.isEmpty()) {
            borrowMoneyDAO.updateBorrowStatus(keys,status);
        }
    }
    // 激活流程
    private void activeBorrowMoneyProc(BorrowMoneyVO borrowMoneyVO){
        ProcessInfo processInfo;
        HashMap<String, Object> variables = new HashMap<>();
        // 借款金额
        variables.put("borrowAmt", borrowMoneyVO.getBorrowAmt());
        // 申请人所在部门
        PrdOrgOrganizationVO org = cacheUtil.getOrg(borrowMoneyVO.getApplicantBuId());
        variables.put("buLevel", org.getExtString1());
        // 三级部门
        if (OrgEnum.BuLevel.BU_LEVEL3.getCode().equals(org.getExtString1())) {
            borrowMoneyVO.setLevel3BuManagerId(org.getManageId());
            // 二级部门负责人
            List<PrdOrgOrganizationRefVO> allFatherOrg = cacheUtil.getAllFatherOrgByOrgId(org.getId());
            Optional<PrdOrgOrganizationRefVO> secondOrgRef = allFatherOrg.stream().filter((vo) -> StringUtils.hasText(vo.getBuLevel()) && vo.getBuLevel().equals(OrgEnum.BuLevel.BU_LEVEL2.getCode()) && vo.getOrgStatus().equalsIgnoreCase(OrgEnum.OrgStatus.ACTIVE.getCode())).findFirst();
            secondOrgRef.ifPresent(prdOrgOrganizationRefVO -> borrowMoneyVO.setLevel2BuManagerId(prdOrgOrganizationRefVO.getManageId()));
            // 一级部门负责人
            PrdOrgOrganizationRefVO buLevel1ByOrgId = cacheUtil.getBULevel1ByOrgId(org.getId());
            borrowMoneyVO.setLevel1BuManagerId(buLevel1ByOrgId.getManageId());
            // 二级部门
        }else if (OrgEnum.BuLevel.BU_LEVEL2.getCode().equals(org.getExtString1())) {
            borrowMoneyVO.setLevel2BuManagerId(org.getManageId());
            PrdOrgOrganizationRefVO buLevel1ByOrgId = cacheUtil.getBULevel1ByOrgId(org.getId());
            borrowMoneyVO.setLevel1BuManagerId(buLevel1ByOrgId.getManageId());
            // 一级部门
        }else if (OrgEnum.BuLevel.BU_LEVEL1.getCode().equals(org.getExtString1())) {
            borrowMoneyVO.setLevel1BuManagerId(org.getManageId());
        }
        String procInstName = borrowMoneyVO.getBorrowName() + "-借款流程";
        // 发起流程
        processInfo = workflowUtil.simpleStartProcess(StartProcessPayload.of(
                ProcDefKey.DIB_BORROW_MONEY.name(),
                procInstName,
                borrowMoneyVO.getId() + "",
                variables), borrowMoneyVO);
        BorrowMoneyPayload payload = new BorrowMoneyPayload();
        payload.setId(borrowMoneyVO.getId());
        if (ProcInstStatus.APPROVING.getDesc().equals(processInfo.getProcInstStatus().getDesc())) {
            payload.setApprStatus(BorrowStatusEnum.APPROVING.getCode());
        } else if (ProcInstStatus.APPROVED.getDesc().equals(processInfo.getProcInstStatus().getDesc())) {
            payload.setProcInstStatus(ProcInstStatus.APPROVED);
            payload.setApprovedTime(LocalDateTime.now());
            payload.setApprStatus(BorrowStatusEnum.PAID.getCode());
        }
        borrowMoneyDAO.updateByKeyDynamic(payload);
    }

    private void transfer(List<BorrowMoneyVO> borrowMoneyVOList){

        if(!CollectionUtils.isEmpty(borrowMoneyVOList)){
            List<Long> idList = borrowMoneyVOList.stream().map(BorrowMoneyVO::getId).collect(Collectors.toList());

            // 查询对应的出让申请数据
            TransferApplyQuery transferApplyQuery = new TransferApplyQuery();
            transferApplyQuery.setBorrowIdList(idList);
            List<TransferApplyVO> transferApplyVOS = transferApplyService.queryListDynamic(transferApplyQuery);
            Map<Long, List<TransferApplyVO>> transferApplyMap = new HashMap<>();
            if(!CollectionUtils.isEmpty(transferApplyVOS)){
                transferApplyMap = transferApplyVOS.stream().collect(Collectors.groupingBy(TransferApplyVO::getBorrowId));
            }
            // 查询核销的数据
            BorrowWriteOffQuery borrowWriteOffQuery = new BorrowWriteOffQuery();
            borrowWriteOffQuery.setBorrowIdList(idList);
            List<BorrowWriteOffVO> borrowWriteOffVOS = borrowWriteOffService.queryListDynamic(borrowWriteOffQuery);
            Map<Long, List<BorrowWriteOffVO>> borrowWriteOffMap = new HashMap<>();
            if(!CollectionUtils.isEmpty(borrowWriteOffVOS)){
                borrowWriteOffMap = borrowWriteOffVOS.stream().collect(Collectors.groupingBy(BorrowWriteOffVO::getBorrowId));
            }
            for(BorrowMoneyVO borrowMoneyVO:borrowMoneyVOList){
                if(transferApplyMap.containsKey(borrowMoneyVO.getId())){
                    borrowMoneyVO.setTransferApplyList(transferApplyMap.get(borrowMoneyVO.getId()));
                }
                if(borrowWriteOffMap.containsKey(borrowMoneyVO.getId())){
                    borrowMoneyVO.setBorrowWriteOffList(borrowWriteOffMap.get(borrowMoneyVO.getId()));
                }
            }
        }
    }
}
