package com.elitesland.tw.tw5.server.prd.crm.dao;

import com.elitesland.tw.tw5.api.prd.crm.payload.CrmGiftInfoPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmGiftInfoQuery;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmGiftListQuery;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmGiftInfoVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmGiftInfoDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmGiftInfoDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmGiftListDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmGiftInfoRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 礼品单明细
 *
 * @author xxb
 * @date 2023-08-31
 */
@Repository
@RequiredArgsConstructor
public class CrmGiftInfoDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CrmGiftInfoRepo repo;
    private final QCrmGiftInfoDO qdo = QCrmGiftInfoDO.crmGiftInfoDO;

    private final QCrmGiftListDO qgiftList = QCrmGiftListDO.crmGiftListDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CrmGiftInfoVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CrmGiftInfoVO.class,
                qdo.id,
                //qdo.remark,
                qdo.createUserId,
                qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 礼品单ID
                qdo.giftListId,
                // 礼品名称
                qdo.giftName,
                // 申请数量
                qdo.giftNumber,
                // 单位
                qdo.giftUnit,
                // 客户表主键
                qdo.customerId,
                // 客户名称
                qdo.custName,
                // 客户方地址
                qdo.custAddress,
                // 客户经营部门
                qdo.custOperBu,
                // 岗位情况
                qdo.jobDetail,
                // 联系方式
                qdo.mobile,
                // 礼品寄送地址
                qdo.locationDetail,
                // 赠送理由
                qdo.reason,
                // 经营人脉
                qdo.peopleId,
                // 经营人脉
                qdo.peopleName
        )).from(qdo);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CrmGiftInfoVO> getJpaQuerySelect2() {
        return jpaQueryFactory.select(Projections.bean(CrmGiftInfoVO.class,
                qdo.id,
                //qdo.remark,
                // qdo.createUserId,
                // qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 礼品单ID
                qdo.giftListId,
                // 礼品名称
                qdo.giftName,
                // 申请数量
                qdo.giftNumber,
                // 单位
                qdo.giftUnit,
                // 客户表主键
                qdo.customerId,
                // 客户名称
                qdo.custName,
                // 客户方地址
                qdo.custAddress,
                // 客户经营部门
                qdo.custOperBu,
                // 岗位情况
                qdo.jobDetail,
                // 联系方式
                qdo.mobile,
                // 礼品寄送地址
                qdo.locationDetail,
                // 赠送理由
                qdo.reason,
                // 经营人脉
                qdo.peopleId,
                // 经营人脉
                qdo.peopleName,
                // 礼品单编号
                qgiftList.giftNo,
                // 申请用户ID
                qgiftList.applyUserId,
                // 申请人bu
                qgiftList.applyUserBuId,
                // 创建人bu
                qgiftList.createUserBuId,
                // 申请日期
                qgiftList.applyDate,
                // 申请状态
                qgiftList.applyStatus,
                // 礼品类型
                qgiftList.giftType1,
                qgiftList.giftType2,
                qgiftList.specialUsersFlag,
                qgiftList.procInstId,
                qgiftList.createUserId,
                qgiftList.creator
        )).from(qdo).leftJoin(qgiftList).on(qdo.giftListId.eq(qgiftList.id));
    }


    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CrmGiftInfoVO> getJpaQueryWhere(CrmGiftInfoQuery query) {
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CrmGiftInfoVO> getJpaQueryWhere(CrmGiftListQuery query) {
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQuerySelect2();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qgiftList, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CrmGiftInfoQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CrmGiftListQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CrmGiftInfoQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 礼品单ID 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftListId())) {
            list.add(qdo.giftListId.eq(query.getGiftListId()));
        }
        /** 礼品名称 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftName())) {
            list.add(qdo.giftName.eq(query.getGiftName()));
        }
        /** 申请数量 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftNumber())) {
            list.add(qdo.giftNumber.eq(query.getGiftNumber()));
        }
        /** 单位 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftUnit())) {
            list.add(qdo.giftUnit.eq(query.getGiftUnit()));
        }
        /** 客户名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustName())) {
            list.add(qdo.custName.like(SqlUtil.toSqlLikeString(query.getCustName())));
        }
        /** 客户方地址 精确 */
        if (!ObjectUtils.isEmpty(query.getCustAddress())) {
            list.add(qdo.custAddress.eq(query.getCustAddress()));
        }
        /** 客户经营部门 精确 */
        if (!ObjectUtils.isEmpty(query.getCustOperBu())) {
            list.add(qdo.custOperBu.eq(query.getCustOperBu()));
        }
        /** 岗位情况 精确 */
        if (!ObjectUtils.isEmpty(query.getJobDetail())) {
            list.add(qdo.jobDetail.eq(query.getJobDetail()));
        }
        /** 联系方式 精确 */
        if (!ObjectUtils.isEmpty(query.getMobile())) {
            list.add(qdo.mobile.eq(query.getMobile()));
        }
        /** 礼品寄送地址 精确 */
        if (!ObjectUtils.isEmpty(query.getLocationDetail())) {
            list.add(qdo.locationDetail.eq(query.getLocationDetail()));
        }
        /** 赠送理由 精确 */
        if (!ObjectUtils.isEmpty(query.getReason())) {
            list.add(qdo.reason.eq(query.getReason()));
        }
        /** 经营人脉 精确 */
        if (!ObjectUtils.isEmpty(query.getPeopleName())) {
            list.add(qdo.peopleName.eq(query.getPeopleName()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CrmGiftListQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 申请人 精确 */
        if (ObjectUtils.isEmpty(query.getApplyUserId())){
            if(!ObjectUtils.isEmpty(query.getUserIds())){
                list.add(qgiftList.createUserId.in(query.getUserIds()).or(qgiftList.applyUserId.in(query.getUserIds())));
            }
        }else{
            list.add(qgiftList.applyUserId.eq(query.getApplyUserId()));
        }
        /** 礼品单编号 模糊 */
        if (!ObjectUtils.isEmpty(query.getGiftNo())) {
            list.add(qgiftList.giftNo.like(SqlUtil.toSqlLikeString(query.getGiftNo())));
        }
        /** 申请人bu 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyUserBuId())) {
            list.add(qgiftList.applyUserBuId.eq(query.getApplyUserBuId()));
        }
        /** 创建人bu 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserBuId())) {
            list.add(qgiftList.createUserBuId.eq(query.getCreateUserBuId()));
        }
        /** 礼品类型 一级 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftType1())) {
            list.add(qgiftList.giftType1.eq(query.getGiftType1()));
        }
        /** 礼品类型 二级 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftType2())) {
            list.add(qgiftList.giftType2.eq(query.getGiftType2()));
        }
        /** 特殊申请用户标志 精确 */
        if (!ObjectUtils.isEmpty(query.getSpecialUsersFlag())) {
            list.add(qgiftList.specialUsersFlag.eq(query.getSpecialUsersFlag()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyStatus())) {
            list.add(qgiftList.applyStatus.eq(query.getApplyStatus()));
        }
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qgiftList.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qgiftList.applyDate.eq(query.getApplyDate()));
        }
        /** 客户名称 模糊 */
        if (!ObjectUtils.isEmpty(query.getCustomerId())) {
            list.add(qdo.customerId.eq(query.getCustomerId()));
        }
        if (!ObjectUtils.isEmpty(query.getCustName())) {
            list.add(qdo.custName.like(SqlUtil.toSqlLikeString(query.getCustName())));
        }
        if (!ObjectUtils.isEmpty(query.getKeys())) {
            list.add(qdo.id.in(query.getKeys()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CrmGiftInfoVO queryByKey(Long id) {
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CrmGiftInfoVO> queryListDynamic(CrmGiftInfoQuery query) {
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CrmGiftInfoVO> queryPaging(CrmGiftInfoQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQueryWhere(query);
        List<CrmGiftInfoVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CrmGiftInfoVO>builder().records(result).total(total).build();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CrmGiftInfoVO> pagingByGiftList(CrmGiftListQuery query) {
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQueryWhere(query);
        List<CrmGiftInfoVO> result = jpaQuery.fetch();
        long total = result.size();
        if (total == 0) {
            return PagingVO.empty();
        }
        List<CrmGiftInfoVO> info = result.stream().skip(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize()).collect(Collectors.toList());
        return PagingVO.<CrmGiftInfoVO>builder().records(info).total(total).build();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public List<CrmGiftInfoVO> byGiftList(CrmGiftListQuery query) {
        JPAQuery<CrmGiftInfoVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CrmGiftInfoDO save(CrmGiftInfoDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CrmGiftInfoDO> saveAll(List<CrmGiftInfoDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CrmGiftInfoPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 礼品单ID
        if (payload.getGiftListId() != null) {
            update.set(qdo.giftListId, payload.getGiftListId());
        }
        // 礼品名称
        if (payload.getGiftName() != null) {
            update.set(qdo.giftName, payload.getGiftName());
        }
        // 申请数量
        if (payload.getGiftNumber() != null) {
            update.set(qdo.giftNumber, payload.getGiftNumber());
        }
        // 单位
        if (payload.getGiftUnit() != null) {
            update.set(qdo.giftUnit, payload.getGiftUnit());
        }
        // 客户名称
        if (payload.getCustName() != null) {
            update.set(qdo.custName, payload.getCustName());
        }
        // 客户方地址
        if (payload.getCustAddress() != null) {
            update.set(qdo.custAddress, payload.getCustAddress());
        }
        // 客户经营部门
        if (payload.getCustOperBu() != null) {
            update.set(qdo.custOperBu, payload.getCustOperBu());
        }
        // 岗位情况
        if (payload.getJobDetail() != null) {
            update.set(qdo.jobDetail, payload.getJobDetail());
        }
        // 联系方式
        if (payload.getMobile() != null) {
            update.set(qdo.mobile, payload.getMobile());
        }
        // 礼品寄送地址
        if (payload.getLocationDetail() != null) {
            update.set(qdo.locationDetail, payload.getLocationDetail());
        }
        // 赠送理由
        if (payload.getReason() != null) {
            update.set(qdo.reason, payload.getReason());
        }
        // 经营人脉
        if (payload.getPeopleName() != null) {
            update.set(qdo.peopleName, payload.getPeopleName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 礼品单ID
            if (nullFields.contains("giftListId")) {
                update.setNull(qdo.giftListId);
            }
            // 礼品名称
            if (nullFields.contains("giftName")) {
                update.setNull(qdo.giftName);
            }
            // 申请数量
            if (nullFields.contains("giftNumber")) {
                update.setNull(qdo.giftNumber);
            }
            // 单位
            if (nullFields.contains("giftUnit")) {
                update.setNull(qdo.giftUnit);
            }
            // 客户名称
            if (nullFields.contains("custName")) {
                update.setNull(qdo.custName);
            }
            // 客户方地址
            if (nullFields.contains("custAddress")) {
                update.setNull(qdo.custAddress);
            }
            // 客户经营部门
            if (nullFields.contains("custOperBu")) {
                update.setNull(qdo.custOperBu);
            }
            // 岗位情况
            if (nullFields.contains("jobDetail")) {
                update.setNull(qdo.jobDetail);
            }
            // 联系方式
            if (nullFields.contains("mobile")) {
                update.setNull(qdo.mobile);
            }
            // 礼品寄送地址
            if (nullFields.contains("locationDetail")) {
                update.setNull(qdo.locationDetail);
            }
            // 赠送理由
            if (nullFields.contains("reason")) {
                update.setNull(qdo.reason);
            }
            // 经营人脉
            if (nullFields.contains("peopleName")) {
                update.setNull(qdo.peopleName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

