package com.elitesland.tw.tw5.server.prd.crm.dao;

import com.elitesland.tw.tw5.api.prd.crm.payload.CrmGiftListPayload;
import com.elitesland.tw.tw5.api.prd.crm.query.CrmGiftListQuery;
import com.elitesland.tw.tw5.api.prd.crm.vo.CrmGiftListVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.server.prd.crm.entity.CrmGiftListDO;
import com.elitesland.tw.tw5.server.prd.crm.entity.QCrmGiftListDO;
import com.elitesland.tw.tw5.server.prd.crm.repo.CrmGiftListRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 礼品单
 *
 * @author xxb
 * @date 2023-08-31
 */
@Repository
@RequiredArgsConstructor
public class CrmGiftListDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CrmGiftListRepo repo;
    private final QCrmGiftListDO qdo = QCrmGiftListDO.crmGiftListDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CrmGiftListVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CrmGiftListVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 礼品单编号
                qdo.giftNo,
                // 申请用户ID
                qdo.applyUserId,
                // 申请人bu
                qdo.applyUserBuId,
                // 创建人bu
                qdo.createUserBuId,
                // 礼品类型 一级
                qdo.giftType1,
                // 礼品类型 二级
                qdo.giftType2,
                // 特殊申请用户标志
                qdo.specialUsersFlag,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.applyDate,
                // 审批时间
                qdo.approvedTime
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CrmGiftListVO> getJpaQueryWhere(CrmGiftListQuery query) {
        JPAQuery<CrmGiftListVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CrmGiftListQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
            .select(qdo.count())
            .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CrmGiftListQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 礼品单编号 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftNo())) {
            list.add(qdo.giftNo.eq(query.getGiftNo()));
        }
        /** 申请人bu 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyUserBuId())) {
            list.add(qdo.applyUserBuId.eq(query.getApplyUserBuId()));
        }
        /** 创建人bu 精确 */
        if (!ObjectUtils.isEmpty(query.getCreateUserBuId())) {
            list.add(qdo.createUserBuId.eq(query.getCreateUserBuId()));
        }
        /** 礼品类型 一级 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftType1())) {
            list.add(qdo.giftType1.eq(query.getGiftType1()));
        }
        /** 礼品类型 二级 精确 */
        if (!ObjectUtils.isEmpty(query.getGiftType2())) {
            list.add(qdo.giftType2.eq(query.getGiftType2()));
        }
        /** 特殊申请用户标志 精确 */
        if (!ObjectUtils.isEmpty(query.getSpecialUsersFlag())) {
            list.add(qdo.specialUsersFlag.eq(query.getSpecialUsersFlag()));
        }
        /** 流程实例ID 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstId())) {
            list.add(qdo.procInstId.eq(query.getProcInstId()));
        }
        /** 流程审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getProcInstStatus())) {
            list.add(qdo.procInstStatus.eq(query.getProcInstStatus()));
        }
        /** 提交时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qdo.applyDate.eq(query.getApplyDate()));
        }
        /** 审批时间 精确 */
        if (!ObjectUtils.isEmpty(query.getApprovedTime())) {
            list.add(qdo.approvedTime.eq(query.getApprovedTime()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyStatus())) {
            list.add(qdo.applyStatus.eq(query.getApplyStatus()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplyDate())) {
            list.add(qdo.applyDate.eq(query.getApplyDate()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CrmGiftListVO queryByKey(Long id) {
        JPAQuery<CrmGiftListVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CrmGiftListVO> queryListDynamic(CrmGiftListQuery query) {
        JPAQuery<CrmGiftListVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CrmGiftListVO> queryPaging(CrmGiftListQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<CrmGiftListVO> jpaQuery = getJpaQueryWhere(query);
        List<CrmGiftListVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CrmGiftListVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CrmGiftListDO save(CrmGiftListDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CrmGiftListDO> saveAll(List<CrmGiftListDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CrmGiftListPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 礼品单编号
        if (payload.getGiftNo() != null) {
            update.set(qdo.giftNo, payload.getGiftNo());
        }
        // 申请人
        if (payload.getApplyUserId() != null) {
            update.set(qdo.applyUserId, payload.getApplyUserId());
        }
        // 申请人bu
        if (payload.getApplyUserBuId() != null) {
            update.set(qdo.applyUserBuId, payload.getApplyUserBuId());
        }
        // 礼品类型 一级
        if (payload.getGiftType1() != null) {
            update.set(qdo.giftType1, payload.getGiftType1());
        }
        // 礼品类型 二级
        if (payload.getGiftType2() != null) {
            update.set(qdo.giftType2, payload.getGiftType2());
        }
        // 特殊申请用户标志
        if (payload.getSpecialUsersFlag() != null) {
            update.set(qdo.specialUsersFlag, payload.getSpecialUsersFlag());
        }
        // 流程实例ID
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        // 流程审批状态
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        if (payload.getApplyStatus() != null) {
            update.set(qdo.applyStatus, payload.getApplyStatus());
        }
        // 提交时间
        if (payload.getApplyDate() != null) {
            update.set(qdo.applyDate, payload.getApplyDate());
        }
        // 审批时间
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 礼品单编号
            if (nullFields.contains("giftNo")) {
                update.setNull(qdo.giftNo);
            }
            // 申请人bu
            if (nullFields.contains("applyUserBuId")) {
                update.setNull(qdo.applyUserBuId);
            }
            // 礼品类型 一级
            if (nullFields.contains("giftType1")) {
                update.setNull(qdo.giftType1);
            }
            // 礼品类型 二级
            if (nullFields.contains("giftType2")) {
                update.setNull(qdo.giftType2);
            }
            // 特殊申请用户标志
            if (nullFields.contains("specialUsersFlag")) {
                update.setNull(qdo.specialUsersFlag);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

