package com.elitesland.tw.tw5.server.prd.inv.dao;

import com.elitesland.tw.tw5.api.prd.inv.payload.InvItemCatPayload;
import com.elitesland.tw.tw5.api.prd.inv.query.InvItemCatQuery;
import com.elitesland.tw.tw5.api.prd.inv.vo.InvItemCatVO;
import com.elitesland.tw.tw5.server.prd.inv.entity.InvItemCatDO;
import com.elitesland.tw.tw5.server.prd.inv.entity.QInvItemCatDO;
import com.elitesland.tw.tw5.server.prd.inv.repo.InvItemCatRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 发票商品分类对象
 *
 * @author zoey
 * @date 2023-07-31
 */
@Repository
@RequiredArgsConstructor
public class InvItemCatDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final InvItemCatRepo repo;
    private final QInvItemCatDO qdo = QInvItemCatDO.invItemCatDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<InvItemCatVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(InvItemCatVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 省份
                qdo.province,
                // 商品编码（税务局）
                qdo.goodsCode,
                // 商品编码名称（税务局）
                qdo.goodsCodeName
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<InvItemCatVO> getJpaQueryWhere(InvItemCatQuery query) {
        JPAQuery<InvItemCatVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(InvItemCatQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(InvItemCatQuery query){
        List<Predicate> list = new ArrayList<>();
        /**  精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 省份 精确 */
        if (!ObjectUtils.isEmpty(query.getProvince())) {
            list.add(qdo.province.eq(query.getProvince()));
        }
        /** 商品编码（税务局） 精确 */
        if (!ObjectUtils.isEmpty(query.getGoodsCode())) {
            list.add(qdo.goodsCode.eq(query.getGoodsCode()));
        }
        /** 商品编码名称（税务局） 模糊 */
        if (!ObjectUtils.isEmpty(query.getGoodsCodeName())) {
             list.add(qdo.goodsCodeName.like(SqlUtil.toSqlLikeString(query.getGoodsCodeName())));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public InvItemCatVO queryByKey(Long id) {
        JPAQuery<InvItemCatVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<InvItemCatVO> queryListDynamic(InvItemCatQuery query) {
        JPAQuery<InvItemCatVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<InvItemCatVO> queryPaging(InvItemCatQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<InvItemCatVO> jpaQuery = getJpaQueryWhere(query);
        List<InvItemCatVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<InvItemCatVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public InvItemCatDO save(InvItemCatDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<InvItemCatDO> saveAll(List<InvItemCatDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(InvItemCatPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 省份
        if (payload.getProvince() != null) {
            update.set(qdo.province, payload.getProvince());
        }
        // 商品编码（税务局）
        if (payload.getGoodsCode() != null) {
            update.set(qdo.goodsCode, payload.getGoodsCode());
        }
        // 商品编码名称（税务局）
        if (payload.getGoodsCodeName() != null) {
            update.set(qdo.goodsCodeName, payload.getGoodsCodeName());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 省份
            if (nullFields.contains("province")) {
                update.setNull(qdo.province);
            }
            // 商品编码（税务局）
            if (nullFields.contains("goodsCode")) {
                update.setNull(qdo.goodsCode);
            }
            // 商品编码名称（税务局）
            if (nullFields.contains("goodsCodeName")) {
                update.setNull(qdo.goodsCodeName);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

