package com.elitesland.tw.tw5.server.prd.purchase.controller;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseContractManagerPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseContractManagerQuery;
import com.elitesland.tw.tw5.api.prd.purchase.service.PurchaseContractManagerService;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseContractManagerVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemLogVO;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.permission.annotation.FunctionDetail;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionDomain;
import com.elitesland.tw.tw5.server.common.permission.annotation.PermissionFunction;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionCodeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.FunctionTypeEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionDomainEnum;
import com.elitesland.tw.tw5.server.common.permission.enums.PermissionTypeEnum;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.Arrays;
import java.util.List;

/**
 * 采购合同管理
 *
 * @author lipengfei
 * @date 2023-05-30
 */
@Api(tags = "采购合同管理")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/purchaseContractManager")
@Slf4j
@PermissionDomain(domain = PermissionDomainEnum.PCN_PURCHAS)
public class PurchaseContractManagerController {

    private final PurchaseContractManagerService purchaseContractManagerService;

    /**
     * 采购合同管理-新增
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PostMapping
    @ApiOperation("采购合同管理-新增")
    public TwOutputUtil insert(@RequestBody PurchaseContractManagerPayload payload) {
        return TwOutputUtil.ok(purchaseContractManagerService.insert(payload));
    }

    /**
     * 采购合同管理-更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    @PutMapping
    @ApiOperation("采购合同管理-更新")
    public TwOutputUtil update(@RequestBody PurchaseContractManagerPayload payload) {
        return TwOutputUtil.ok(purchaseContractManagerService.update(payload));
    }

    /**
     * 采购合同管理-主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/{key}")
    @UdcNameClass
    @ApiOperation("采购合同管理-主键查询详细页面")
    public TwOutputUtil<PurchaseContractManagerVO> queryOneByKey(@PathVariable Long key) {
        return TwOutputUtil.ok(purchaseContractManagerService.queryByKey(key));
    }

    /**
     * 采购合同管理-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/paging")
    @UdcNameClass
    @ApiOperation("采购合同管理-分页")
    public TwOutputUtil<PagingVO<PurchaseContractManagerVO>> paging(PurchaseContractManagerQuery query) {
        return TwOutputUtil.ok(purchaseContractManagerService.queryPaging(query));
    }


    /**
     * 采购合同管理-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/permissionPaging")
    @UdcNameClass
    @ApiOperation("采购合同管理-权限分页")
    @PermissionFunction(functions = {
            @FunctionDetail(functionCode = FunctionCodeEnum.PCN_PURCHAS_LIST_DATA_POWER, type = FunctionTypeEnum.TAB)
    }, fieldPermission = true, permissionType = PermissionTypeEnum.DATA_PERMISSION)
    public TwOutputUtil<PagingVO<PurchaseContractManagerVO>> permissionPaging(@RequestBody PurchaseContractManagerQuery query) {
        return TwOutputUtil.ok(purchaseContractManagerService.queryPermissionPaging(query));
    }
    /**
     * 采购合同管理-查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/list")
    @UdcNameClass
    @ApiOperation("采购合同管理-查询列表")
    public TwOutputUtil<List<PurchaseContractManagerVO>> queryList(PurchaseContractManagerQuery query) {
        return TwOutputUtil.ok(purchaseContractManagerService.queryListDynamic(query));
    }

    /**
     * 采购合同管理-删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/deleteSoft")
    @ApiOperation("采购合同管理-删除")
    public TwOutputUtil deleteSoft(Long[] keys) {
        purchaseContractManagerService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

//    /**
//     * 采购公司/供应商-列表查询
//     *
//     * @return result
//     */
//    @GetMapping("/ab/queryList/{relateType}")
//    public TwOutputUtil<List<PrdAbVO>> abQueryList(@PathVariable String relateType) {
//        return TwOutputUtil.ok(purchaseContractManagerService.abQueryList(relateType));
//    }

    /**
     * 查询采购合同操作日志
     *
     * @param key 采购合同主键
     * @return
     */
    @GetMapping("/log/list/{key}")
    @ApiOperation("采购合同管理-查询采购合同操作日志")
    public TwOutputUtil<List<PrdSystemLogVO>> queryLogList(@PathVariable Long key) {
        return TwOutputUtil.ok(purchaseContractManagerService.queryLogList(key));
    }

    /**
     * 激活/提交审批
     *
     * @param keys 主键id
     * @return result
     */
    @PatchMapping("/active")
    //@GlobalTransactional
    @ApiOperation("激活/提交审批")
    public TwOutputUtil active(Long[] keys) {
        purchaseContractManagerService.active(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 批量合同归档 合同
     *
     * @param ids
     * @return
     */
    @PatchMapping("/filing/{ids}")
    @ApiOperation("采购合同管理-批量合同归档合同")
    public TwOutputUtil contractFilingByIds(@PathVariable Long[] ids) {
        purchaseContractManagerService.contractFilingByIds(Arrays.asList(ids));
        return TwOutputUtil.ok();
    }

    /**
     * 批量合同归档 合同
     *
     * @param
     * @return
     */
    @PatchMapping("/closeContract")
    @ApiOperation("采购合同管理-采购合同的关闭")
    public TwOutputUtil closeContract(@RequestParam List<Long> ids,@RequestParam String closeReason) {
        purchaseContractManagerService.closeContract(ids,closeReason);
        return TwOutputUtil.ok();
    }

    /**
     * 导出excel
     *
     * @param query 查询条件
     * @return result
     */
    @GetMapping("/downloadPurCon")
    @ApiOperation("采购合同管理-导出excel")
    public void downloadPurCon(HttpServletResponse response, PurchaseContractManagerQuery query) {
        purchaseContractManagerService.downloadPurCon(response, query);
    }

    /**
     * 权限-导出excel
     *
     * @param query 查询条件
     * @return result
     */
    @PostMapping("/permissionDownloadPurCon")
    @ApiOperation("采购合同管理-权限-导出excel")
    public void permissionDownloadPurCon(HttpServletResponse response,@RequestBody PurchaseContractManagerQuery query) {
        purchaseContractManagerService.permissionDownloadPurCon(response, query);
    }

    /**
     * 采购合同终止 根据终止ID查询采购合同信息
     *
     * @param overId 采购合同终止id
     * @return 采购合同信息
     */
    @ApiOperation(value = "根据采购合同终止ID查询采购合同信息")
    @GetMapping("/over/purchase/{overId}")
    public TwOutputUtil<PurchaseContractManagerVO> selectPurchaseContractByOverId(@PathVariable long overId) {
        PurchaseContractManagerVO purchaseContractVO = purchaseContractManagerService.selectPurchaseContractByOverId(overId);
        return TwOutputUtil.ok(purchaseContractVO);
    }

    /**
     * 采购合同合同归档
     *
     * @param purConId 归档合同Id
     * @return 采购合同信息
     */
    @ApiOperation(value = "采购合同合同归档")
    @GetMapping("/contractArchiving/{purConId}")
    public TwOutputUtil purContractArchiving(@PathVariable Long purConId, String stampFileCodes) {
        purchaseContractManagerService.purContractArchiving(purConId, stampFileCodes);
        return TwOutputUtil.ok();
    }
}
