package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentSlipPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.CostUndertakeDepartmentQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.CostUndertakeDepartmentVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccFinancialSubjectDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QCostUndertakeDepartmentDO;
import com.elitesland.tw.tw5.server.prd.purchase.repo.CostUndertakeDepartmentRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * 费用承担部门
 *
 * @author likunpeng
 * @date 2023-11-07
 */
@Repository
@RequiredArgsConstructor
public class CostUndertakeDepartmentDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CostUndertakeDepartmentRepo repo;
    private final QCostUndertakeDepartmentDO qdo = QCostUndertakeDepartmentDO.costUndertakeDepartmentDO;
    private final QAccFinancialSubjectDO qAccFinancialSubjectDO = QAccFinancialSubjectDO.accFinancialSubjectDO;


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CostUndertakeDepartmentVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CostUndertakeDepartmentVO.class,
                        qdo.id,
                        //qdo.remark,
                        //qdo.createUserId,
                        //qdo.creator,
                        //qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 付款申请单id
                        qdo.paymentApplyId,
                        // 费用承担BU_ID
                        qdo.bearBuId,
                        qdo.invoiceNo,
                        qdo.jdeFlag,
                        // 费用承担金额
                        qdo.bearAmt,
                        // 科目说明(pms_acc_mas表id)
                        qdo.accMasId,
                        // 科目说明Desc
                        qAccFinancialSubjectDO.accName.as("accMasName")
                )).from(qdo)
                .leftJoin(qAccFinancialSubjectDO).on(qdo.accMasId.eq(qAccFinancialSubjectDO.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CostUndertakeDepartmentVO> getJpaQueryWhere(CostUndertakeDepartmentQuery query) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CostUndertakeDepartmentQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CostUndertakeDepartmentQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 付款申请单id 精确 */
        if (!ObjectUtils.isEmpty(query.getPaymentApplyId())) {
            list.add(qdo.paymentApplyId.eq(query.getPaymentApplyId()));
        }
        /** 费用承担BU_ID 精确 */
        if (!ObjectUtils.isEmpty(query.getBearBuId())) {
            list.add(qdo.bearBuId.eq(query.getBearBuId()));
        }
        /** 费用承担金额 精确 */
        if (!ObjectUtils.isEmpty(query.getBearAmt())) {
            list.add(qdo.bearAmt.eq(query.getBearAmt()));
        }
        /** 科目说明(pms_acc_mas表id) 精确 */
        if (!ObjectUtils.isEmpty(query.getAccMasId())) {
            list.add(qdo.accMasId.eq(query.getAccMasId()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CostUndertakeDepartmentVO queryByKey(Long id) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CostUndertakeDepartmentVO> queryListDynamic(CostUndertakeDepartmentQuery query) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CostUndertakeDepartmentVO> queryPaging(CostUndertakeDepartmentQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQueryWhere(query);
        List<CostUndertakeDepartmentVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CostUndertakeDepartmentVO>builder().records(result).total(total).build();
    }


    /**
     * 根据付款申请单id删除费用承担部门
     *
     * @param paymentApplyId 付款申请单id
     */
    public void deleteByPaymentApplyId(Long paymentApplyId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.paymentApplyId.eq(paymentApplyId));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }

    /**
     * 根据付款申请单ID查询费用承担部门
     *
     * @param paymentApplyId 付款申请单ID
     * @return 费用承担部门VO
     */
    public List<CostUndertakeDepartmentVO> queryListByPaymentApplyId(Long paymentApplyId) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.paymentApplyId.eq(paymentApplyId));
        return jpaQuery.fetch();
    }

    /**
     * 根据发票号查询费用承担部门
     *
     * @param invoiceNo 发票号
     * @return 费用承担部门VO
     */
    public List<CostUndertakeDepartmentVO> queryListByJdeInvoiceNo(String invoiceNo) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.invoiceNo.eq(invoiceNo));
        jpaQuery.where(qdo.jdeFlag.eq(1).or(qdo.jdeFlag.isNotNull()));
        jpaQuery.where(qdo.deleteFlag.eq(0));

        return jpaQuery.fetch();
    }


    /**
     * 根据发票号批量查询费用承担部门
     *
     * @param payload
     * @return 费用承担部门VO
     */
    public List<CostUndertakeDepartmentVO> queryListByJdeInvoiceNos(PaymentSlipPayload payload) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQuerySelect();
        if(!Objects.isNull(payload.getPaymentApplyId())){
            jpaQuery.where(qdo.paymentApplyId.notIn(payload.getPaymentApplyId()));
        }
        jpaQuery.where(qdo.invoiceNo.in(payload.getInvoiceNos()));
        jpaQuery.where(qdo.jdeFlag.eq(1).or(qdo.jdeFlag.isNotNull()));
        jpaQuery.where(qdo.deleteFlag.eq(0));

        return jpaQuery.fetch();
    }

    /**
     * 根据发票号付款申请单ID查询费用承担部门
     *
     * @param invoiceNo 发票号
     * @return 费用承担部门VO
     */
    public List<CostUndertakeDepartmentVO> queryListByJdePaymentApplyIdOrInvoiceNo(String invoiceNo, Long paymentApplyId) {
        JPAQuery<CostUndertakeDepartmentVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.paymentApplyId.eq(paymentApplyId));
        jpaQuery.where(qdo.invoiceNo.eq(invoiceNo));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }

    /**
     * @param ids
     * @return
     * @Description
     **/
    public void queryTW4IdsByIdsupdateCostUndertakeDepartmentIds(List<Long> ids) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.jdeFlag, 1)
                .where(qdo.id.in(ids));
        ;
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }
}

