package com.elitesland.tw.tw5.server.prd.task.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.task.payload.TaskSettleTimesheetPayload;
import com.elitesland.tw.tw5.api.prd.task.query.TaskSettleTimesheetQuery;
import com.elitesland.tw.tw5.api.prd.task.vo.TaskSettleTimesheetVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QTimesheetDO;
import com.elitesland.tw.tw5.server.prd.task.entity.QTaskInfoDO;
import com.elitesland.tw.tw5.server.prd.task.entity.QTaskPackageDO;
import com.elitesland.tw.tw5.server.prd.task.entity.QTaskSettleTimesheetDO;
import com.elitesland.tw.tw5.server.prd.task.entity.TaskSettleTimesheetDO;
import com.elitesland.tw.tw5.server.prd.task.repo.TaskSettleTimesheetRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 任务结算工时关系表
 *
 * @author carl
 * @date 2023-12-20
 */
@Repository
@RequiredArgsConstructor
public class TaskSettleTimesheetDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final TaskSettleTimesheetRepo repo;
    private final QTaskSettleTimesheetDO qdo = QTaskSettleTimesheetDO.taskSettleTimesheetDO;
    private final QTaskInfoDO qTaskInfoDO = QTaskInfoDO.taskInfoDO;
    private final QTimesheetDO qTimesheetDO = QTimesheetDO.timesheetDO;
    private final QTaskPackageDO packageQdo = QTaskPackageDO.taskPackageDO;

    /**
     * 根据事由查询 任务
     *
     * @param reasonType
     * @param reasonId
     * @return
     */
    public List<TaskSettleTimesheetVO> queryTaskByReason(String reasonType, Long reasonId) {
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = jpaQueryFactory.select(Projections.bean(TaskSettleTimesheetVO.class,
                        qdo.id,
                        qdo.relatedActId,
                        qdo.eqva,
                        qdo.taskId,
                        qTaskInfoDO.taskName,
                        packageQdo.reasonType,
                        packageQdo.reasonId
                )).from(qdo).leftJoin(qTaskInfoDO).on(qdo.taskId.eq(qTaskInfoDO.id))
                .leftJoin(packageQdo).on(qTaskInfoDO.taskPackageId.eq(packageQdo.id))
                .where(packageQdo.reasonType.eq(reasonType))
                .where(packageQdo.reasonId.eq(reasonId));
        return jpaQuery.fetch();
    }

    /**
     * 根据事由查询 任务
     *
     * @param reasonTypes
     * @param reasonIds
     * @return
     */
    public List<TaskSettleTimesheetVO> queryTaskByReasons(List<String> reasonTypes, List<Long> reasonIds) {
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = jpaQueryFactory.select(Projections.bean(TaskSettleTimesheetVO.class,
                        qdo.id,
                        qdo.relatedActId,
                        qdo.eqva,
                        qdo.taskId,
                        qTaskInfoDO.taskName,
                        packageQdo.reasonType,
                        packageQdo.reasonId
                )).from(qdo).leftJoin(qTaskInfoDO).on(qdo.taskId.eq(qTaskInfoDO.id))
                .leftJoin(packageQdo).on(qTaskInfoDO.taskPackageId.eq(packageQdo.id))
                .where(packageQdo.reasonType.in(reasonTypes))
                .where(packageQdo.reasonId.in(reasonIds));
        return jpaQuery.fetch();
    }


    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<TaskSettleTimesheetVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(TaskSettleTimesheetVO.class,
                        qdo.id,
                        qdo.remark,
                        qdo.createUserId,
                        qdo.creator,
                        qdo.createTime,
                        //qdo.modifyUserId,
                        //qdo.updater,
                        //qdo.modifyTime,
                        // 任务id
                        qdo.taskId,
                        //任务名称
                        qTaskInfoDO.taskName,
                        // 结算id
                        qdo.settleId,
                        // 工时id
                        qdo.timesheetId,
                        //工时填报人id
                        qTimesheetDO.tsUserId,
                        // 工时
                        qdo.workHour,
                        // 当量
                        qdo.eqva,
                        // 工作日期
                        qdo.workDate,
                        // 工作日志
                        qdo.workDesc,
                        // 结算标记（0：否，1：是）
                        qdo.settleFlag
                )).from(qdo)
                .leftJoin(qTaskInfoDO).on(qdo.taskId.eq(qTaskInfoDO.id))
                .leftJoin(qTimesheetDO).on(qdo.timesheetId.eq(qTimesheetDO.id));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<TaskSettleTimesheetVO> getJpaQueryWhere(TaskSettleTimesheetQuery query) {
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(TaskSettleTimesheetQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo)
                .leftJoin(qTaskInfoDO).on(qdo.taskId.eq(qTaskInfoDO.id))
                .leftJoin(qTimesheetDO).on(qdo.timesheetId.eq(qTimesheetDO.id));
        ;
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(TaskSettleTimesheetQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 任务id 精确 */
        if (!ObjectUtils.isEmpty(query.getTaskId())) {
            list.add(qdo.taskId.eq(query.getTaskId()));
        }
        /** 任务id 精确 */
        if (!ObjectUtils.isEmpty(query.getTaskIds())) {
            list.add(qdo.taskId.in(query.getTaskIds()));
        }
        /** 结算id 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleId())) {
            list.add(qdo.settleId.eq(query.getSettleId()));
        }
        /** 结算id 精确 */
        if (!ObjectUtils.isEmpty(query.getSettleIds())) {
            list.add(qdo.settleId.in(query.getSettleIds()));
        }
        /** 工时id 精确 */
        if (!ObjectUtils.isEmpty(query.getTimesheetId())) {
            list.add(qdo.timesheetId.eq(query.getTimesheetId()));
        }
        /** 工时 精确 */
        if (!ObjectUtils.isEmpty(query.getWorkHour())) {
            list.add(qdo.workHour.eq(query.getWorkHour()));
        }
        /** 当量 精确 */
        if (!ObjectUtils.isEmpty(query.getEqva())) {
            list.add(qdo.eqva.eq(query.getEqva()));
        }
        /** 工作日期 精确 */
        if (!ObjectUtils.isEmpty(query.getWorkDate())) {
            list.add(qdo.workDate.eq(query.getWorkDate()));
        }
        /** 工作日志 模糊 */
        if (!ObjectUtils.isEmpty(query.getWorkDesc())) {
            list.add(qdo.workDesc.like(SqlUtil.toSqlLikeString(query.getWorkDesc())));
        }
        /** 结算标记（0：否，1：是） 精确 */
        if (query.getSettleFlag() != null) {
            list.add(qdo.settleFlag.eq(query.getSettleFlag()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public TaskSettleTimesheetVO queryByKey(Long id) {
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键查询
     *
     * @param timesheetId 工时主键
     * @return 结果
     */
    public TaskSettleTimesheetVO queryByTimesheetId(Long timesheetId) {
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.timesheetId.eq(timesheetId));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TaskSettleTimesheetVO> queryListDynamic(TaskSettleTimesheetQuery query) {
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<TaskSettleTimesheetDO> queryList(TaskSettleTimesheetQuery query) {
        JPAQuery<TaskSettleTimesheetDO> jpaQuery = jpaQueryFactory.select(qdo).from(qdo);
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<TaskSettleTimesheetVO> queryPaging(TaskSettleTimesheetQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<TaskSettleTimesheetVO> jpaQuery = getJpaQueryWhere(query);
        List<TaskSettleTimesheetVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<TaskSettleTimesheetVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public TaskSettleTimesheetDO save(TaskSettleTimesheetDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<TaskSettleTimesheetDO> saveAll(List<TaskSettleTimesheetDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(TaskSettleTimesheetPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 任务id
        if (payload.getTaskId() != null) {
            update.set(qdo.taskId, payload.getTaskId());
        }
        // 结算id
        if (payload.getSettleId() != null) {
            update.set(qdo.settleId, payload.getSettleId());
        }
        // 工时id
        if (payload.getTimesheetId() != null) {
            update.set(qdo.timesheetId, payload.getTimesheetId());
        }
        // 工时
        if (payload.getWorkHour() != null) {
            update.set(qdo.workHour, payload.getWorkHour());
        }
        // 当量
        if (payload.getEqva() != null) {
            update.set(qdo.eqva, payload.getEqva());
        }
        // 工作日期
        if (payload.getWorkDate() != null) {
            update.set(qdo.workDate, payload.getWorkDate());
        }
        // 工作日志
        if (payload.getWorkDesc() != null) {
            update.set(qdo.workDesc, payload.getWorkDesc());
        }
        // 结算标记（0：否，1：是）
        if (payload.getSettleFlag() != null) {
            update.set(qdo.settleFlag, payload.getSettleFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 任务id
            if (nullFields.contains("taskId")) {
                update.setNull(qdo.taskId);
            }
            // 结算id
            if (nullFields.contains("settleId")) {
                update.setNull(qdo.settleId);
            }
            // 工时id
            if (nullFields.contains("timesheetId")) {
                update.setNull(qdo.timesheetId);
            }
            // 工时
            if (nullFields.contains("workHour")) {
                update.setNull(qdo.workHour);
            }
            // 当量
            if (nullFields.contains("eqva")) {
                update.setNull(qdo.eqva);
            }
            // 工作日期
            if (nullFields.contains("workDate")) {
                update.setNull(qdo.workDate);
            }
            // 工作日志
            if (nullFields.contains("workDesc")) {
                update.setNull(qdo.workDesc);
            }
            // 结算标记（0：否，1：是）
            if (nullFields.contains("settleFlag")) {
                update.setNull(qdo.settleFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

