package com.elitesland.tw.tw5.server.prd.acc.dao;


import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimTripStdPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimTripStdQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimTripStdVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimTripStdDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimTripStdDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimTripStdRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 差旅报销额度管理
 *
 * @author sunxw
 * @date 2023-12-04
 */
@Repository
@RequiredArgsConstructor
public class AccReimTripStdDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimTripStdRepo repo;
    private final QAccReimTripStdDO qdo = QAccReimTripStdDO.accReimTripStdDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimTripStdVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimTripStdVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // P职级区间下限
                qdo.jobGradePlLow,
                // P职级区间上限
                qdo.jobGradePlUp,
                // M职级区间下限
                qdo.jobGradeMlLow,
                // M职级区间上限
                qdo.jobGradeMlUp,
                // FEE_TYPE
                qdo.feeType,
                // 城市级别
                qdo.cityLevel,
                // 项目报销级别
                qdo.projectFeeLevel,
                // 报销额度
                qdo.feeAmt
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimTripStdVO> getJpaQueryWhere(AccReimTripStdQuery query) {
        JPAQuery<AccReimTripStdVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimTripStdQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimTripStdQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** P职级区间下限 精确 */
        if (!ObjectUtils.isEmpty(query.getJobGradePlLow())) {
            list.add(qdo.jobGradePlLow.eq(query.getJobGradePlLow()));
        }
        /** P职级区间上限 精确 */
        if (!ObjectUtils.isEmpty(query.getJobGradePlUp())) {
            list.add(qdo.jobGradePlUp.eq(query.getJobGradePlUp()));
        }
        if (!ObjectUtils.isEmpty(query.getJobGrade())) {
            list.add(qdo.jobGradePlLow.loe(query.getJobGrade()));
            list.add(qdo.jobGradePlUp.goe(query.getJobGrade()));
        }
        /** M职级区间下限 精确 */
        /*if (!ObjectUtils.isEmpty(query.getJobGradeMlLow())) {
            list.add(qdo.jobGradeMlLow.eq(query.getJobGradeMlLow()));
        }*/
        /** M职级区间上限 精确 */
        /*if (!ObjectUtils.isEmpty(query.getJobGradeMlUp())) {
            list.add(qdo.jobGradeMlUp.eq(query.getJobGradeMlUp()));
        }*/
        /** FEE_TYPE 精确 */
        if (!ObjectUtils.isEmpty(query.getFeeType())) {
            list.add(qdo.feeType.eq(query.getFeeType()));
        }
        /** 城市级别 精确 */
        if (!ObjectUtils.isEmpty(query.getCityLevel())) {
            list.add(qdo.cityLevel.eq(query.getCityLevel()));
        }
        /** 项目报销级别 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectFeeLevel())) {
            list.add(qdo.projectFeeLevel.eq(query.getProjectFeeLevel()));
        }
        /** 报销额度 精确 */
        if (!ObjectUtils.isEmpty(query.getFeeAmt())) {
            list.add(qdo.feeAmt.eq(query.getFeeAmt()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimTripStdVO queryByKey(Long id) {
        JPAQuery<AccReimTripStdVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimTripStdVO> queryListDynamic(AccReimTripStdQuery query) {
        JPAQuery<AccReimTripStdVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimTripStdVO> queryPaging(AccReimTripStdQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimTripStdVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimTripStdVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimTripStdVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimTripStdDO save(AccReimTripStdDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimTripStdDO> saveAll(List<AccReimTripStdDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimTripStdPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // P职级区间下限
        if (payload.getJobGradePlLow() != null) {
            update.set(qdo.jobGradePlLow, payload.getJobGradePlLow());
        }
        // P职级区间上限
        if (payload.getJobGradePlUp() != null) {
            update.set(qdo.jobGradePlUp, payload.getJobGradePlUp());
        }
        // M职级区间下限
        if (payload.getJobGradeMlLow() != null) {
            update.set(qdo.jobGradeMlLow, payload.getJobGradeMlLow());
        }
        // M职级区间上限
        if (payload.getJobGradeMlUp() != null) {
            update.set(qdo.jobGradeMlUp, payload.getJobGradeMlUp());
        }
        // FEE_TYPE
        if (payload.getFeeType() != null) {
            update.set(qdo.feeType, payload.getFeeType());
        }
        // 城市级别
        if (payload.getCityLevel() != null) {
            update.set(qdo.cityLevel, payload.getCityLevel());
        }
        // 项目报销级别
        if (payload.getProjectFeeLevel() != null) {
            update.set(qdo.projectFeeLevel, payload.getProjectFeeLevel());
        }
        // 报销额度
        if (payload.getFeeAmt() != null) {
            update.set(qdo.feeAmt, payload.getFeeAmt());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // P职级区间下限
            if (nullFields.contains("jobGradePlLow")) {
                update.setNull(qdo.jobGradePlLow);
            }
            // P职级区间上限
            if (nullFields.contains("jobGradePlUp")) {
                update.setNull(qdo.jobGradePlUp);
            }
            // M职级区间下限
            if (nullFields.contains("jobGradeMlLow")) {
                update.setNull(qdo.jobGradeMlLow);
            }
            // M职级区间上限
            if (nullFields.contains("jobGradeMlUp")) {
                update.setNull(qdo.jobGradeMlUp);
            }
            // FEE_TYPE
            if (nullFields.contains("feeType")) {
                update.setNull(qdo.feeType);
            }
            // 城市级别
            if (nullFields.contains("cityLevel")) {
                update.setNull(qdo.cityLevel);
            }
            // 项目报销级别
            if (nullFields.contains("projectFeeLevel")) {
                update.setNull(qdo.projectFeeLevel);
            }
            // 报销额度
            if (nullFields.contains("feeAmt")) {
                update.setNull(qdo.feeAmt);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

