package com.elitesland.tw.tw5.server.prd.acc.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccSubjectTemplateDetailPayload;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccSubjectTemplatePayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccSubjectTemplateDetailQuery;
import com.elitesland.tw.tw5.api.prd.acc.query.AccSubjectTemplateQuery;
import com.elitesland.tw.tw5.api.prd.acc.service.AccSubjectTemplateDetailService;
import com.elitesland.tw.tw5.api.prd.acc.service.AccSubjectTemplateService;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccBudgetItemVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccSubjectTemplateDetailVO;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccSubjectTemplateVO;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemSelectionVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.acc.convert.AccSubjectTemplateConvert;
import com.elitesland.tw.tw5.server.prd.acc.dao.AccSubjectTemplateDAO;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccSubjectTemplateDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccSubjectTemplateRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.AccTemplTypeEnum;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 科目模板管理
 *
 * @author carl
 * @date 2023-09-25
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class AccSubjectTemplateServiceImpl extends BaseServiceImpl implements AccSubjectTemplateService {
    private final AccSubjectTemplateDetailService accSubjectTemplateDetailService;
    private final AccSubjectTemplateRepo accSubjectTemplateRepo;
    private final AccSubjectTemplateDAO accSubjectTemplateDAO;

    private final CacheUtil cacheUtil;

    @Override
    public PagingVO<AccSubjectTemplateVO> queryPaging(AccSubjectTemplateQuery query) {
        PagingVO<AccSubjectTemplateVO> pagingVO = accSubjectTemplateDAO.queryPaging(query);
        if (!ObjectUtils.isEmpty(pagingVO)) {
            pagingVO.getRecords().forEach(vo -> transferData(vo));
        }
        return pagingVO;
    }

    @Override
    public List<AccSubjectTemplateVO> queryListDynamic(AccSubjectTemplateQuery query) {
        return accSubjectTemplateDAO.queryListDynamic(query);
    }

    @Override
    public AccSubjectTemplateVO queryByKey(Long key) {
        AccSubjectTemplateVO accSubjectTemplateVO = accSubjectTemplateDAO.queryByKey(key);
        if (accSubjectTemplateVO != null) {
            List<AccSubjectTemplateDetailVO> accSubjectTemplateDetailVOS = accSubjectTemplateDetailService.queryListByTmplId(key);
            accSubjectTemplateVO.setTmpldVOs(accSubjectTemplateDetailVOS);
            transferData(accSubjectTemplateVO);
        }
        return accSubjectTemplateVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccSubjectTemplateVO insert(AccSubjectTemplatePayload payload) {
        // 数据验证
        checkData(payload);
        List<Long> ids = accSubjectTemplateDAO.queryByName(payload.getTmplName());
        if (ObjectUtils.isEmpty(ids)) {
            String code = generateSeqNum("ACC_SUBJECT_TEMPL");
            payload.setTmplNo(code);
            AccSubjectTemplateDO entityDo = AccSubjectTemplateConvert.INSTANCE.toDo(payload);
            AccSubjectTemplateDO save = accSubjectTemplateDAO.save(entityDo);
            AccSubjectTemplateVO accSubjectTemplateVO = AccSubjectTemplateConvert.INSTANCE.toVo(save);
            return accSubjectTemplateVO;
        } else {
            throw TwException.error("", "模板名称不可重复");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public AccSubjectTemplateVO update(AccSubjectTemplatePayload payload) {
        // 数据验证
        checkData(payload);
        List<Long> ids = accSubjectTemplateDAO.queryByName(payload.getTmplName());
        if (ObjectUtils.isEmpty(ids) || (ids.size() == 1 && ids.get(0).equals(payload.getId()))) {
            AccSubjectTemplateDO entityDo = AccSubjectTemplateConvert.INSTANCE.toDo(payload);
            AccSubjectTemplateDO save = accSubjectTemplateRepo.save(entityDo);
            AccSubjectTemplateVO accSubjectTemplateVO = AccSubjectTemplateConvert.INSTANCE.toVo(save);
            if (!ObjectUtils.isEmpty(payload.getTmpldPayloads())) {
                //添加模板科目明细
                payload.getTmpldPayloads().forEach(tmpldPayload -> tmpldPayload.setTmplId(save.getId()));
                accSubjectTemplateDetailService.bacthInsert(payload.getTmpldPayloads(), accSubjectTemplateVO);
            }
            return accSubjectTemplateVO;
        } else {
            throw TwException.error("", "模板名称不可重复");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(AccSubjectTemplatePayload payload) {
        long result = accSubjectTemplateDAO.updateByKeyDynamic(payload);
        return result;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            if (!keys.isEmpty()) {
                //删除模板数据
                accSubjectTemplateDAO.deleteSoft(keys);
                //删除模板关联科目
                accSubjectTemplateDetailService.deleteSoftByTemlIds(keys);
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateStatus(List<Long> keys, String status) {
        accSubjectTemplateDAO.updateStatus(keys, status);
    }

    @Override
    public void deleteSoftDetail(List<Long> keys) {
        //删除模板关联科目
        accSubjectTemplateDetailService.deleteSoft(keys);
    }

    @Override
    public List<AccSubjectTemplateDetailVO> bacthInsert(List<AccSubjectTemplateDetailPayload> payloads) {
        List<Long> collect = payloads.stream().filter(payload -> payload.getTmplId() != null).map(AccSubjectTemplateDetailPayload::getTmplId).distinct().collect(Collectors.toList());
        if (ObjectUtils.isEmpty(collect)) {
            throw TwException.error("", "模板id不可为空");
        }
        if (collect.size() > 1) {
            throw TwException.error("", "模板id不可多个");
        }
        AccSubjectTemplateVO accSubjectTemplateVO = accSubjectTemplateDAO.queryByKey(collect.get(0));


        return accSubjectTemplateDetailService.bacthInsert(payloads, accSubjectTemplateVO);
    }

    @Override
    public PagingVO<AccSubjectTemplateDetailVO> queryPagingDetail(AccSubjectTemplateDetailQuery query) {
        return accSubjectTemplateDetailService.queryPaging(query);
    }

    @Override
    public List<AccBudgetItemVO> queryBudgetItemList(Long tmplId) {
        return accSubjectTemplateDetailService.queryBudgetItemList(tmplId);
    }


    /**
     * 数据验证
     *
     * @param payload
     */
    private void checkData(AccSubjectTemplatePayload payload) {
        if (ObjectUtils.isEmpty(payload.getTmplClass())) {
            throw TwException.error("", "模板类别不可为空");
        }
//        if (ObjectUtils.isEmpty(payload.getTmplType())) {
//            throw TwException.error("", "适用类型不可为空");
//        }
        // 适用类型选择有限制
        // 当模板类型是 项目科目模板时，适用类型下拉框选择工作类，支持多选
        if (AccTemplTypeEnum.PROJ_ACC.getCode().equals(payload.getTmplClass())) {
            PrdSystemSelectionVO selection = cacheUtil.getSystemSelection("salecon:work_type");
            if (!ObjectUtils.isEmpty(selection)) {
                List<String> valueList = selection.getChildren().stream().map(v -> v.getSelectionValue()).collect(Collectors.toList());
                String[] arr = payload.getTmplType().split(",");
                for (String value : arr) {
                    if (!valueList.contains(value)) {
                        throw TwException.error("", "适用类型不符合限制,请核验！");
                    }
                }
            }
        }
        //当模板类型是BU科目模板时，适用类型下拉框，平台支持类BU、销售类BU、事业部BU、研发事业部BU 只能单选
        if (AccTemplTypeEnum.BU_ACC.getCode().equals(payload.getTmplClass())) {
            if (payload.getTmplType().contains(",")) {
                throw TwException.error("", "当模板类型是BU科目模板时，适用类型只能单选");
            }
            PrdSystemSelectionVO selection = cacheUtil.getSystemSelection("BU:SUIT_TYPY");
            if (!ObjectUtils.isEmpty(selection)) {
                long count = selection.getChildren().stream().filter(v -> payload.getTmplType().contains(v.getSelectionValue())).count();
                if (count == 0) {
                    throw TwException.error("", "适用类型不符合限制,请核验！");
                }
            }
        }
    }

    /**
     * 数据翻译
     *
     * @param templateVO
     */
    void transferData(AccSubjectTemplateVO templateVO) {
        // 适用类型选择有限制
        // 当模板类型是 项目科目模板时，适用类型下拉框选择工作类，支持多选
        if (AccTemplTypeEnum.PROJ_ACC.getCode().equals(templateVO.getTmplClass())) {
            if (!ObjectUtils.isEmpty(templateVO.getTmplType())) {
                String[] arr = templateVO.getTmplType().split(",");
                PrdSystemSelectionVO selection = cacheUtil.getSystemSelection("salecon:work_type");
                if (selection != null && !ObjectUtils.isEmpty(selection.getChildren())) {
                    List<String> tmplTypeNames = new ArrayList<>();
                    List<PrdSystemSelectionVO> childrens = selection.getChildren();
                    for (String value : arr) {
                        for (PrdSystemSelectionVO selectionVO : childrens) {
                            if (selectionVO.getSelectionValue().equalsIgnoreCase(value)) {
                                tmplTypeNames.add(selectionVO.getSelectionName());
                            }
                        }
                    }
                    templateVO.setTmplTypeNames(String.join(",", tmplTypeNames));
                }
            }
        }
        //当模板类型是BU科目模板时，适用类型下拉框，平台支持类BU、销售类BU、事业部BU、研发事业部BU 只能单选
        if (AccTemplTypeEnum.BU_ACC.getCode().equals(templateVO.getTmplClass())) {
            if (!ObjectUtils.isEmpty(templateVO.getTmplType())) {
                String tmplTypeNames = cacheUtil.transferSystemSelection("BU:SUIT_TYPY", templateVO.getTmplType());
                templateVO.setTmplTypeNames(tmplTypeNames);
            }
        }
    }

}
