package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalEqvaIncomePayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalEqvaIncomeQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalEqvaIncomeVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalEqvaIncomeDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalEqvaIncomeDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalEqvaIncomeRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 当量收入配置管理
 *
 * @author carl
 * @date 2023-11-07
 */
@Repository
@RequiredArgsConstructor
public class CalEqvaIncomeDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalEqvaIncomeRepo repo;
    private final QCalEqvaIncomeDO qdo = QCalEqvaIncomeDO.calEqvaIncomeDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaIncomeVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalEqvaIncomeVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 核算年度
                qdo.finYear,
                // 核算期间id
                qdo.finPeriod,

                // 工种
                qdo.jobType1,
                // 工种子类
                qdo.jobType2,
                // 合作方式
                qdo.coopType,
                // 城市级别
                qdo.cityLevel,
                // 单位当量收入
                qdo.preeqvaAmt,
                // 状态
                qdo.lineStatus,
                // 项目id
                qdo.projId,
                qdo.projName,
                // 资源id
                qdo.resId,
                //起始时间
                qdo.startDate,
                //失效时间
                qdo.endDate,
                //版本号
                qdo.version,
                //结算方式
                qdo.settleType
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaIncomeVO> getJpaQueryWhere(CalEqvaIncomeQuery query) {
        JPAQuery<CalEqvaIncomeVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalEqvaIncomeQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalEqvaIncomeQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        }
        /** 核算期间id 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        }
        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        }
        /** 合作方式 精确 */
        if (!ObjectUtils.isEmpty(query.getCoopType())) {
            list.add(qdo.coopType.eq(query.getCoopType()));
        }
        /** 城市级别 精确 */
        if (!ObjectUtils.isEmpty(query.getCityLevel())) {
            list.add(qdo.cityLevel.eq(query.getCityLevel()));
        }
        /** 单位当量收入 精确 */
        if (!ObjectUtils.isEmpty(query.getPreeqvaAmt())) {
            list.add(qdo.preeqvaAmt.eq(query.getPreeqvaAmt()));
        }
        /** 状态 精确 */
        if (!ObjectUtils.isEmpty(query.getLineStatus())) {
            list.add(qdo.lineStatus.eq(query.getLineStatus()));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        }
        /** 资源类型 精确 */
        if (!ObjectUtils.isEmpty(query.getResType())) {
            list.add(qdo.resType.eq(query.getResType()));
        } else {
            list.add(qdo.resType.eq("1"));
        }
        /** 起始时间 */
        if (!ObjectUtils.isEmpty(query.getStartDate())) {
            list.add(qdo.startDate.loe(query.getStartDate()));
        }
        /** 结算方式 */
        if (!ObjectUtils.isEmpty(query.getSettleType())) {
            list.add(qdo.settleType.eq(query.getSettleType()));
        }
        /** 版本号 */
        if (!ObjectUtils.isEmpty(query.getVersion())) {
            list.add(qdo.version.eq(query.getVersion()));
        }

        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalEqvaIncomeVO queryByKey(Long id) {
        JPAQuery<CalEqvaIncomeVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalEqvaIncomeVO> queryListDynamic(CalEqvaIncomeQuery query) {
        JPAQuery<CalEqvaIncomeVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalEqvaIncomeVO> queryPaging(CalEqvaIncomeQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalEqvaIncomeVO> jpaQuery = getJpaQueryWhere(query);
        List<CalEqvaIncomeVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalEqvaIncomeVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalEqvaIncomeDO save(CalEqvaIncomeDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalEqvaIncomeDO> saveAll(List<CalEqvaIncomeDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalEqvaIncomePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 核算年度
        if (payload.getFinYear() != null) {
            update.set(qdo.finYear, payload.getFinYear());
        }
        // 核算期间id
        if (payload.getFinPeriod() != null) {
            update.set(qdo.finPeriod, payload.getFinPeriod());
        }
        // 工种
        if (payload.getJobType1() != null) {
            update.set(qdo.jobType1, payload.getJobType1());
        }
        // 工种子类
        if (payload.getJobType2() != null) {
            update.set(qdo.jobType2, payload.getJobType2());
        }
        // 合作方式
        if (payload.getCoopType() != null) {
            update.set(qdo.coopType, payload.getCoopType());
        }
        // 城市级别
        if (payload.getCityLevel() != null) {
            update.set(qdo.cityLevel, payload.getCityLevel());
        }
        // 单位当量收入
        if (payload.getPreeqvaAmt() != null) {
            update.set(qdo.preeqvaAmt, payload.getPreeqvaAmt());
        }
        // 状态
        if (payload.getLineStatus() != null) {
            update.set(qdo.lineStatus, payload.getLineStatus());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 项目名称
        if (payload.getProjName() != null) {
            update.set(qdo.projName, payload.getProjName());
        }
        // 资源id
        if (payload.getResId() != null) {
            update.set(qdo.resId, payload.getResId());
        }

        // 起始时间
        if (payload.getStartDate() != null) {
            update.set(qdo.startDate, payload.getStartDate());
        }
        // 失效时间
        if (payload.getEndDate() != null) {
            update.set(qdo.endDate, payload.getEndDate());
        }
        // 备注
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 版本号
        if (payload.getVersion()!=null) {
            update.set(qdo.version, payload.getVersion());
        }
        // 结算方式
        if (payload.getSettleType()!=null) {
            update.set(qdo.settleType, payload.getSettleType());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 核算年度
            if (nullFields.contains("finYear")) {
                update.setNull(qdo.finYear);
            }
            // 核算期间id
            if (nullFields.contains("finPeriod")) {
                update.setNull(qdo.finPeriod);
            }
            // 工种
            if (nullFields.contains("jobType1")) {
                update.setNull(qdo.jobType1);
            }
            // 工种子类
            if (nullFields.contains("jobType2")) {
                update.setNull(qdo.jobType2);
            }
            // 合作方式
            if (nullFields.contains("coopType")) {
                update.setNull(qdo.coopType);
            }
            // 城市级别
            if (nullFields.contains("cityLevel")) {
                update.setNull(qdo.cityLevel);
            }
            // 单位当量收入
            if (nullFields.contains("preeqvaAmt")) {
                update.setNull(qdo.preeqvaAmt);
            }
            // 状态
            if (nullFields.contains("lineStatus")) {
                update.setNull(qdo.lineStatus);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 项目id
            if (nullFields.contains("projName")) {
                update.setNull(qdo.projName);
            }
            // 资源id
            if (nullFields.contains("resId")) {
                update.setNull(qdo.resId);
            }
            // 起始时间
            if (nullFields.contains("startDate")) {
                update.setNull(qdo.startDate);
            }
            // 起始时间
            if (nullFields.contains("endDate")) {
                update.setNull(qdo.endDate);
            }
            // 备注
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
            // 结算方式
            if (nullFields.contains("settleType")) {
                update.setNull(qdo.settleType);
            }

        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate checkDataWhere(CalEqvaIncomeQuery query) {
        List<Predicate> list = new ArrayList<>();


        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        } else {
            list.add(qdo.jobType1.isNull());
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        } else {
            list.add(qdo.jobType2.isNull());
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        } else {
            list.add(qdo.finYear.isNull());
        }
        /** 核算期间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        } else {
            list.add(qdo.finPeriod.isNull());
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        } else {
            list.add(qdo.projId.isNull());
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        } else {
            list.add(qdo.resId.isNull());
        }
        /** 合作方式 精确 */
        if (!ObjectUtils.isEmpty(query.getCoopType())) {
            list.add(qdo.coopType.eq(query.getCoopType()));
        } else {
            list.add(qdo.coopType.isNull());
        }
        /** 城市级别 精确 */
        if (!ObjectUtils.isEmpty(query.getCityLevel())) {
            list.add(qdo.cityLevel.eq(query.getCityLevel()));
        } else {
            list.add(qdo.cityLevel.isNull());
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<Long> getCheckDataWhere(CalEqvaIncomeQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdo.id).from(qdo);
        // 条件封装
        jpaQuery.where(checkDataWhere(query));
        return jpaQuery;
    }

    /**
     * 校验用
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<Long> queryCheckData(CalEqvaIncomeQuery query) {
        JPAQuery<Long> jpaQuery = getCheckDataWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate eqvaIncomePlusWhere(CalEqvaIncomeQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 城市级别 精确 */
        if (!ObjectUtils.isEmpty(query.getCityLevel())) {
            list.add(qdo.cityLevel.eq(query.getCityLevel()).or(qdo.jobType1.isNull()));
        } else {
            list.add(qdo.cityLevel.isNull());
        }

        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()).or(qdo.jobType1.isNull()));
        } else {
            list.add(qdo.jobType1.isNull());
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()).or(qdo.jobType2.isNull()));
        } else {
            list.add(qdo.jobType2.isNull());
        }
//        /** 核算年度 精确 */
//        if (!ObjectUtils.isEmpty(query.getFinYear())) {
//            list.add(qdo.finYear.eq(query.getFinYear()).or(qdo.finYear.isNull()));
//        } else {
//            list.add(qdo.finYear.isNull());
//        }
//        /** 核算期间 精确 */
//        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
//            list.add(qdo.finPeriod.eq(query.getFinPeriod()).or(qdo.finPeriod.isNull()));
//        } else {
//            list.add(qdo.finPeriod.isNull());
//        }
        /** 合作方式 精确 */
        if (!ObjectUtils.isEmpty(query.getCoopType())) {
            list.add(qdo.coopType.eq(query.getCoopType()).or(qdo.coopType.isNull()));
        } else {
            list.add(qdo.coopType.isNull());
        }
        /** buId 精确 */
        if (!ObjectUtils.isEmpty(query.getBuId())) {
            list.add(qdo.buId.eq(query.getBuId()).or(qdo.buId.isNull()));
        } else {
            list.add(qdo.buId.isNull());
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()).or(qdo.projId.isNull()));
        } else {
            list.add(qdo.projId.isNull());
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()).or(qdo.resId.isNull()));
        } else {
            list.add(qdo.resId.isNull());
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalEqvaIncomeVO> getEqvaIncomePlusWhere(CalEqvaIncomeQuery query) {
        JPAQuery<CalEqvaIncomeVO> jpaQuery = jpaQueryFactory.select(Projections.bean(CalEqvaIncomeVO.class,
                qdo.id,
                // 工种
                qdo.jobType1,
                // 工种子类
                qdo.jobType2,
                // 核算年度
                qdo.finYear,
                // 核算期间id
                qdo.finPeriod,
                // 单位当量收入
                qdo.preeqvaAmt
        )).from(qdo);
        // 条件封装
        jpaQuery.where(eqvaIncomePlusWhere(query));
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.desc("resId"));
        orderse.add(OrderItem.desc("projId"));
        orderse.add(OrderItem.desc("buId"));
        orderse.add(OrderItem.desc("coopType"));
        orderse.add(OrderItem.desc("jobType2"));
        orderse.add(OrderItem.desc("jobType1"));
        orderse.add(OrderItem.desc("finPeriod"));
        orderse.add(OrderItem.desc("finYear"));
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));

        return jpaQuery;
    }

    /**
     * 获取最精准数据匹配
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public CalEqvaIncomeVO getEqvaIncomePlus(CalEqvaIncomeQuery query) {
        JPAQuery<CalEqvaIncomeVO> jpaQuery = getEqvaIncomePlusWhere(query);
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据资源id更新状态
     *
     * @param resId 资源id
     * @return
     */
    public long updateExterEqvaIncomeStatus(Long resId) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.lineStatus, "0")
                .where(qdo.resType.eq("0"))
                .where(qdo.resId.eq(resId));
        return update.execute();
    }
}

