package com.elitesland.tw.tw5.server.prd.cal.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.param.OrderItem;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalSettlePricePayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalSettlePriceQuery;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalSettlePriceVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalSettlePriceDO;
import com.elitesland.tw.tw5.server.prd.cal.entity.QCalSettlePriceDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalSettlePriceRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 当量结算定价管理
 *
 * @author carl
 * @date 2023-11-07
 */
@Repository
@RequiredArgsConstructor
public class CalSettlePriceDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final CalSettlePriceRepo repo;
    private final QCalSettlePriceDO qdo = QCalSettlePriceDO.calSettlePriceDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<CalSettlePriceVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(CalSettlePriceVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // from bu结算类型码
                qdo.fromBuDealtype,
                // to bu结算类型码
                qdo.toBuDealtype,
                // from bu_id
                qdo.fromBuId,
                // to bu_id
                qdo.toBuId,
                // 工种
                qdo.jobType1,
                // 工种子类
                qdo.jobType2,
                // 核算年度
                qdo.finYear,
                // 核算期间id
                qdo.finPeriod,

                // 项目id
                qdo.projId,
                qdo.projName,
                // 资源id
                qdo.resId,
                // 定价策略（mark up百分比，mark up金额，绝对结算金额）
                qdo.priceStrategy,
                // 调整价格（mark up百分比，mark up金额，绝对结算金额）
                qdo.markupPrice
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalSettlePriceVO> getJpaQueryWhere(CalSettlePriceQuery query) {
        JPAQuery<CalSettlePriceVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(CalSettlePriceQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(CalSettlePriceQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 主键 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** from bu结算类型码 精确 */
        if (!ObjectUtils.isEmpty(query.getFromBuDealtype())) {
            list.add(qdo.fromBuDealtype.eq(query.getFromBuDealtype()));
        }
        /** to bu结算类型码 精确 */
        if (!ObjectUtils.isEmpty(query.getToBuDealtype())) {
            list.add(qdo.toBuDealtype.eq(query.getToBuDealtype()));
        }
        /** from bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getFromBuId())) {
            list.add(qdo.fromBuId.eq(query.getFromBuId()));
        }
        /** to bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getToBuId())) {
            list.add(qdo.toBuId.eq(query.getToBuId()));
        }
        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        }
        /** 核算期间id 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        }
        /** 定价策略（mark up百分比，mark up金额，绝对结算金额） 精确 */
        if (!ObjectUtils.isEmpty(query.getPriceStrategy())) {
            list.add(qdo.priceStrategy.eq(query.getPriceStrategy()));
        }
        /** 调整价格（mark up百分比，mark up金额，绝对结算金额） 精确 */
        if (!ObjectUtils.isEmpty(query.getMarkupPrice())) {
            list.add(qdo.markupPrice.eq(query.getMarkupPrice()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public CalSettlePriceVO queryByKey(Long id) {
        JPAQuery<CalSettlePriceVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<CalSettlePriceVO> queryListDynamic(CalSettlePriceQuery query) {
        JPAQuery<CalSettlePriceVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<CalSettlePriceVO> queryPaging(CalSettlePriceQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<CalSettlePriceVO> jpaQuery = getJpaQueryWhere(query);
        List<CalSettlePriceVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<CalSettlePriceVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public CalSettlePriceDO save(CalSettlePriceDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<CalSettlePriceDO> saveAll(List<CalSettlePriceDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(CalSettlePricePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 主键
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // from bu结算类型码
        if (payload.getFromBuDealtype() != null) {
            update.set(qdo.fromBuDealtype, payload.getFromBuDealtype());
        }
        // to bu结算类型码
        if (payload.getToBuDealtype() != null) {
            update.set(qdo.toBuDealtype, payload.getToBuDealtype());
        }
        // from bu_id
        if (payload.getFromBuId() != null) {
            update.set(qdo.fromBuId, payload.getFromBuId());
        }
        // to bu_id
        if (payload.getToBuId() != null) {
            update.set(qdo.toBuId, payload.getToBuId());
        }
        // 工种
        if (payload.getJobType1() != null) {
            update.set(qdo.jobType1, payload.getJobType1());
        }
        // 工种子类
        if (payload.getJobType2() != null) {
            update.set(qdo.jobType2, payload.getJobType2());
        }
        // 核算年度
        if (payload.getFinYear() != null) {
            update.set(qdo.finYear, payload.getFinYear());
        }
        // 核算期间id
        if (payload.getFinPeriod() != null) {
            update.set(qdo.finPeriod, payload.getFinPeriod());
        }
        // 项目id
        if (payload.getProjId() != null) {
            update.set(qdo.projId, payload.getProjId());
        }
        // 资源id
        if (payload.getResId() != null) {
            update.set(qdo.resId, payload.getResId());
        }
        // 定价策略（mark up百分比，mark up金额，绝对结算金额）
        if (payload.getPriceStrategy() != null) {
            update.set(qdo.priceStrategy, payload.getPriceStrategy());
        }
        // 调整价格（mark up百分比，mark up金额，绝对结算金额）
        if (payload.getMarkupPrice() != null) {
            update.set(qdo.markupPrice, payload.getMarkupPrice());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 主键
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // from bu结算类型码
            if (nullFields.contains("fromBuDealtype")) {
                update.setNull(qdo.fromBuDealtype);
            }
            // to bu结算类型码
            if (nullFields.contains("toBuDealtype")) {
                update.setNull(qdo.toBuDealtype);
            }
            // from bu_id
            if (nullFields.contains("fromBuId")) {
                update.setNull(qdo.fromBuId);
            }
            // to bu_id
            if (nullFields.contains("toBuId")) {
                update.setNull(qdo.toBuId);
            }
            // 工种
            if (nullFields.contains("jobType1")) {
                update.setNull(qdo.jobType1);
            }
            // 工种子类
            if (nullFields.contains("jobType2")) {
                update.setNull(qdo.jobType2);
            }
            // 核算年度
            if (nullFields.contains("finYear")) {
                update.setNull(qdo.finYear);
            }
            // 核算期间id
            if (nullFields.contains("finPeriod")) {
                update.setNull(qdo.finPeriod);
            }
            // 项目id
            if (nullFields.contains("projId")) {
                update.setNull(qdo.projId);
            }
            // 资源id
            if (nullFields.contains("resId")) {
                update.setNull(qdo.resId);
            }
            // 定价策略（mark up百分比，mark up金额，绝对结算金额）
            if (nullFields.contains("priceStrategy")) {
                update.setNull(qdo.priceStrategy);
            }
            // 调整价格（mark up百分比，mark up金额，绝对结算金额）
            if (nullFields.contains("markupPrice")) {
                update.setNull(qdo.markupPrice);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate checkDataWhere(CalSettlePriceQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** 平台编号 精确 */
        if (!ObjectUtils.isEmpty(query.getFromBuDealtype())) {
            list.add(qdo.fromBuDealtype.eq(query.getFromBuDealtype()));
        } else {
            list.add(qdo.fromBuDealtype.isNull());
        }
        /** to bu结算类型码 精确 */
        if (!ObjectUtils.isEmpty(query.getToBuDealtype())) {
            list.add(qdo.toBuDealtype.eq(query.getToBuDealtype()));
        } else {
            list.add(qdo.toBuDealtype.isNull());
        }

        /** from bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getFromBuId())) {
            list.add(qdo.fromBuId.eq(query.getFromBuId()));
        } else {
            list.add(qdo.fromBuId.isNull());
        }
        /** to bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getToBuId())) {
            list.add(qdo.toBuId.eq(query.getToBuId()));
        } else {
            list.add(qdo.toBuId.isNull());
        }
        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()));
        } else {
            list.add(qdo.jobType1.isNull());
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()));
        } else {
            list.add(qdo.jobType2.isNull());
        }
        /** 核算年度 精确 */
        if (!ObjectUtils.isEmpty(query.getFinYear())) {
            list.add(qdo.finYear.eq(query.getFinYear()));
        } else {
            list.add(qdo.finYear.isNull());
        }
        /** 核算期间 精确 */
        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
            list.add(qdo.finPeriod.eq(query.getFinPeriod()));
        } else {
            list.add(qdo.finPeriod.isNull());
        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()));
        } else {
            list.add(qdo.projId.isNull());
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()));
        } else {
            list.add(qdo.resId.isNull());
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<Long> getCheckDataWhere(CalSettlePriceQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory.select(qdo.id).from(qdo);
        // 条件封装
        jpaQuery.where(checkDataWhere(query));
        return jpaQuery;
    }

    /**
     * 校验用
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<Long> queryCheckData(CalSettlePriceQuery query) {
        JPAQuery<Long> jpaQuery = getCheckDataWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate settlePricePlusWhere(CalSettlePriceQuery query) {
        List<Predicate> list = new ArrayList<>();

        /** from bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getFromBuId())) {
            list.add(qdo.fromBuId.eq(query.getFromBuId()).or(qdo.fromBuId.isNull()));
        } else {
            list.add(qdo.fromBuId.isNull());
        }
        /** to bu_id 精确 */
        if (!ObjectUtils.isEmpty(query.getToBuId())) {
            list.add(qdo.toBuId.eq(query.getToBuId()).or(qdo.toBuId.isNull()));
        } else {
            list.add(qdo.toBuId.isNull());
        }
        /** 工种 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType1())) {
            list.add(qdo.jobType1.eq(query.getJobType1()).or(qdo.jobType1.isNull()));
        } else {
            list.add(qdo.jobType1.isNull());
        }
        /** 工种子类 精确 */
        if (!ObjectUtils.isEmpty(query.getJobType2())) {
            list.add(qdo.jobType2.eq(query.getJobType2()).or(qdo.jobType2.isNull()));
        } else {
            list.add(qdo.jobType2.isNull());
        }
//        /** 核算年度 精确 */
//        if (!ObjectUtils.isEmpty(query.getFinYear())) {
//            list.add(qdo.finYear.eq(query.getFinYear()).or(qdo.finYear.isNull()));
//        } else {
//            list.add(qdo.finYear.isNull());
//        }
//        /** 核算期间 精确 */
//        if (!ObjectUtils.isEmpty(query.getFinPeriod())) {
//            list.add(qdo.finPeriod.eq(query.getFinPeriod()).or(qdo.finPeriod.isNull()));
//        } else {
//            list.add(qdo.finPeriod.isNull());
//        }
        /** 项目id 精确 */
        if (!ObjectUtils.isEmpty(query.getProjId())) {
            list.add(qdo.projId.eq(query.getProjId()).or(qdo.projId.isNull()));
        } else {
            list.add(qdo.projId.isNull());
        }
        /** 资源id 精确 */
        if (!ObjectUtils.isEmpty(query.getResId())) {
            list.add(qdo.resId.eq(query.getResId()).or(qdo.resId.isNull()));
        } else {
            list.add(qdo.resId.isNull());
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<CalSettlePriceVO> getSettlePricePlusWhere(CalSettlePriceQuery query) {
        JPAQuery<CalSettlePriceVO> jpaQuery = jpaQueryFactory.select(Projections.bean(CalSettlePriceVO.class,
                qdo.id,
                // 工种
                qdo.jobType1,
                // 工种子类
                qdo.jobType2,
                // 核算年度
                qdo.finYear,
                // 核算期间id
                qdo.finPeriod,
                // 定价策略（mark up百分比，mark up金额，绝对结算金额）
                qdo.priceStrategy,
                // 调整价格（mark up百分比，mark up金额，绝对结算金额）
                qdo.markupPrice
        )).from(qdo);
        // 条件封装
        jpaQuery.where(settlePricePlusWhere(query));
        List<OrderItem> orderse = new ArrayList<>();
        orderse.add(OrderItem.desc("resId"));
        orderse.add(OrderItem.desc("projId"));
        orderse.add(OrderItem.desc("toBuId"));
        orderse.add(OrderItem.desc("fromBuId"));
        orderse.add(OrderItem.desc("jobType2"));
        orderse.add(OrderItem.desc("jobType1"));
        orderse.add(OrderItem.desc("finPeriod"));
        orderse.add(OrderItem.desc("finYear"));
        orderse.add(OrderItem.desc("toBuDealtype"));
        orderse.add(OrderItem.desc("fromBuDealtype"));
        orderse.add(OrderItem.desc("createTime"));
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, orderse));

        return jpaQuery;
    }

    /**
     * 获取最精准数据匹配
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public CalSettlePriceVO getSettlePricePlus(CalSettlePriceQuery query) {
        JPAQuery<CalSettlePriceVO> jpaQuery = getSettlePricePlusWhere(query);
        return jpaQuery.fetchFirst();
    }
}

