package com.elitesland.tw.tw5.server.prd.my.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.RewardLeaveConfigPayload;
import com.elitesland.tw.tw5.api.prd.my.query.RewardLeaveConfigQuery;
import com.elitesland.tw.tw5.api.prd.my.vo.RewardLeaveConfigVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QRewardLeaveConfigDO;
import com.elitesland.tw.tw5.server.prd.my.entity.RewardLeaveConfigDO;
import com.elitesland.tw.tw5.server.prd.my.repo.RewardLeaveConfigRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

@Repository
@RequiredArgsConstructor
public class RewardLeaveConfigDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final RewardLeaveConfigRepo repo;
    private final QRewardLeaveConfigDO qdo = QRewardLeaveConfigDO.rewardLeaveConfigDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<RewardLeaveConfigVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(RewardLeaveConfigVO.class,
                qdo.id,
                //qdo.remark,
                //qdo.createUserId,
                //qdo.creator,
                //qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 年度
                qdo.leaveYear,
                // 交付BU
                qdo.deliBuId,
                // 项目
                qdo.projectId,
                // 职级范围
                qdo.jobGradeList,
                // 管理职级范围
                qdo.jobGrade2List,
                // 上限天数
                qdo.leaveDaysUp
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<RewardLeaveConfigVO> getJpaQueryWhere(RewardLeaveConfigQuery query) {
        JPAQuery<RewardLeaveConfigVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(RewardLeaveConfigQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(RewardLeaveConfigQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 年度 精确 */
        if (!ObjectUtils.isEmpty(query.getLeaveYear())) {
            list.add(qdo.leaveYear.eq(query.getLeaveYear()));
        }
        /** 交付BU 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliBuId())) {
            list.add(qdo.deliBuId.eq(query.getDeliBuId()));
        }
        /** 项目 精确 */
        if (!ObjectUtils.isEmpty(query.getProjectId())) {
            list.add(qdo.projectId.eq(query.getProjectId()));
        }
        /** 职级范围 精确 */
        if (!ObjectUtils.isEmpty(query.getJobGradeList())) {
            list.add(qdo.jobGradeList.eq(query.getJobGradeList()));
        }
        /** 上限天数 精确 */
        if (!ObjectUtils.isEmpty(query.getLeaveDaysUp())) {
            list.add(qdo.leaveDaysUp.eq(query.getLeaveDaysUp()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public RewardLeaveConfigVO queryByKey(Long id) {
        JPAQuery<RewardLeaveConfigVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<RewardLeaveConfigVO> queryListDynamic(RewardLeaveConfigQuery query) {
        JPAQuery<RewardLeaveConfigVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<RewardLeaveConfigVO> queryPaging(RewardLeaveConfigQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<RewardLeaveConfigVO> jpaQuery = getJpaQueryWhere(query);
        List<RewardLeaveConfigVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<RewardLeaveConfigVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public RewardLeaveConfigDO save(RewardLeaveConfigDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<RewardLeaveConfigDO> saveAll(List<RewardLeaveConfigDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(RewardLeaveConfigPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 年度
        if (payload.getLeaveYear() != null) {
            update.set(qdo.leaveYear, payload.getLeaveYear());
        }
        // 交付BU
        if (payload.getDeliBuId() != null) {
            update.set(qdo.deliBuId, payload.getDeliBuId());
        }
        // 项目
        if (payload.getProjectId() != null) {
            update.set(qdo.projectId, payload.getProjectId());
        }
        // 职级范围
        if (payload.getJobGradeList() != null) {
            update.set(qdo.jobGradeList, payload.getJobGradeList());
        }
        // 上限天数
        if (payload.getLeaveDaysUp() != null) {
            update.set(qdo.leaveDaysUp, payload.getLeaveDaysUp());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 年度
            if (nullFields.contains("leaveYear")) {
                update.setNull(qdo.leaveYear);
            }
            // 交付BU
            if (nullFields.contains("deliBuId")) {
                update.setNull(qdo.deliBuId);
            }
            // 项目
            if (nullFields.contains("projectId")) {
                update.setNull(qdo.projectId);
            }
            // 职级范围
            if (nullFields.contains("jobGradeList")) {
                update.setNull(qdo.jobGradeList);
            }
            // 上限天数
            if (nullFields.contains("leaveDaysUp")) {
                update.setNull(qdo.leaveDaysUp);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    public List<RewardLeaveConfigVO> queryBuConfig(RewardLeaveConfigQuery query) {
        JPAQuery<RewardLeaveConfigVO> jpaQuery = getJpaQuerySelect();
        return jpaQuery.where(qdo.projectId.isNull())
                .where(qdo.deliBuId.eq(query.getDeliBuId()))
                .where(qdo.deleteFlag.eq(0))
                .where(qdo.leaveYear.eq(query.getLeaveYear()))
                .fetch();
    }
}
