package com.elitesland.tw.tw5.server.prd.system.dao;

import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemRemindPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemRemindQuery;
import com.elitesland.tw.tw5.api.prd.system.vo.PrdSystemRemindVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemRemindDO;
import com.elitesland.tw.tw5.server.prd.system.entity.QPrdSystemRemindDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemRemindRepo;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPADeleteClause;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.util.ObjectUtils;

import java.util.List;

/**
 * 消息提醒dao
 *
 * @author wangding
 */
@Repository
@RequiredArgsConstructor
public class PrdSystemRemindDAO {
    private final JPAQueryFactory jpaQueryFactory;
    private final PrdSystemRemindRepo repo;
    private final QPrdSystemRemindDO qdo = QPrdSystemRemindDO.prdSystemRemindDO;

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PrdSystemRemindDO save(PrdSystemRemindDO ado) {
        return repo.save(ado);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemRemindVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PrdSystemRemindVO.class,
                qdo.id,
                qdo.remindCode,
                qdo.remindContent,
                qdo.remindContentLocale,
                qdo.remindType,
                qdo.portalFlag,
                qdo.createUserId,
                qdo.createTime
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PrdSystemRemindVO> getJpaQueryWhere(PrdSystemRemindQuery query) {
        JPAQuery<PrdSystemRemindVO> jpaQuery = getJpaQuerySelect();

        if (!ObjectUtils.isEmpty(query.getRemindCode())) {
            jpaQuery.where(qdo.remindCode.eq(query.getRemindCode()));
        }
        if (!ObjectUtils.isEmpty(query.getRemindContent())) {
            jpaQuery.where(qdo.remindContent.like(SqlUtil.toSqlLikeString(query.getRemindContent())));
        }
        if (!ObjectUtils.isEmpty(query.getRemindContentLocale())) {
            jpaQuery.where(qdo.remindContentLocale.eq(query.getRemindContentLocale()));
        }
        if (!ObjectUtils.isEmpty(query.getRemindType())) {
            jpaQuery.where(qdo.remindType.eq(query.getRemindType()));
        }
        if (query.getPortalFlag() != null) {
            jpaQuery.where(qdo.portalFlag.eq(query.getPortalFlag()));
        }
        if (!ObjectUtils.isEmpty(query.getId())) {
            jpaQuery.where(qdo.id.ne(query.getId()));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PrdSystemRemindVO> queryListDynamic(PrdSystemRemindQuery query) {
        JPAQuery<PrdSystemRemindVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    public long updateByKeyDynamic(PrdSystemRemindPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));

        if (payload.getRemindCode() != null) {
            update.set(qdo.remindCode, payload.getRemindCode());
        }
        if (payload.getRemindContent() != null) {
            update.set(qdo.remindContent, payload.getRemindContent());
        }
        if (payload.getRemindContentLocale() != null) {
            update.set(qdo.remindContentLocale, payload.getRemindContentLocale());
        }
        if (payload.getRemindType() != null) {
            update.set(qdo.remindType, payload.getRemindType());
        }
        if (payload.getPortalFlag() != null) {
            update.set(qdo.portalFlag, payload.getPortalFlag());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {

            if (nullFields.contains("remindContent")) {
                update.setNull(qdo.remindContent);
            }
            if (nullFields.contains("remindContentLocale")) {
                update.setNull(qdo.remindContentLocale);
            }
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
        }
        // 执行修改
        return update.execute();
    }

    /**
     * 物理删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long delete(List<Long> keys) {

        JPADeleteClause delete = jpaQueryFactory.delete(qdo).where(qdo.id.in(keys));

        return delete.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主键
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        return update.execute();
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PrdSystemRemindVO queryByKey(Long id) {
        JPAQuery<PrdSystemRemindVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PrdSystemRemindVO> queryPaging(PrdSystemRemindQuery query) {
        JPAQuery<PrdSystemRemindVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<PrdSystemRemindVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        return PagingVO.<PrdSystemRemindVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }
}
