package com.elitesland.tw.tw5.server.prd.system.service;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.core.annotation.TenantTransaction;
import com.elitescloud.cloudt.core.annotation.common.TenantIsolateType;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.elitescloud.cloudt.system.vo.SysUserDTO;
import com.elitesland.tw.tw5.api.prd.system.payload.PrdSystemBusinessObjectPayload;
import com.elitesland.tw.tw5.api.prd.system.query.PrdSystemBusinessObjectQuery;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemBusinessObjectService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemPermissionFieldService;
import com.elitesland.tw.tw5.api.prd.system.service.PrdSystemPermissionFunctionObjectService;
import com.elitesland.tw.tw5.api.prd.system.vo.*;
import com.elitesland.tw.tw5.server.common.QueryHelp;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.util.PageUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.system.convert.PrdSystemBusinessObjectConvert;
import com.elitesland.tw.tw5.server.prd.system.dao.*;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemBusinessObjectDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemBusinessObjectMenuDO;
import com.elitesland.tw.tw5.server.prd.system.entity.PrdSystemNewFunctionDO;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemBusinessObjectRepo;
import com.elitesland.tw.tw5.server.prd.system.repo.PrdSystemNewFunctionRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.Page;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 业务对象impl
 *
 * @Author Bill
 * @Date 2023/9/15 11:29
 **/
@Service
@RequiredArgsConstructor
@Slf4j
public class PrdSystemBusinessObjectServiceImpl implements PrdSystemBusinessObjectService {

    private final PrdSystemBusinessObjectRepo repo;
    private final PrdSystemBusinessObjectDAO dao;
    private final PrdSystemRoleDAO roleDAO;
    private final PrdSystemNewFunctionDAO newFunctionDAO;

    private final PrdSystemNewFunctionRepo newFunctionRepo;

    private final PrdSystemPermissionFunctionObjectDAO functionObjectDAO;

    private final PrdSystemPermissionFieldDAO fieldDAO;

    private final PrdSystemPermissionTableFieldsDAO tableFieldsDAO;

    /**
     * 业务对象新增
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdSystemBusinessObjectVO insert(PrdSystemBusinessObjectPayload payload) {
        //校验
        check(payload);
        //对象转换
        PrdSystemBusinessObjectDO entityDo = PrdSystemBusinessObjectConvert.INSTANCE.toDo(payload);
        repo.save(entityDo);
        return PrdSystemBusinessObjectConvert.INSTANCE.toVO(entityDo);
    }

    /**
     * 业务对象更新
     * @param payload
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public PrdSystemBusinessObjectVO update(PrdSystemBusinessObjectPayload payload) {
        //校验
        checkUpdate(payload);
        //查询对应的数据是否存在
        PrdSystemBusinessObjectDO entity = repo.findById(payload.getId()).orElseGet(PrdSystemBusinessObjectDO::new);
        Assert.notNull(entity.getId(), "不存在");
        //校验编号是不是重复
        if (StringUtils.hasText(payload.getObjectCode()) && !entity.getObjectCode().equals(payload.getObjectCode())){
            Assert.isNull(dao.queryByObjectCode(payload.getObjectCode()),"objectCode不能重复");
        }
        PrdSystemBusinessObjectDO entityDo = PrdSystemBusinessObjectConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        PrdSystemBusinessObjectDO save = repo.save(entity);
        //菜单列表不为null说明要该改变
        if (payload.getMenuIds() != null){
            if (payload.getMenuIds().size() > 0){
                List<PrdSystemBusinessObjectMenuDO> businessObjectMenuDOS = new ArrayList<>();
                for (Long menuId : payload.getMenuIds()) {
                    PrdSystemBusinessObjectMenuDO businessObjectMenuDO = new PrdSystemBusinessObjectMenuDO();
                    businessObjectMenuDO.setMenuId(menuId);
                    businessObjectMenuDO.setObjectId(payload.getId());
                    businessObjectMenuDOS.add(businessObjectMenuDO);
                }
                //全删
                dao.deleteBusinessObjectMenusByObjectId(payload.getId());
                //全插
                dao.saveBusinessObjectMenuAll(businessObjectMenuDOS);
            }else{
                //全删
                dao.deleteBusinessObjectMenusByObjectId(payload.getId());
            }
        }
        PrdSystemBusinessObjectVO businessObjectVO = PrdSystemBusinessObjectConvert.INSTANCE.toVO(save);
        businessObjectVO.setMenuIds(payload.getMenuIds());
        return businessObjectVO;
    }

    /**
     * 校验 修改参数
     * @param payload
     */
    private void checkUpdate(PrdSystemBusinessObjectPayload payload) {
        if (payload.getId() == null){
            throw TwException.error("", "id不能为空");
        }
    }

    /**
     * 业务对象分页查询
     * @param query
     * @return
     */
    @Override
    public PagingVO<PrdSystemBusinessObjectVO> paging(PrdSystemBusinessObjectQuery query) {
        return dao.queryByKeyword(query);
    }

    /**
     * 业务对象 逻辑删除
     * @param keys
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!CollectionUtils.isEmpty(keys)){
            dao.deleteSoft(keys);
        }
    }

    /**
     * 主键查询
     * @param id
     * @return
     */
    @Override
    public PrdSystemBusinessObjectVO get(Long id) {
        PrdSystemBusinessObjectDO entity = repo.findById(id).orElseGet(PrdSystemBusinessObjectDO::new);
        Assert.notNull(entity.getId(), "不存在");
        PrdSystemBusinessObjectVO prdSystemBusinessObjectVO = PrdSystemBusinessObjectConvert.INSTANCE.toVO(entity);
        //填充业务对象菜单
        prdSystemBusinessObjectVO.setMenuIds(dao.queryBusinessObjectMenuIds(id));
        //填充业务对象功能
        return prdSystemBusinessObjectVO;
    }

    /**
     * 业务对象入参校验
     *
     * @param payload
     */
    private void check(PrdSystemBusinessObjectPayload payload) {
        //校验规则
        if (!StringUtils.hasText(payload.getObjectCode())){
            throw TwException.error("", "objectCode不能为空");
        }
        if (StringUtils.hasText(payload.getObjectCode())){
            //查此编号
            Assert.isNull(dao.queryByObjectCode(payload.getObjectCode()),"objectCode已存在");
        }
        if (!StringUtils.hasText(payload.getObjectName())){
            throw TwException.error("", "objectName不能为空");
        }
        if (payload.getObjectStatus() == null){
            throw TwException.error("", "objectStatus不能为空");
        }
        if (!StringUtils.hasText(payload.getObjectVersion())){
            throw TwException.error("", "objectVersion不能为空");
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void autoCreate(Map<String, PrdSystemBusinessObjectPayload> map) {
        List<String> objectCodes = new ArrayList<>(map.keySet());

        List<PrdSystemBusinessObjectVO> objectList = dao.queryByObjectCodes(objectCodes);
        List<PrdSystemBusinessObjectVO> deleteObjectList = dao.queryByNotObjectCodes(objectCodes);
        Map<String, PrdSystemBusinessObjectVO> objectMap = objectList.stream().collect(Collectors.toMap(PrdSystemBusinessObjectVO::getObjectCode, e -> e, (l, r) -> l));

        List<PrdSystemBusinessObjectDO> insertList = new ArrayList<>();
        List<PrdSystemBusinessObjectDO> updateList = new ArrayList<>();

        map.forEach((k, v) -> {
            if (objectMap.containsKey(k)) {
                PrdSystemBusinessObjectVO prdSystemBusinessObjectVO = objectMap.get(k);
                PrdSystemBusinessObjectDO prdSystemBusinessObjectDO = PrdSystemBusinessObjectConvert.INSTANCE.toEntity(prdSystemBusinessObjectVO);
                prdSystemBusinessObjectDO.setObjectName(v.getObjectName());

                updateList.add(prdSystemBusinessObjectDO);
            } else {
                PrdSystemBusinessObjectDO entityDo = PrdSystemBusinessObjectConvert.INSTANCE.toDo(v);
                entityDo.setObjectStatus(0);
                entityDo.setObjectVersion("1.0");

                insertList.add(entityDo);
            }
        });

        dao.deleteByCodeNotIn(objectCodes);

        List<Long> deleteFucIds = new ArrayList<>();
        List<Long> deleteFucObjIds = new ArrayList<>();
        List<Long> deleteFieldIds = new ArrayList<>();
        List<Long> deleteTableFieldIds = new ArrayList<>();

        // 如果删除对象不为空 处理下关联信息
        if (!CollectionUtils.isEmpty(deleteObjectList)) {

            // 查询所有功能
            List<PrdSystemNewFunctionDO> functionDOS = newFunctionRepo.findAll();

            for (PrdSystemBusinessObjectVO objectVO : deleteObjectList) {
                for (PrdSystemNewFunctionDO functionDO : functionDOS) {
                    if (functionDO.getObjectId().equals(objectVO.getId())) {
                        deleteFucIds.add(functionDO.getId());
                    }
                }
            }

            // 拿到同步表 删除数据
            List<Long> ids = tableFieldsDAO.queryListByTableIds(deleteFucIds);
            deleteTableFieldIds.addAll(ids);

            // 查询所有功能对象
            if (!CollectionUtils.isEmpty(functionDOS)) {
                List<PrdSystemPermissionFunctionObjectVO> functionObjectVOS = functionObjectDAO.queryByFunctionId(deleteFucIds);
                if (!CollectionUtils.isEmpty(functionObjectVOS)) {
                    for (PrdSystemPermissionFunctionObjectVO functionObjectVO : functionObjectVOS) {
                        deleteFucObjIds.add(functionObjectVO.getId());
                    }
                }
            }

            // 查询功能对象关联的字段
            if (!CollectionUtils.isEmpty(deleteFucObjIds)) {
                List<PrdSystemPermissionFieldVO> fieldVOS = fieldDAO.queryByFunctionObjectIds(new HashSet<>(deleteFucObjIds));
                for (PrdSystemPermissionFieldVO fieldVO : fieldVOS) {
                    deleteFieldIds.add(fieldVO.getId());
                }
            }

            newFunctionDAO.deleteByIds(deleteFucIds);
            functionObjectDAO.deleteByIds(deleteFucObjIds);
            fieldDAO.deleteByIds(deleteFieldIds);
            tableFieldsDAO.deleteSoftCopy(deleteTableFieldIds);

        }

        if(!CollectionUtils.isEmpty(insertList)) {
            repo.saveAll(insertList);
        }
        if(!CollectionUtils.isEmpty(updateList)) {
            repo.saveAll(updateList);
        }
    }

    /**
     * 业务对象附带 功能列表
     * @return
     */
    @Override
    public List<PrdSystemBusinessObjectVO> listFunction() {
        List<PrdSystemBusinessObjectDO> all = repo.findAll();
        List<PrdSystemBusinessObjectVO> result = new ArrayList<>();
        for (PrdSystemBusinessObjectDO prdSystemBusinessObjectDO : all) {
            PrdSystemBusinessObjectVO entityVO = PrdSystemBusinessObjectConvert.INSTANCE.toVO(prdSystemBusinessObjectDO);
            entityVO.setFunctionList(dao.queryFunctionByObjectId(entityVO.getId()));
            result.add(entityVO);
        }
        return result;
    }

    /**
     * 业务对象的 对应 角色下配置信息获取
     * @param query
     * @return
     */
    @TenantTransaction(isolateType = TenantIsolateType.DEFAULT)
    @Override
    public PrdSystemBusinessObjectVO listObjectConfig(PrdSystemBusinessObjectQuery query) {
        //检查查询参数
        checkListObjectConfig(query);
        //查询业务对象是否
        PrdSystemBusinessObjectVO businessObjectVO = dao.queryByObjectCode(query.getObjectCode());
        Assert.notNull(businessObjectVO, "不存在");
        //定义一个 Set集合 避免不必要的判断
        Set<Long> set = new HashSet<>();
        //拿到用户信息 需要用到角色信息
        SysUserDTO user = GlobalUtil.getLoginGeneralUser().getUser();
        //取用户所有角色 配置并集  首先获取角色不重复的功能
        Set<Long> roleIds = user.getRoleIds();
        if (CollectionUtils.isEmpty(roleIds)) {
            //如果没有获取到角色，则我们重新查询下
            roleIds = new HashSet<>(dao.getRoleIdsByUserId(user.getId()));
        }
        //查询并集的功能
        List<PrdSystemNewFunctionVO> functionVOS = dao.getDistinctFunctionByRoleIds(roleIds);
        businessObjectVO.setFunctionList(functionVOS);
        //查询关联的功能对象
        List<PrdSystemPermissionFieldObjRoleFunctionVO> functionObjectVOS = roleDAO.queryFunctionObjectByFunctionIds(functionVOS.stream().map(PrdSystemNewFunctionVO::getId).collect(Collectors.toList()));
        //查询关联的字段规则
        List<PrdSystemPermissionFieldObjRoleFunctionVO> fieldRuleVOS = dao.getDistinctFieldRuleByRoleIds(roleIds);
        //拼装字段数据
        for (PrdSystemPermissionFieldObjRoleFunctionVO functionObjectVO : functionObjectVOS) {
            //判断是否为该功能对象下
            for (PrdSystemPermissionFieldObjRoleFunctionVO fieldRuleVO : fieldRuleVOS) {
                //如果该字段已经处理过不需要再处理
                if (!set.contains(fieldRuleVO.getFieldId())) {
                    //是否为该功能对象下
                    if (functionObjectVO.getFunctionObjectId().equals(fieldRuleVO.getFunctionObjectId())) {
                        List<PrdSystemPermissionFieldObjRoleFunctionVO> field = functionObjectVO.getField();
                        if (CollectionUtils.isEmpty(field)) {
                            field = new ArrayList<>();
                            functionObjectVO.setField(field);
                        }
                        field.add(fieldRuleVO);
                        set.add(fieldRuleVO.getFieldId());
                    }else {
                        continue;
                    }
                }
            }
        }
        businessObjectVO.setFieldRuleList(functionObjectVOS);
        return businessObjectVO;
    }

    /**
     * 功能对应的 角色下的配置信息 获取
     * @param query
     * @return
     */
    @Override
    public PrdSystemBusinessObjectVO listFunctionConfig(PrdSystemBusinessObjectQuery query) {
        //检查参数
        checkListFunctionConfig(query);

        //检查object是否存在
        PrdSystemBusinessObjectVO businessObjectVO = dao.queryByObjectCode(query.getObjectCode());
        Assert.notNull(businessObjectVO.getId(), "不存在");
        //检查function是否存在
        PrdSystemNewFunctionVO newFunctionVO = newFunctionDAO.queryByFunctionCode(query.getFunctionCode());
        Assert.notNull(newFunctionVO.getId(), "不存在");

        //定义一个 Set集合 避免不必要的判断
        Set<Long> set = new HashSet<>();
        //拿到用户信息 需要用到角色信息
        SysUserDTO user = GlobalUtil.getLoginGeneralUser().getUser();
        //取用户所有角色 配置并集  首先获取角色不重复的功能
        Set<Long> roleIds = user.getRoleIds();
        if (CollectionUtils.isEmpty(roleIds)) {
            //如果没有获取到角色，则我们重新查询下
            roleIds = new HashSet<>(dao.getRoleIdsByUserId(user.getId()));
        }

        //拿到该用户 角色下 拥有的该部分的功能
        List<PrdSystemNewFunctionVO> newFunctionVOS = dao.getDistinctFunctionByRoleIds(roleIds);
        //对该部分的功能进行过滤
        List<PrdSystemNewFunctionVO> newFunctionByRole = newFunctionVOS.stream().filter(function -> query.getFunctionCode().equals(function.getParentFunctionCode()) || query.getFunctionCode().equals(function.getFunctionCode())).toList();
        //校验该用户的角色下查询到的功能，是否包含前端传过来的功能编码，没有，则不设置字段，直接返回
        Boolean flag = checkContainsFunction(query, newFunctionVOS);
        if(!flag) {
            businessObjectVO.setFunctionList(newFunctionByRole);
            businessObjectVO.setFieldRuleList(new ArrayList<>());
            return businessObjectVO;
        }

        //拿到该用户 角色下 拥有的该部分功能的字段
        List<PrdSystemPermissionFieldObjRoleFunctionVO> fieldRuleVOS = dao.getDistinctFieldRuleByRoleIds(roleIds);
        //查询关联的功能对象
        List<PrdSystemPermissionFieldObjRoleFunctionVO> functionObjectVOS = roleDAO.queryFunctionObjectByFunctionIds(List.of(newFunctionVO.getId()));
        //拼装字段数据
        for (PrdSystemPermissionFieldObjRoleFunctionVO functionObjectVO : functionObjectVOS) {
            //判断是否为该功能对象下
            for (PrdSystemPermissionFieldObjRoleFunctionVO fieldRuleVO : fieldRuleVOS) {
                //如果该字段已经处理过不需要再处理
                if (!set.contains(fieldRuleVO.getFieldId())) {
                    //是否为该功能对象下
                    if (functionObjectVO.getFunctionObjectId().equals(fieldRuleVO.getFunctionObjectId())) {
                        List<PrdSystemPermissionFieldObjRoleFunctionVO> field = functionObjectVO.getField();
                        if (CollectionUtils.isEmpty(field)) {
                            field = new ArrayList<>();
                            functionObjectVO.setField(field);
                        }
                        field.add(fieldRuleVO);
                        set.add(fieldRuleVO.getFieldId());
                    }else {
                        continue;
                    }
                }
            }
        }
        //组装数据
        businessObjectVO.setFieldRuleList(functionObjectVOS);
        businessObjectVO.setFunctionList(newFunctionByRole);
        return businessObjectVO;
    }

    /**
     * 带拥有权限规则的功能
     * @param query
     * @return
     */
    @Override
    public List<PrdSystemBusinessObjectVO> listFunction(PrdSystemBusinessObjectQuery query) {
        //业务对象查询
        List<PrdSystemBusinessObjectVO> records = dao.findAll();
        //拿到拥有功能对象的功能（功能对象指）
        List<PrdSystemNewFunctionVO> functions = dao.queryFunctionByHaveFunctionObject();
        Set<Long> functionSet = new HashSet<>();
        //给业务对象放入我们有权限规则的功能
        for (PrdSystemBusinessObjectVO record : records) {
            //如果发现功能分配完了就结束
            if (functionSet.size() == functions.size()) break;
            for (PrdSystemNewFunctionVO function : functions) {
                //如果功能已经分配过不用进行下一步
                if (record.getId().equals(function.getObjectId()) && !functionSet.contains(function.getId())) {
                    List<PrdSystemNewFunctionVO> functionList = record.getFunctionList();
                    if (CollectionUtils.isEmpty(functionList)) {
                        functionList = new ArrayList<>();
                    }
                    functionList.add(function);
                    record.setFunctionList(functionList);
                    functionSet.add(function.getId());
                }
            }
        }
        return records;
    }

    /**
     * listFunctionConfig 对象配置 参数检查
     * @param query
     */
    private void checkListFunctionConfig(PrdSystemBusinessObjectQuery query) {
        if (!StringUtils.hasText(query.getFunctionCode())) {
            throw TwException.error("", "functionCode不能为空");
        }
        if (!StringUtils.hasText(query.getObjectCode())) {
            throw TwException.error("", "objectCode不能为空");
        }
    }

    /**
     * listObjectConfig 对象配置 参数检查
     * @param query
     */
    private void checkListObjectConfig(PrdSystemBusinessObjectQuery query) {
        if (!StringUtils.hasText(query.getObjectCode())) {
            throw TwException.error("", "objectCode不能为空");
        }
    }


    /**
     * 校验用户已有的功能中，是否包含前端传递过来的功能
     * @param query
     * @param newFunctionVOS
     * @return
     */
    private Boolean checkContainsFunction(PrdSystemBusinessObjectQuery query, List<PrdSystemNewFunctionVO> newFunctionVOS) {
        Boolean flag = false;
        for (PrdSystemNewFunctionVO vo : newFunctionVOS) {
            if(query.getFunctionCode().equals(vo.getFunctionCode())){
                flag = true;
            }
        }
        return flag;
    }
}
