package com.elitesland.tw.tw5.server.prd.borrow.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.borrow.payload.BorrowMoneyPayload;
import com.elitesland.tw.tw5.api.prd.borrow.payload.BorrowWriteOffPayload;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowMoneyQuery;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowWriteOffQuery;
import com.elitesland.tw.tw5.api.prd.borrow.query.TransferApplyQuery;
import com.elitesland.tw.tw5.api.prd.borrow.service.BorrowMoneyService;
import com.elitesland.tw.tw5.api.prd.borrow.service.BorrowWriteOffService;
import com.elitesland.tw.tw5.api.prd.borrow.service.TransferApplyService;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowMoneyVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowWriteOffVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.TransferApplyVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationRefVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationVO;
import com.elitesland.tw.tw5.server.common.GenerateSeqNumConstants;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.prd.acc.common.functionEnum.AccReimDocTypeEnum;
import com.elitesland.tw.tw5.server.prd.borrow.constant.BorrowStatusEnum;
import com.elitesland.tw.tw5.server.prd.borrow.constant.BorrowWriteOffTypeEnum;
import com.elitesland.tw.tw5.server.prd.borrow.convert.BorrowMoneyConvert;
import com.elitesland.tw.tw5.server.prd.borrow.dao.BorrowMoneyDAO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.BorrowMoneyDO;
import com.elitesland.tw.tw5.server.prd.borrow.repo.BorrowMoneyRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.FileUtil;
import com.elitesland.tw.tw5.server.prd.common.GlobalUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.tw.tw5.server.prd.common.functionEnum.OrgEnum;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 借款管理
 *
 * @author wangly
 * @date 2024-07-15
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class BorrowMoneyServiceImpl extends BaseServiceImpl implements BorrowMoneyService {

    private final BorrowMoneyRepo borrowMoneyRepo;
    private final BorrowMoneyDAO borrowMoneyDAO;
    private final WorkflowUtil workflowUtil;
    private final BorrowWriteOffService borrowWriteOffService;
    private final TransferApplyService transferApplyService;
    private final CacheUtil cacheUtil;
    private final TransactionUtilService transactionUtilService;
    private final FileUtil fileUtil;

    @Override
    public PagingVO<BorrowMoneyVO> queryPaging(BorrowMoneyQuery query){
        PagingVO<BorrowMoneyVO> borrowMoneyVOPagingVO = borrowMoneyDAO.queryPaging(query);
        transferList(borrowMoneyVOPagingVO.getRecords());
        return borrowMoneyVOPagingVO;
    }

    @Override
    public List<BorrowMoneyVO> queryListDynamic(BorrowMoneyQuery query){
        return borrowMoneyDAO.queryListDynamic(query);
    }

    @Override
    public BorrowMoneyVO queryByKey(Long key) {
        BorrowMoneyVO borrowMoneyVO = borrowMoneyDAO.queryByKey(key);
        if (borrowMoneyVO != null) {
            transfer(Collections.singletonList(borrowMoneyVO));
        }
        return borrowMoneyVO;
    }

    @Override
    public BorrowMoneyVO insert(BorrowMoneyPayload payload) {
        // 保存数据
        BorrowMoneyDO save = transactionUtilService.executeWithCallable(() -> borrowMoneyRepo.save(this.processData(payload)));
        // 发起流程
        BorrowMoneyVO borrowMoneyVO = BorrowMoneyConvert.INSTANCE.toVo(save);
        activeBorrowMoneyProc(borrowMoneyVO);
        return borrowMoneyVO;
    }

    /**
     * 处理数据
     * @param payload payload
     * @return
     */
    private BorrowMoneyDO processData(BorrowMoneyPayload payload) {
        BorrowMoneyDO entityDo = BorrowMoneyConvert.INSTANCE.toDo(payload);
        entityDo.setRepaymentUserId(entityDo.getApplicantUserId());
        entityDo.setBorrowNo(generateSeqNum(GenerateSeqNumConstants.BORROW_MONEY_NO));
        entityDo.setAlreadyWriteOffAmt(BigDecimal.ZERO);
        entityDo.setUnwriteOffAmt(entityDo.getBorrowAmt());
        if (entityDo.getBorrowName() == null) {
            // 借款名称：申请人+申请日期+借款金额+业务类型
            String userName = cacheUtil.getUserName(entityDo.getApplicantUserId());
            String applyDate = entityDo.getApplicantTime().format(DateTimeFormatter.ofPattern("yyyy-MM-dd"));
            String businessTypeDesc = cacheUtil.transferSystemSelection("BORROW_MONEY:BUSINESS_TYPE", entityDo.getBusinessType());
            BigDecimal borrowAmt = entityDo.getBorrowAmt() == null ? BigDecimal.ZERO : entityDo.getBorrowAmt();
            entityDo.setBorrowName(userName + "-" + applyDate + "-" + borrowAmt + "-" + businessTypeDesc);
        }
        return entityDo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public BorrowMoneyVO update(BorrowMoneyPayload payload) {
        BorrowMoneyDO entity = borrowMoneyRepo.findById(payload.getId()).orElseGet(BorrowMoneyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        BorrowMoneyDO entityDo = BorrowMoneyConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        return BorrowMoneyConvert.INSTANCE.toVo(borrowMoneyRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(BorrowMoneyPayload payload) {
        BorrowMoneyDO entity = borrowMoneyRepo.findById(payload.getId()).orElseGet(BorrowMoneyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        return borrowMoneyDAO.updateByKeyDynamic(payload);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            borrowMoneyDAO.deleteSoft(keys);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void updateBorrowStatus(List<Long> keys,String status){
       if (!keys.isEmpty()) {
            borrowMoneyDAO.updateBorrowStatus(keys,status);
        }
    }
    // 激活流程
    private void activeBorrowMoneyProc(BorrowMoneyVO borrowMoneyVO){
        ProcessInfo processInfo;
        HashMap<String, Object> variables = new HashMap<>();
        // 借款金额
        variables.put("borrowAmt", borrowMoneyVO.getBorrowAmt());
        // 申请人所在部门
        PrdOrgOrganizationVO org = cacheUtil.getOrg(borrowMoneyVO.getApplicantBuId());
        variables.put("buLevel", org.getExtString1());
        // 三级部门
        if (OrgEnum.BuLevel.BU_LEVEL3.getCode().equals(org.getExtString1())) {
            borrowMoneyVO.setLevel3BuManagerId(org.getManageId());
            // 二级部门负责人
            List<PrdOrgOrganizationRefVO> allFatherOrg = cacheUtil.getAllFatherOrgByOrgId(org.getId());
            Optional<PrdOrgOrganizationRefVO> secondOrgRef = allFatherOrg.stream().filter((vo) -> StringUtils.hasText(vo.getBuLevel()) && vo.getBuLevel().equals(OrgEnum.BuLevel.BU_LEVEL2.getCode()) && vo.getOrgStatus().equalsIgnoreCase(OrgEnum.OrgStatus.ACTIVE.getCode())).findFirst();
            secondOrgRef.ifPresent(prdOrgOrganizationRefVO -> borrowMoneyVO.setLevel2BuManagerId(prdOrgOrganizationRefVO.getManageId()));
            // 一级部门负责人
            PrdOrgOrganizationRefVO buLevel1ByOrgId = cacheUtil.getBULevel1ByOrgId(org.getId());
            borrowMoneyVO.setLevel1BuManagerId(buLevel1ByOrgId.getManageId());
            // 二级部门
        }else if (OrgEnum.BuLevel.BU_LEVEL2.getCode().equals(org.getExtString1())) {
            borrowMoneyVO.setLevel2BuManagerId(org.getManageId());
            PrdOrgOrganizationRefVO buLevel1ByOrgId = cacheUtil.getBULevel1ByOrgId(org.getId());
            borrowMoneyVO.setLevel1BuManagerId(buLevel1ByOrgId.getManageId());
            // 一级部门
        }else if (OrgEnum.BuLevel.BU_LEVEL1.getCode().equals(org.getExtString1())) {
            borrowMoneyVO.setLevel1BuManagerId(org.getManageId());
        }
        String procInstName = borrowMoneyVO.getBorrowName() + "-借款流程";
        // 发起流程
        processInfo = workflowUtil.simpleStartProcess(StartProcessPayload.of(
                ProcDefKey.DIB_BORROW_MONEY.name(),
                procInstName,
                borrowMoneyVO.getId() + "",
                variables), borrowMoneyVO);
        if (ProcInstStatus.APPROVING.getDesc().equals(processInfo.getProcInstStatus().getDesc())) {
            BorrowMoneyPayload payload = new BorrowMoneyPayload();
            payload.setId(borrowMoneyVO.getId());
            payload.setApprStatus(BorrowStatusEnum.APPROVING.getCode());
            borrowMoneyDAO.updateByKeyDynamic(payload);
        }

    }

    private void transfer(List<BorrowMoneyVO> borrowMoneyVOList) {

        if (!CollectionUtils.isEmpty(borrowMoneyVOList)) {
            List<Long> idList = borrowMoneyVOList.stream().map(BorrowMoneyVO::getId).collect(Collectors.toList());
            // 查询对应的出让申请数据
            TransferApplyQuery transferApplyQuery = new TransferApplyQuery();
            transferApplyQuery.setBorrowIdList(idList);
            List<TransferApplyVO> transferApplyVOS = transferApplyService.queryListDynamic(transferApplyQuery);
            Map<Long, List<TransferApplyVO>> transferApplyMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(transferApplyVOS)) {
                transferApplyMap = transferApplyVOS.stream().collect(Collectors.groupingBy(TransferApplyVO::getBorrowId));
            }
            // 查询核销的数据
            BorrowWriteOffQuery borrowWriteOffQuery = new BorrowWriteOffQuery();
            borrowWriteOffQuery.setBorrowIdList(idList);
            List<BorrowWriteOffVO> borrowWriteOffVOS = borrowWriteOffService.queryListDynamic(borrowWriteOffQuery);
            Map<Long, List<BorrowWriteOffVO>> borrowWriteOffMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(borrowWriteOffVOS)) {
                borrowWriteOffMap = borrowWriteOffVOS.stream().collect(Collectors.groupingBy(BorrowWriteOffVO::getBorrowId));
            }
            for (BorrowMoneyVO borrowMoneyVO : borrowMoneyVOList) {
                if (transferApplyMap.containsKey(borrowMoneyVO.getId())) {
                    borrowMoneyVO.setTransferApplyList(transferApplyMap.get(borrowMoneyVO.getId()));
                }
                if (borrowWriteOffMap.containsKey(borrowMoneyVO.getId())) {
                    borrowMoneyVO.setBorrowWriteOffList(borrowWriteOffMap.get(borrowMoneyVO.getId()));
                }
                // 处理附件
                if (StringUtils.hasText(borrowMoneyVO.getFileCodes())) {
                    borrowMoneyVO.setFileData(fileUtil.getFileDatas(borrowMoneyVO.getFileCodes()));
                }
                // 出让标志
                if (borrowMoneyVO.getTransferFlag() != null && borrowMoneyVO.getTransferFlag()) {
                    borrowMoneyVO.setApprStatus(BorrowStatusEnum.TRANSFER.getCode());
                }
            }
        }
    }

    /**
     * 处理列表数据
     *
     * @param borrowMoneyVOS vos
     */
    private void transferList(List<BorrowMoneyVO> borrowMoneyVOS) {
        borrowMoneyVOS.forEach(borrowMoneyVO -> {
            if (borrowMoneyVO.getTransferFlag() != null && borrowMoneyVO.getTransferFlag()) {
                borrowMoneyVO.setApprStatus(BorrowStatusEnum.TRANSFER.getCode());
            }
        });
    }

    // 释放借款数据
    @Override
    @Transactional
    public void relateBorrowData(Long reimId){
        List<BorrowMoneyVO> borrowMoneyVOList = borrowWriteOffService.queryByReimId(reimId);
        if(!CollectionUtils.isEmpty(borrowMoneyVOList)){
            List<Long> borrowIdList = borrowMoneyVOList.stream().map(vo -> vo.getId()).collect(Collectors.toList());
            BorrowMoneyQuery borrowMoneyQuery = new BorrowMoneyQuery();
            borrowMoneyQuery.setIds(borrowIdList);
            List<BorrowMoneyVO> borrowMoneyVOS = queryListDynamic(borrowMoneyQuery);
            borrowMoneyVOS.stream().forEach(borrowMoneyVO -> {
                BorrowMoneyPayload borrowMoneyPayload = new BorrowMoneyPayload();
                borrowMoneyPayload.setId(borrowMoneyVO.getId());
                // 当已核销金额为空 或者为0 的时候 状态是已付款 否则就是部分核销
                if(borrowMoneyVO.getAlreadyWriteOffAmt()== null || borrowMoneyVO.getAlreadyWriteOffAmt().compareTo(BigDecimal.ZERO)==0){
                    borrowMoneyPayload.setApprStatus(BorrowStatusEnum.PAID.getCode());
                }else{
                    borrowMoneyPayload.setApprStatus(BorrowStatusEnum.WRITE_OFF_PART.getCode());
                }
                updateByKeyDynamic(borrowMoneyPayload);
            });
        }
        // 删除核销的数据
        borrowWriteOffService.deleteByReimId(reimId);
    }

    @Override
    @Transactional
    public void cashWriteOff(Long key, BigDecimal writeOffAmt) {
        BorrowMoneyVO borrowMoneyVO = this.queryByKey(key);
        if (borrowMoneyVO == null || !(BorrowStatusEnum.PAID.getCode().equals(borrowMoneyVO.getApprStatus())
                || BorrowStatusEnum.WRITE_OFF_PART.getCode().equals(borrowMoneyVO.getApprStatus()))) {
            throw TwException.error("", "只有已付款/部分核销的单据才能进行核销！");
        }
        // 借款金额
        BigDecimal borrowAmt = borrowMoneyVO.getBorrowAmt() == null ? BigDecimal.ZERO : borrowMoneyVO.getBorrowAmt();
        // 已核销金额
        BigDecimal alreadyWriteOffAmt = borrowMoneyVO.getAlreadyWriteOffAmt() == null ? BigDecimal.ZERO : borrowMoneyVO.getAlreadyWriteOffAmt();
        // 本次核销金额
        writeOffAmt = writeOffAmt == null ? BigDecimal.ZERO : writeOffAmt;
        BorrowMoneyPayload borrowMoneyPayload = new BorrowMoneyPayload();
        borrowMoneyPayload.setId(borrowMoneyVO.getId());
        borrowMoneyPayload.setAlreadyWriteOffAmt(alreadyWriteOffAmt.add(writeOffAmt));
        borrowMoneyPayload.setUnwriteOffAmt(borrowAmt.subtract(alreadyWriteOffAmt.add(writeOffAmt)));
        int i = alreadyWriteOffAmt.add(writeOffAmt).compareTo(borrowAmt);
        if (i > 0) {
            throw TwException.error("", "核销金额已达上限，请重新输入!");
        }else if (i == 0) {
            borrowMoneyPayload.setApprStatus(BorrowStatusEnum.WRITTEN_OFF.getCode());
        }else {
            borrowMoneyPayload.setApprStatus(BorrowStatusEnum.WRITE_OFF_PART.getCode());
        }
        // 更新借款数据
        updateByKeyDynamic(borrowMoneyPayload);
        // 新增核销数据
        Long loginUserId = GlobalUtil.getLoginUserId();
        BorrowWriteOffPayload borrowWriteOffPayload = new BorrowWriteOffPayload();
        borrowWriteOffPayload.setBorrowId(borrowMoneyPayload.getId());
        borrowWriteOffPayload.setWriteOffAmt(writeOffAmt);
        borrowWriteOffPayload.setWriteOffDate(LocalDate.now());
        borrowWriteOffPayload.setWriteOffType(BorrowWriteOffTypeEnum.CASH.getCode());
        borrowWriteOffPayload.setWriteOffUserId(loginUserId);
        borrowWriteOffService.insert(borrowWriteOffPayload);
    }
}
