package com.elitesland.tw.tw5.server.prd.my.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.my.payload.UserVacationPayload;
import com.elitesland.tw.tw5.api.prd.my.query.UserVacationQuery;
import com.elitesland.tw.tw5.api.prd.my.vo.UserVacationVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.my.entity.QUserVacationDO;
import com.elitesland.tw.tw5.server.prd.my.entity.UserVacationDO;
import com.elitesland.tw.tw5.server.prd.my.repo.UserVacationRepo;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgEmployeeDO;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgEmployeeRefDO;
import com.elitesland.tw.tw5.server.prd.org.entity.QPrdOrgOrganizationDO;
import com.querydsl.core.QueryResults;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.math.BigDecimal;
import java.util.List;

@Repository
@RequiredArgsConstructor
public class UserVacationDAO {

    private final UserVacationRepo userVacationRepo;

    private static final QUserVacationDO userVacationDO = QUserVacationDO.userVacationDO;

    private static final QPrdOrgEmployeeDO employeeDO = QPrdOrgEmployeeDO.prdOrgEmployeeDO;
    private final QPrdOrgEmployeeRefDO qdoRef = QPrdOrgEmployeeRefDO.prdOrgEmployeeRefDO;

    private final QPrdOrgOrganizationDO qdoOrg = QPrdOrgOrganizationDO.prdOrgOrganizationDO;


    private final JPAQueryFactory jpaQueryFactory;

    public void save(UserVacationDO userVacationDO) {
        userVacationRepo.save(userVacationDO);
    }

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<UserVacationVO> getJpaQuerySelect() {
        JPAQuery<UserVacationVO> jpaQuery = jpaQueryFactory.select(Projections.bean(UserVacationVO.class,
                        userVacationDO.id,
                        userVacationDO.userId,
                        userVacationDO.vacationType,
                        userVacationDO.vacationYear,
                        userVacationDO.usedDays,
                        userVacationDO.startDate,
                        userVacationDO.endDate,
                        userVacationDO.expirationDate,
                        userVacationDO.totalDays,
                        userVacationDO.totalDays.subtract(userVacationDO.usedDays).as("availableDays"),
                        userVacationDO.remark,
                        employeeDO.employeeName,
                        employeeDO.employeeNo,
                        employeeDO.extString5,
                        qdoRef.orgId
                )).from(userVacationDO)
                .leftJoin(employeeDO).on(employeeDO.userId.eq(userVacationDO.userId))
                .leftJoin(qdoRef).on(userVacationDO.userId.longValue().eq(qdoRef.userId.longValue()).and(qdoRef.isDefault.eq(0)).and(qdoRef.isCopy.eq(0)).and(qdoRef.deleteFlag.eq(0)))
                .leftJoin(qdoOrg).on(qdoOrg.id.longValue().eq(qdoRef.orgId.longValue()).and(qdoOrg.isCopy.eq(0)));

        return jpaQuery;
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<UserVacationVO> getJpaQueryWhere(UserVacationQuery query) {
        JPAQuery<UserVacationVO> jpaQuery = getJpaQuerySelect();
        if (!ObjectUtils.isEmpty(query.getIds())) {
            jpaQuery.where(userVacationDO.id.in(query.getIds()));
        }
        if (!ObjectUtils.isEmpty(query.getVacationYear())) {
            jpaQuery.where(userVacationDO.vacationYear.eq(query.getVacationYear()));
        }
        if (!ObjectUtils.isEmpty(query.getVacationType())) {
            jpaQuery.where(userVacationDO.vacationType.eq(query.getVacationType()));
        }
        if (!ObjectUtils.isEmpty(query.getOrgId())) {
            jpaQuery.where(qdoRef.orgId.eq(query.getOrgId()));
        }
        if (!ObjectUtils.isEmpty(query.getUserId())) {
            jpaQuery.where(userVacationDO.userId.eq(query.getUserId()));
        }
        if (!ObjectUtils.isEmpty(query.getExtString5())) {
            jpaQuery.where(employeeDO.extString5.eq(query.getExtString5()));
        }

        if (!ObjectUtils.isEmpty(query.getExpirationStartDate())) {
            jpaQuery.where(userVacationDO.expirationDate.loe(query.getExpirationStartDate()));
        }
        if (!ObjectUtils.isEmpty(query.getExpirationEndDate())) {
            jpaQuery.where(userVacationDO.expirationDate.goe(query.getExpirationEndDate()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonType())) {
            jpaQuery.where(userVacationDO.reasonType.eq(query.getReasonType()));
        }
        if (!ObjectUtils.isEmpty(query.getReasonId())) {
            jpaQuery.where(userVacationDO.reasonId.eq(query.getReasonId()));
        }
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, userVacationDO._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(userVacationDO, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<UserVacationVO> queryPaging(UserVacationQuery query) {
        JPAQuery<UserVacationVO> jpaQuery = getJpaQueryWhere(query);
        QueryResults<UserVacationVO> result = jpaQuery.offset(query.getPageRequest().getOffset()).limit(query.getPageRequest().getPageSize()).fetchResults();
        return PagingVO.<UserVacationVO>builder().records(result.getResults()).total(result.getTotal()).build();
    }

    /**
     * 列表查询
     *
     * @param query 查询参数
     * @return 结果
     */
    public List<UserVacationVO> list(UserVacationQuery query) {
        JPAQuery<UserVacationVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    public UserVacationDO queryById(Long id) {
        return userVacationRepo.findById(id).orElseGet(UserVacationDO::new);
    }

    public void deleteByIds(List<Long> ids) {
        jpaQueryFactory.update(userVacationDO)
                .set(userVacationDO.deleteFlag, 1)
                .where(userVacationDO.id.in(ids))
                .execute();
    }

    public void updateExpirationDate(UserVacationPayload payload) {
        jpaQueryFactory.update(userVacationDO)
                .set(userVacationDO.expirationDate, payload.getExpirationDate())
                .where(userVacationDO.id.in(payload.getId()))
                .execute();
    }

    public List<UserVacationVO> getAvailableVacation(Long userId) {
        JPAQuery<UserVacationVO> jpaQuery = getJpaQuerySelect();
        return jpaQuery.where(userVacationDO.userId.eq(userId))
                .fetch();
    }

    @Transactional
    public void updateUsedDays(Long vacationId, BigDecimal days) {
        jpaQueryFactory.update(userVacationDO)
                .set(userVacationDO.usedDays, userVacationDO.usedDays.add(days))
                .where(userVacationDO.id.eq(vacationId))
                .execute();
    }

    public void deleteAllByUserId(Long userId) {
        jpaQueryFactory.update(userVacationDO)
                .set(userVacationDO.deleteFlag, 1)
                .where(userVacationDO.userId.eq(userId))
                .execute();
    }


}
