package com.elitesland.tw.tw5.server.prd.purchase.controller;

import com.alibaba.excel.EasyExcel;
import com.alibaba.excel.write.builder.ExcelWriterSheetBuilder;
import com.alibaba.excel.write.style.column.LongestMatchColumnWidthStyleStrategy;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimPayload;
import com.elitesland.tw.tw5.api.prd.partner.common.vo.BookAccountVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PaymentSlipPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PaymentSlipQuery;
import com.elitesland.tw.tw5.api.prd.purchase.service.PaymentSlipService;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PaymentSlipVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PrintPaySerialNumVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.excel.ExcelUtil;
import com.elitesland.tw.tw5.server.prd.purchase.convert.PaymentSlipConvert;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.elitesland.tw.tw5.server.udc.UdcNameClass;
import com.elitesland.tw.tw5.server.udc.UdcUtil;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URLEncoder;
import java.util.Arrays;
import java.util.List;

/**
 * 付款单记录
 *
 * @author likunpeng
 * @folder 付款申请单-付款单记录
 * @date 2023-11-23
 */
@Api(tags = "付款单记录")
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/crm/paymentSlip")
@Slf4j
public class PaymentSlipController {

    private final PaymentSlipService paymentSlipService;

    private final UdcUtil udcUtil;
    /**
     * 付款单记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单记录
     */
    @GetMapping("/paging")
    @UdcNameClass
    @ApiOperation("付款单记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paging(PaymentSlipQuery query) {
        return TwOutputUtil.ok(paymentSlipService.queryPaging(query));
    }

    /**
     * 付款单提交记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单提交记录
     */
    @GetMapping("/paymentCommitPaging")
    @UdcNameClass
    @ApiOperation("付款单提交记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentCommitPaging(PaymentSlipQuery query) {
        //待应付会计提交 财务经理驳回 出纳驳回至应付会计
        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.READY.getCode(),PurchasePaymentEnum.PaymentSlipStatus.FINANCE_REJECTED.getCode(),PurchasePaymentEnum.PaymentSlipStatus.CASHIER_TO_SUBMIT.getCode()));
        return TwOutputUtil.ok(paymentSlipService.queryPaging(query));
    }

    /**
     * 付款单提交记录-列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单提交记录
     */
    @GetMapping("/paymentCommitList")
    @UdcNameClass
    @ApiOperation("付款单提交记录-列表")
    public TwOutputUtil<List<PaymentSlipVO>> paymentCommitList(PaymentSlipQuery query) {
        //待应付会计提交 财务经理驳回 出纳驳回至应付会计
        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.READY.getCode(),PurchasePaymentEnum.PaymentSlipStatus.FINANCE_REJECTED.getCode(),PurchasePaymentEnum.PaymentSlipStatus.CASHIER_TO_SUBMIT.getCode()));
        return TwOutputUtil.ok(paymentSlipService.queryListDynamic(query));
    }

    /**
     * 付款单批准记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单批准记录
     */
    @GetMapping("/paymentApprovePaging")
    @UdcNameClass
    @ApiOperation("付款单批准记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentApprovePaging(PaymentSlipQuery query) {

        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.FINANCE_MANAGER_APPR.getCode(),PurchasePaymentEnum.PaymentSlipStatus.CASHIER_TO_FINANCE.getCode()));
        return TwOutputUtil.ok(paymentSlipService.queryPaging(query));
    }

    /**
     * 付款单批准分组记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单批准分组记录
     */
    @GetMapping("/paymentApproveGroupPaging")
    @UdcNameClass
    @ApiOperation("付款单批准分组记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentApproveGroupPaging(PaymentSlipQuery query) {
        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.FINANCE_MANAGER_APPR.getCode(),PurchasePaymentEnum.PaymentSlipStatus.CASHIER_TO_FINANCE.getCode()));
        return TwOutputUtil.ok(paymentSlipService.paymentApproveGroupPaging(query));
    }


    /**
     * 付款单CFO记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单批准记录
     */
    @GetMapping("/paymentCfoPaging")
    @UdcNameClass
    @ApiOperation("付款单CFO记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentCfoPaging(PaymentSlipQuery query) {

        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.CFO_APPR.getCode()));
        return TwOutputUtil.ok(paymentSlipService.queryPaging(query));
    }

    /**
     * 付款单批准分组记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单批准分组记录
     */
    @GetMapping("/paymentCfoApproveGroupPaging")
    @UdcNameClass
    @ApiOperation("付款单CFO审批分组记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentCfoApproveGroupPaging(PaymentSlipQuery query) {
        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.CFO_APPR.getCode()));
        return TwOutputUtil.ok(paymentSlipService.paymentApproveGroupPaging(query));
    }
    /**
     * 付款单确认记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单确认记录分页
     */
    @GetMapping("/paymentConfirmPaging")
    @UdcNameClass
    @ApiOperation("付款单确认记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentConfirmPaging(PaymentSlipQuery query) {
        query.setStateList(Arrays.asList(PurchasePaymentEnum.PaymentSlipStatus.CASHIER_APPR.getCode()));
        return TwOutputUtil.ok(paymentSlipService.queryPaging(query));
    }

    /**
     * 付款单付款待记账确认记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单付款待记账确认记录
     */
    @GetMapping("/paymentConfirmAccountPaging")
    @UdcNameClass
    @ApiOperation("付款单付款待记账确认记录-分页")
    public TwOutputUtil<PagingVO<PaymentSlipVO>> paymentConfirmAccountPaging(PaymentSlipQuery query) {
        query.setStateList(Arrays.asList(query.getState()));
        return TwOutputUtil.ok(paymentSlipService.queryPaging(query));
    }


    /**
     * 付款单记录-更新
     *
     * @param payload 单据
     * @return {@link TwOutputUtil}
     * @folder 付款单记录-更新
     */
    @PutMapping
    @UdcNameClass
    @ApiOperation("付款单记录-更新")
    public TwOutputUtil update(@RequestBody PaymentSlipPayload payload) {
        paymentSlipService.update(payload);
        return TwOutputUtil.ok();
    }

    /**
     * 付款单记录-通过驳回
     *
     * @param payloadList 单据集合
     */
    @PostMapping("/passOrReject")
    @UdcNameClass
    @ApiOperation("付款单记录-通过驳回")
    public TwOutputUtil passOrReject(@RequestBody List<PaymentSlipPayload> payloadList) {
        paymentSlipService.passOrReject(payloadList);
        return TwOutputUtil.ok();
    }

    /**
     * 付款单记录-提交
     *
     * @param payloadList 单据集合
     */
    @PostMapping("/paymentCommit")
    @UdcNameClass
    @ApiOperation("付款单记录-提交")
    public TwOutputUtil paymentCommit(@RequestBody List<PaymentSlipPayload> payloadList) {
        paymentSlipService.paymentCommit(payloadList);
        return TwOutputUtil.ok();
    }


    /**
     * 根据申请单Id修改付款申请单记录状态
     *
     * @param paymentApplyId 单据
     * @param state          状态
     * @return {@link TwOutputUtil}
     * @folder 根据申请单Id修改付款申请单记录状态
     */
    @PutMapping("/update")
    @UdcNameClass
    @ApiOperation("根据申请单Id修改付款申请单记录状态")
    public TwOutputUtil updateStatusByPaymentApplyId(Long paymentApplyId, String state) {
        paymentSlipService.updateStatusByPaymentApplyId(paymentApplyId, state);
        return TwOutputUtil.ok();
    }

    /**
     * dealPaymentData
     */
    @GetMapping("/dealPaymentData")
    @UdcNameClass
    @ApiOperation("处理是否需要关闭合同")
    public TwOutputUtil dealPaymentData(Long id) {
        paymentSlipService.dealPaymentData(Arrays.asList(id));
        return TwOutputUtil.ok();
    }
    /**
     * 确认线下付款
     *
     * @param payloadList 单据集合
     * @return {@link TwOutputUtil}
     * @folder 付款单记录-更新
     */
    @PostMapping("/confirmOfflinePayment")
    @UdcNameClass
    @ApiOperation("确认线下付款")
    public TwOutputUtil confirmOfflinePayment(@RequestBody List<PaymentSlipPayload> payloadList) {
        if (CollectionUtils.isEmpty(payloadList)) {
            throw TwException.error("", "请选择数据");
        }
        paymentSlipService.confirmOfflinePayment(payloadList);
        return TwOutputUtil.ok();
    }


    /**
     * 提交网银支付
     *
     * @param payloadList 单据集合
     * @return {@link TwOutputUtil}
     * @folder 付款单记录-更新
     */
    @PostMapping("/commitOnlinePayment")
    @UdcNameClass
    @ApiOperation("提交网银支付")
    public TwOutputUtil commitOnlinePayment(@RequestBody List<PaymentSlipPayload> payloadList) {
        if (CollectionUtils.isEmpty(payloadList)) {
            throw TwException.error("", "请选择数据");
        }
        paymentSlipService.commitOnlinePayment(payloadList);
        return TwOutputUtil.ok();
    }
    /**
     * 生成流水号
     *
     * @return {@link TwOutputUtil}
     * @folder 生成流水号
     */
    @PostMapping("/createPaySerialNum")
    @UdcNameClass
    @ApiOperation("生成流水号")
    public TwOutputUtil createPaySerialNum(@RequestBody List<PaymentSlipPayload> payloadList) {
        paymentSlipService.createPaySerialNum(payloadList);
        return TwOutputUtil.ok();
    }


    /**
     * 付款单提交记录-查询流水号
     *
     * @return {@link TwOutputUtil}
     * @folder 付款单提交记录-查询流水号
     */
    @GetMapping("/queryPaySerialNum")
    @UdcNameClass
    @ApiOperation("付款单提交记录-查询流水号")
    public TwOutputUtil<List<String>> queryPaySerialNum() {
        return TwOutputUtil.ok(paymentSlipService.queryPaySerialNum());
    }

    /**
     * 修改网银用途
     *
     * @return {@link TwOutputUtil}
     * @folder 修改网银用途
     */
    @GetMapping("/updatePayPurpose")
    @UdcNameClass
    @ApiOperation("修改网银用途")
    public TwOutputUtil<String> updatePayPurpose(Long id, String payPurpose,String paymentApplyNo) {
        paymentSlipService.updatePayPurpose(id, payPurpose,paymentApplyNo);
        return TwOutputUtil.ok();
    }


    /**
     * 修改收款账号
     *
     * @return {@link TwOutputUtil}
     * @folder 修改网银用途
     */
    @GetMapping("/updateReceiveAccount")
    @UdcNameClass
    @ApiOperation("修改收款账号")
    public TwOutputUtil<String> updateReceiveAccount(Long id, String receivingBank,String receivingAccount,String paymentApplyNo) {
        paymentSlipService.updateReceiveAccount(id, receivingBank,receivingAccount,paymentApplyNo);
        return TwOutputUtil.ok();
    }


    /**
     * 付款单批准分组记录-分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     * @folder 付款申请单-付款单批准分组记录
     */
    @GetMapping("/excelExport")
    @ApiOperation("付款单记录-导出")
    public void excelExport(PaymentSlipQuery query, HttpServletResponse response) throws IOException {
        List<PaymentSlipVO> vos;
        if (query.getIds() != null && query.getIds().size() > 0) {
            vos = paymentSlipService.queryListByIds(query.getIds());
        } else {
            vos = paymentSlipService.queryListDynamic(query);
        }
        if(!CollectionUtils.isEmpty(vos)){
            vos = udcUtil.translateList(vos);
            //定义文件名称
            String sheetName = "付款记录导出";
            //对文件名进行固定格式编码
            String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
            //设置请求响应内容类型
            //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            //设置请求响应内容编码方式
            response.setCharacterEncoding("utf-8");
            //文件下载，指定默认名
            response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

            final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), PaymentSlipVO.class)
                    .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                    .sheet(sheetName);
            // 列
            ExcelUtil.excelHelper(sheet, PaymentSlipVO.class, null);
            //写入
            sheet.doWrite(vos);
        }
    }

    /**
     * 打印流水号
     *
     * @return {@link TwOutputUtil}
     * @folder 打印流水号
     */
    @GetMapping("/printPaySerialNum")
    @UdcNameClass
    @ApiOperation("打印流水号")
    public void printPaySerialNum(PaymentSlipQuery query, HttpServletResponse response)throws IOException {
        List<PaymentSlipVO> vos;
        if (query.getIds() != null && query.getIds().size() > 0) {
            vos = paymentSlipService.queryListByIds(query.getIds());
        } else {
            vos = paymentSlipService.queryListDynamic(query);
        }
        if(!CollectionUtils.isEmpty(vos)){
            vos = udcUtil.translateList(vos);
            List<PrintPaySerialNumVO> printPaySerialNumVOS = PaymentSlipConvert.INSTANCE.voListVoExcelExport(vos);
            //定义文件名称
            String sheetName = "打印流水号列表";
            //对文件名进行固定格式编码
            String fileName = URLEncoder.encode(sheetName + System.currentTimeMillis() + ".xlsx", "UTF-8");
            //设置请求响应内容类型
            //作用:使客户端浏览器，区分不同种类的数据，并根据不同的MIME调用浏览器内不同的程序嵌入模块来处理相应的数据。
            response.setContentType("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
            //设置请求响应内容编码方式
            response.setCharacterEncoding("utf-8");
            //文件下载，指定默认名
            response.addHeader("Content-Disposition", "attachment;filename=" + fileName);

            final ExcelWriterSheetBuilder sheet = EasyExcel.write(response.getOutputStream(), PrintPaySerialNumVO.class)
                    .registerWriteHandler(new LongestMatchColumnWidthStyleStrategy())
                    .sheet(sheetName);
            // 列
            ExcelUtil.excelHelper(sheet, PrintPaySerialNumVO.class, null);
            //写入
            sheet.doWrite(printPaySerialNumVOS);
        }
    }

    /**
     * 付款单提交记录-查询收款账号信息
     *
     * @return {@link TwOutputUtil}
     * @folder 付款单提交记录-查询流水号
     */
    @GetMapping("/queryReceiveAccountInfo")
    @UdcNameClass
    @ApiOperation("付款单提交记录-查询收款账号信息")
    public TwOutputUtil<BookAccountVO> queryReceiveAccountInfo(Long id) {
        return TwOutputUtil.ok(paymentSlipService.queryReceiveAccountInfo(id));
    }

}

