package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseAgreementDetailsPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseAgreementDetailsQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurConOrAgreementDetailsSimpleVO;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseAgreementDetailsVO;
import com.elitesland.tw.tw5.server.prd.product.entity.QPrdProductDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PurchaseAgreementDetailsDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseAgreementDetailsDO;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PurchaseAgreementDetailsRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 采购协议明细
 *
 * @author Echo
 * @date 2023-06-08
 */
@Repository
@RequiredArgsConstructor
public class PurchaseAgreementDetailsDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PurchaseAgreementDetailsRepo repo;
    private final QPurchaseAgreementDetailsDO qdo = QPurchaseAgreementDetailsDO.purchaseAgreementDetailsDO;
    private final QPrdProductDO qPrdProductDO = QPrdProductDO.prdProductDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementDetailsVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(PurchaseAgreementDetailsVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                //qdo.modifyUserId,
                //qdo.updater,
                //qdo.modifyTime,
                // 关联产品
                qdo.relatedProductId,
                // 采购产品
                qdo.productName,
                // 产品大类
                qdo.classId,
                // 产品小类
                qdo.subClassId,
                // 数量
                qdo.quantity,
                // 含税单价
                qdo.taxPrice,
                // 税率
                qdo.taxRate,
                // 含税总额
                qdo.taxAmt,
                // 不含税总额
                qdo.taxNotAmt,
                // 交期
                qdo.deliveryDate,
                // 说明
                qdo.note
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseAgreementDetailsVO> getJpaQueryWhere(PurchaseAgreementDetailsQuery query) {
        JPAQuery<PurchaseAgreementDetailsVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchaseAgreementDetailsQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchaseAgreementDetailsQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /**
         * 采购协议id
         */
        if (!ObjectUtils.isEmpty(query.getAgreementId())) {
            list.add(qdo.agreementId.eq(query.getAgreementId()));
        }
        /** 关联产品 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedProductId())) {
            list.add(qdo.relatedProductId.eq(query.getRelatedProductId()));
        }
        /** 采购产品 精确 */
        if (!ObjectUtils.isEmpty(query.getProductName())) {
            list.add(qdo.productName.eq(query.getProductName()));
        }
        /** 产品大类 精确 */
        if (!ObjectUtils.isEmpty(query.getClassId())) {
            list.add(qdo.classId.eq(query.getClassId()));
        }
        /** 产品小类 精确 */
        if (!ObjectUtils.isEmpty(query.getSubClassId())) {
            list.add(qdo.subClassId.eq(query.getSubClassId()));
        }
        /** 数量 精确 */
        if (!ObjectUtils.isEmpty(query.getQuantity())) {
            list.add(qdo.quantity.eq(query.getQuantity()));
        }
        /** 含税单价 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxPrice())) {
            list.add(qdo.taxPrice.eq(query.getTaxPrice()));
        }
        /** 税率 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxRate())) {
            list.add(qdo.taxRate.eq(query.getTaxRate()));
        }
        /** 含税总额 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAmt())) {
            list.add(qdo.taxAmt.eq(query.getTaxAmt()));
        }
        /** 不含税总额 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxNotAmt())) {
            list.add(qdo.taxNotAmt.eq(query.getTaxNotAmt()));
        }
        /** 交期 精确 */
        if (!ObjectUtils.isEmpty(query.getDeliveryDate())) {
            list.add(qdo.deliveryDate.eq(query.getDeliveryDate()));
        }
        /** 说明 精确 */
        if (!ObjectUtils.isEmpty(query.getNote())) {
            list.add(qdo.note.eq(query.getNote()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchaseAgreementDetailsVO queryByKey(Long id) {
        JPAQuery<PurchaseAgreementDetailsVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchaseAgreementDetailsVO> queryListDynamic(PurchaseAgreementDetailsQuery query) {
        JPAQuery<PurchaseAgreementDetailsVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchaseAgreementDetailsVO> queryPaging(PurchaseAgreementDetailsQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<PurchaseAgreementDetailsVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchaseAgreementDetailsVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchaseAgreementDetailsVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PurchaseAgreementDetailsDO save(PurchaseAgreementDetailsDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PurchaseAgreementDetailsDO> saveAll(List<PurchaseAgreementDetailsDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchaseAgreementDetailsPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 关联产品
        if (payload.getRelatedProductId() != null) {
            update.set(qdo.relatedProductId, payload.getRelatedProductId());
        }
        // 采购产品
        if (payload.getProductName() != null) {
            update.set(qdo.productName, payload.getProductName());
        }
        // 产品大类
        if (payload.getClassId() != null) {
            update.set(qdo.classId, payload.getClassId());
        }
        // 产品小类
        if (payload.getSubClassId() != null) {
            update.set(qdo.subClassId, payload.getSubClassId());
        }
//        // 采购合同id
//        if (payload.getPurchaseContractId() != null) {
//            update.set(qdo.purchaseContractId, payload.getPurchaseContractId());
//        }
//        // 采购合同编号
//        if (payload.getPurchaseContractNo() != null) {
//            update.set(qdo.purchaseContractNo, payload.getPurchaseContractNo());
//        }
        // 数量
        if (payload.getQuantity() != null) {
            update.set(qdo.quantity, payload.getQuantity());
        }
        // 含税单价
        if (payload.getTaxPrice() != null) {
            update.set(qdo.taxPrice, payload.getTaxPrice());
        }
        // 税率
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        // 含税总额
        if (payload.getTaxAmt() != null) {
            update.set(qdo.taxAmt, payload.getTaxAmt());
        }
        // 不含税总额
        if (payload.getTaxNotAmt() != null) {
            update.set(qdo.taxNotAmt, payload.getTaxNotAmt());
        }
        // 交期
        if (payload.getDeliveryDate() != null) {
            update.set(qdo.deliveryDate, payload.getDeliveryDate());
        }
        // 说明
        if (payload.getNote() != null) {
            update.set(qdo.note, payload.getNote());
        }
        if (payload.getRemark() != null) {
            update.set(qdo.remark, payload.getRemark());
        }
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 关联产品
            if (nullFields.contains("relatedProductId")) {
                update.setNull(qdo.relatedProductId);
            }
            // 采购产品
            if (nullFields.contains("productName")) {
                update.setNull(qdo.productName);
            }
            // 产品大类
            if (nullFields.contains("classId")) {
                update.setNull(qdo.classId);
            }
            // 产品小类
            if (nullFields.contains("subClassId")) {
                update.setNull(qdo.subClassId);
            }
//            // 采购合同id
//            if (nullFields.contains("purchaseContractId")) {
//                update.setNull(qdo.purchaseContractId);
//            }
//            // 采购合同编号
//            if (nullFields.contains("purchaseContractNo")) {
//                update.setNull(qdo.purchaseContractNo);
//            }
            // 数量
            if (nullFields.contains("quantity")) {
                update.setNull(qdo.quantity);
            }
            // 含税单价
            if (nullFields.contains("taxPrice")) {
                update.setNull(qdo.taxPrice);
            }
            // 税率
            if (nullFields.contains("taxRate")) {
                update.setNull(qdo.taxRate);
            }
            // 含税总额
            if (nullFields.contains("taxAmt")) {
                update.setNull(qdo.taxAmt);
            }
            // 不含税总额
            if (nullFields.contains("taxNotAmt")) {
                update.setNull(qdo.taxNotAmt);
            }
            // 交期
            if (nullFields.contains("deliveryDate")) {
                update.setNull(qdo.deliveryDate);
            }
            // 说明
            if (nullFields.contains("note")) {
                update.setNull(qdo.note);
            }
            if (nullFields.contains("remark")) {
                update.setNull(qdo.remark);
            }
            if (nullFields.contains("deleteFlag")) {
                update.setNull(qdo.deleteFlag);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 删除协议明细
     * @param documentIdList 协议IdList
     */
    public void deleteSoftByDocumentIdList(List<Long> documentIdList) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.agreementId.in(documentIdList));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        update.execute();
    }

    /**
     * 根据采购协议Id查询采购明细
     * @param purAgreementId 采购协议ID
     * @return 采购明细
     */
    public List<PurConOrAgreementDetailsSimpleVO> queryByPurAgreementId(Long purAgreementId) {
        JPAQuery<PurConOrAgreementDetailsSimpleVO> jpaQuerySelect = jpaQueryFactory.select(Projections.bean(PurConOrAgreementDetailsSimpleVO.class,
                        qdo.id.as("purConOrAgreementDetailId"),
                        // 关联产品
                        qdo.relatedProductId,
                        qPrdProductDO.prodName.as("relatedProductName"),
                        // 税率
                        qdo.taxRate
                )).from(qdo)
                .leftJoin(qPrdProductDO).on(qdo.relatedProductId.eq(qPrdProductDO.id)
                        .and(qPrdProductDO.deleteFlag.eq(0)))
                .where(qdo.agreementId.eq(purAgreementId));
        return jpaQuerySelect.fetch();
    }
}

