package com.elitesland.tw.tw5.server.prd.purchase.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.purchase.payload.PurchaseContractManagerPayload;
import com.elitesland.tw.tw5.api.prd.purchase.query.PurchaseContractManagerQuery;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseContractDocumentView;
import com.elitesland.tw.tw5.api.prd.purchase.vo.PurchaseContractManagerVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.pms.entity.QPmsProjectDO;
import com.elitesland.tw.tw5.server.prd.product.entity.QPrdProductClassDO;
import com.elitesland.tw.tw5.server.prd.product.entity.QPrdProductDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.PurchaseContractManagerDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPaymentSlipDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseContractDetailsDO;
import com.elitesland.tw.tw5.server.prd.purchase.entity.QPurchaseContractManagerDO;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchaseContractEnum;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchaseContractStatusEnum;
import com.elitesland.tw.tw5.server.prd.purchase.purenum.PurchasePaymentEnum;
import com.elitesland.tw.tw5.server.prd.purchase.repo.PurchaseContractManagerRepo;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConPurchaseDemandDDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QSaleConContractDO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 采购合同管理
 *
 * @author lipengfei
 * @date 2023-05-30
 */
@Repository
@RequiredArgsConstructor
public class PurchaseContractManagerDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final PurchaseContractManagerRepo repo;
    private final QPurchaseContractManagerDO qdo = QPurchaseContractManagerDO.purchaseContractManagerDO;
    private final QSaleConContractDO saleConContractDO = QSaleConContractDO.saleConContractDO;

    private final QPmsProjectDO projectDO = QPmsProjectDO.pmsProjectDO;
    private final QPurchaseContractDetailsDO detailsDO = QPurchaseContractDetailsDO.purchaseContractDetailsDO;
    // private final QPrdAbDO qPrdAbDO = QPrdAbDO.prdAbDO;
    private final QPrdProductDO prdProductDO = QPrdProductDO.prdProductDO;
    private final QPrdProductClassDO classDO = QPrdProductClassDO.prdProductClassDO;
    private final QConPurchaseDemandDDO qConPurchaseDemandDDO = QConPurchaseDemandDDO.conPurchaseDemandDDO;
    private final QPurchaseContractDetailsDO qPurchaseContractDetailsDO = QPurchaseContractDetailsDO.purchaseContractDetailsDO;
    private final QPrdProductClassDO subClassDO = new QPrdProductClassDO("subClassDO");
    private final QPaymentSlipDO qPaymentSlipDO = QPaymentSlipDO.paymentSlipDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseContractManagerVO> getJpaQuerySelect() {
        return jpaQueryFactory.selectDistinct(Projections.bean(PurchaseContractManagerVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 采购合同编号
                qdo.purchaseContractNo,
                // 采购合同名称
                qdo.purchaseContractName,
                // 平台合同类型 udc[salecon:plat_type]
                qdo.platType,
                // 采购合同类型 udc[purchase:con_type]
                qdo.purchaseContractType,
                // 业务类型 udc[purchase:business_type]
                qdo.businessType,
                // 验收方式 udc[purchase:check]
                qdo.acceptanceType,
                // 采购公司BookId
                qdo.purchaseCompanyBookId,
                // 供应商BookID
                qdo.supplierBookId,
                // 采购BU
                qdo.purchaseBuId,
                // 采购负责人
                qdo.purchaseInchargeResId,
                // 签约日期
                qdo.signDate,
                // 申请日期
                qdo.applicationDate,
                // 币种 udc[SYSTEM_BASIC:CURRENCY]
                qdo.currCode,
                // 金额
                qdo.amt,
                // 税率
                qdo.taxRate,
                // 税额
                qdo.taxAmt,
                // 合同状态
                qdo.contractStatus,
                // 审批状态
                qdo.apprStatus,
                // 关联销售合同
                qdo.relatedSalesContract,
                saleConContractDO.name.as("relatedSalesContractName"),
                // 关联协议
                qdo.relatedAgreement,
                // 需求编号
                qdo.demandNo,
                // 比价资料
                qdo.pricecompereFileCodes,
                // 合同附件
                qdo.contractFileCodes,
                // 盖章附件
                qdo.stampFileCodes,
                // 项目ID
                qdo.relatedProjectId,
                projectDO.projName.as("relatedProjectName"),
                // 项目名称
                qdo.relatedProjectName,
                // 任务包
                qdo.relatedTask,
                // 开票方
                qdo.invoice,
                // 付款方式
                qdo.payMethod,
                // 采购来源
                qdo.contractSource,
                // 采购来源编号
                qdo.contractSourceNo,
                // 激活时间
                qdo.activateDate,
                // 终止原因
                qdo.overWhy,
                // 终止时间
                qdo.overDate,
                // 合同归档标志，0：未归档，1：归档
                qdo.filingFlag,
                qdo.closeReason,
                qdo.closeDate,
                // 流程实例ID
                qdo.procInstId,
                // 流程审批状态
                qdo.procInstStatus,
                // 提交时间
                qdo.submitTime,
                // 审批时间
                qdo.approvedTime
        )).from(qdo)
                .leftJoin(detailsDO).on(detailsDO.purchaseContractId.longValue().eq(qdo.id.longValue()))
                .leftJoin(saleConContractDO).on(qdo.relatedSalesContract.longValue().eq(saleConContractDO.id.longValue()))
                .leftJoin(projectDO).on(qdo.relatedProjectId.longValue().eq(projectDO.id.longValue()))
                .leftJoin(prdProductDO).on(detailsDO.relatedProductId.longValue().eq(prdProductDO.id.longValue()))
                .leftJoin(classDO).on(prdProductDO.classId.longValue().eq(classDO.id.longValue()));
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<PurchaseContractManagerVO> getJpaQueryWhere(PurchaseContractManagerQuery query) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(PurchaseContractManagerQuery query) {
        long total = getJpaQueryWhere(query).fetch().size();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(PurchaseContractManagerQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }

        if (!CollectionUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 采购合同编号 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseContractNo())) {
            list.add(qdo.purchaseContractNo.like(SqlUtil.toSqlLikeString(query.getPurchaseContractNo())));
        }
        /** 采购合同名称 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseContractName())) {
            list.add(qdo.purchaseContractName.like(SqlUtil.toSqlLikeString(query.getPurchaseContractName())).or(qdo.purchaseContractNo.like(SqlUtil.toSqlLikeString(query.getPurchaseContractName()))));
        }
        if (!ObjectUtils.isEmpty(query.getPurchaseContractNameEq())) {
            list.add(qdo.purchaseContractName.eq(query.getPurchaseContractNameEq()));
        }
        /** 平台合同类型 udc[salecon:plat_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getPlatType())) {
            list.add(qdo.platType.eq(query.getPlatType()));
        }
        /** 采购合同类型 udc[purchase:con_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseContractType())) {
            list.add(qdo.purchaseContractType.eq(query.getPurchaseContractType()));
        }
        /** 业务类型 udc[purchase:business_type] 精确 */
        if (!ObjectUtils.isEmpty(query.getBusinessType())) {
            list.add(qdo.businessType.eq(query.getBusinessType()));
        }
        /** 验收方式 udc[purchase:check] 精确 */
        if (!ObjectUtils.isEmpty(query.getAcceptanceType())) {
            list.add(qdo.acceptanceType.eq(query.getAcceptanceType()));
        }
        /** 采购公司地址簿Id 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseCompanyBookId())) {
            list.add(qdo.purchaseCompanyBookId.eq(query.getPurchaseCompanyBookId()));
        }
        /** 供应商地址簿ID 精确 */
        if (!ObjectUtils.isEmpty(query.getSupplierBookId())) {
            list.add(qdo.supplierBookId.eq(query.getSupplierBookId()));
        }
        /** 采购BU 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseBuId())) {
            list.add(qdo.purchaseBuId.eq(query.getPurchaseBuId()));
        }
        /** 采购负责人 精确 */
        if (!ObjectUtils.isEmpty(query.getPurchaseInchargeResId())) {
            list.add(qdo.purchaseInchargeResId.eq(query.getPurchaseInchargeResId()));
        }
        /** 签约日期 精确 */
        if (!ObjectUtils.isEmpty(query.getSignDateStart())) {
            list.add(qdo.signDate.goe(query.getSignDateStart()));
        }
        /** 签约日期 精确 */
        if (!ObjectUtils.isEmpty(query.getSignDateEnd())) {
            list.add(qdo.signDate.loe(query.getSignDateEnd()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplicationDateStart())) {
            list.add(qdo.applicationDate.goe(query.getApplicationDateStart()));
        }
        /** 申请日期 精确 */
        if (!ObjectUtils.isEmpty(query.getApplicationDateEnd())) {
            list.add(qdo.applicationDate.loe(query.getApplicationDateEnd()));
        }
        /** 币种 udc[SYSTEM_BASIC:CURRENCY] 精确 */
        if (!ObjectUtils.isEmpty(query.getCurrCode())) {
            list.add(qdo.currCode.eq(query.getCurrCode()));
        }
        /** 金额 精确 */
        if (!ObjectUtils.isEmpty(query.getAmt())) {
            list.add(qdo.amt.eq(query.getAmt()));
        }
        /** 税率 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxRate())) {
            list.add(qdo.taxRate.eq(query.getTaxRate()));
        }
        /** 税额 精确 */
        if (!ObjectUtils.isEmpty(query.getTaxAmt())) {
            list.add(qdo.taxAmt.eq(query.getTaxAmt()));
        }
        /** 合同状态 精确 */
        if (!ObjectUtils.isEmpty(query.getContractStatus())) {
            list.add(qdo.contractStatus.eq(query.getContractStatus()));
        }
        /** 合同状态 精确 */
        if (!ObjectUtils.isEmpty(query.getContractStatusList())) {
            list.add(qdo.contractStatus.in(query.getContractStatusList()));
        }
        /** 审批状态 精确 */
        if (!ObjectUtils.isEmpty(query.getApprStatus())) {
            list.add(qdo.apprStatus.eq(query.getApprStatus()));
        }
        /** 关联销售合同 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedSalesContract())) {
            list.add(qdo.relatedSalesContract.eq(query.getRelatedSalesContract()));
        }
        /** 关联协议 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedAgreement())) {
            list.add(qdo.relatedAgreement.eq(query.getRelatedAgreement()));
        }
        /** 需求编号 精确 */
        if (!ObjectUtils.isEmpty(query.getDemandNo())) {
            list.add(qdo.demandNo.eq(query.getDemandNo()));
        }
        /** 比价资料 精确 */
        if (!ObjectUtils.isEmpty(query.getPricecompereFileCodes())) {
            list.add(qdo.pricecompereFileCodes.eq(query.getPricecompereFileCodes()));
        }
        /** 合同附件 精确 */
        if (!ObjectUtils.isEmpty(query.getContractFileCodes())) {
            list.add(qdo.contractFileCodes.eq(query.getContractFileCodes()));
        }
        /** 盖章附件 精确 */
        if (!ObjectUtils.isEmpty(query.getStampFileCodes())) {
            list.add(qdo.stampFileCodes.eq(query.getStampFileCodes()));
        }
        /** 项目ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedProjectId())) {
            list.add(qdo.relatedProjectId.eq(query.getRelatedProjectId()));
        }
        /** 项目ID 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedProjectIdList())) {
            list.add(qdo.relatedProjectId.in(query.getRelatedProjectIdList()));
        }
        /** 项目名称 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedProjectName())) {
            list.add(qdo.relatedProjectName.like(SqlUtil.toSqlLikeString(query.getRelatedProjectName())));
        }
        /** 任务包 精确 */
        if (!ObjectUtils.isEmpty(query.getRelatedTask())) {
            list.add(qdo.relatedTask.eq(query.getRelatedTask()));
        }
        /** 开票方 精确 */
        if (!ObjectUtils.isEmpty(query.getInvoice())) {
            list.add(qdo.invoice.eq(query.getInvoice()));
        }
        /** 付款方式 精确 */
        if (!ObjectUtils.isEmpty(query.getPayMethod())) {
            list.add(qdo.payMethod.eq(query.getPayMethod()));
        }
        /** 采购来源 精确 */
        if (!ObjectUtils.isEmpty(query.getContractSource())) {
            list.add(qdo.contractSource.eq(query.getContractSource()));
        }
        /** 采购来源编号 精确 */
        if (!ObjectUtils.isEmpty(query.getContractSourceNo())) {
            list.add(qdo.contractSourceNo.like(SqlUtil.toSqlLikeString(query.getContractSourceNo())));
        }
        /** 激活时间 精确 */
        if (!ObjectUtils.isEmpty(query.getActivateDate())) {
            list.add(qdo.activateDate.eq(query.getActivateDate()));
        }
        /** 终止原因 精确 */
        if (!ObjectUtils.isEmpty(query.getOverWhy())) {
            list.add(qdo.overWhy.like(SqlUtil.toSqlLikeString(query.getOverWhy())));
        }
        /** 终止时间 精确 */
        if (!ObjectUtils.isEmpty(query.getOverDate())) {
            list.add(qdo.overDate.eq(query.getOverDate()));
        }
        /** 合同归档标志，0：未归档，1：归档 精确 */
        if (!ObjectUtils.isEmpty(query.getFilingFlag())) {
            list.add(qdo.filingFlag.eq(query.getFilingFlag()));
        }
        /**
         * 采购大类
         */
        if (!ObjectUtils.isEmpty(query.getClassId())) {
            list.add(classDO.id.eq(query.getClassId()));
        }
        if (!ObjectUtils.isEmpty(query.getContractNmNo())) {
            list.add(saleConContractDO.name.like(SqlUtil.toSqlLikeString(query.getContractNmNo())).or(saleConContractDO.code.like(SqlUtil.toSqlLikeString(query.getContractNmNo()))));
        }
        if (!ObjectUtils.isEmpty(query.getProjNmNo())) {
            list.add(projectDO.projName.like(SqlUtil.toSqlLikeString(query.getProjNmNo())).or(projectDO.projNo.like(SqlUtil.toSqlLikeString(query.getProjNmNo()))));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public PurchaseContractManagerVO queryByKey(Long id) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 根据主键集合查询
     *
     * @param ids 主键集合
     * @return 结果
     */
    public List<PurchaseContractManagerVO> queryByKeys(List<Long> ids) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.in(ids));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }


    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<PurchaseContractManagerVO> queryListDynamic(PurchaseContractManagerQuery query) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<PurchaseContractManagerVO> queryPaging(PurchaseContractManagerQuery query) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = getJpaQueryWhere(query);
        List<PurchaseContractManagerVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<PurchaseContractManagerVO>builder().records(result).total(jpaQuery.fetchCount()).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public PurchaseContractManagerDO save(PurchaseContractManagerDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<PurchaseContractManagerDO> saveAll(List<PurchaseContractManagerDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(PurchaseContractManagerPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 采购合同编号
        if (payload.getPurchaseContractNo() != null) {
            update.set(qdo.purchaseContractNo, payload.getPurchaseContractNo());
        }
        // 采购合同名称
        if (payload.getPurchaseContractName() != null) {
            update.set(qdo.purchaseContractName, payload.getPurchaseContractName());
        }
        // 平台合同类型 udc[salecon:plat_type]
        if (payload.getPlatType() != null) {
            update.set(qdo.platType, payload.getPlatType());
        }
        // 采购合同类型 udc[purchase:con_type]
        if (payload.getPurchaseContractType() != null) {
            update.set(qdo.purchaseContractType, payload.getPurchaseContractType());
        }
        // 业务类型 udc[purchase:business_type]
        if (payload.getBusinessType() != null) {
            update.set(qdo.businessType, payload.getBusinessType());
        }
        // 验收方式 udc[purchase:check]
        if (payload.getAcceptanceType() != null) {
            update.set(qdo.acceptanceType, payload.getAcceptanceType());
        }
        // 采购公司/法人号
        if (payload.getPurchaseCompanyBookId() != null) {
            update.set(qdo.purchaseCompanyBookId, payload.getPurchaseCompanyBookId());
        }
        // 供应商BookID
        if (payload.getSupplierBookId() != null) {
            update.set(qdo.supplierBookId, payload.getSupplierBookId());
        }
        // 采购BU
        if (payload.getPurchaseBuId() != null) {
            update.set(qdo.purchaseBuId, payload.getPurchaseBuId());
        }
        // 采购负责人
        if (payload.getPurchaseInchargeResId() != null) {
            update.set(qdo.purchaseInchargeResId, payload.getPurchaseInchargeResId());
        }
        // 签约日期
        if (payload.getSignDate() != null) {
            update.set(qdo.signDate, payload.getSignDate());
        }
        // 申请日期
        if (payload.getApplicationDate() != null) {
            update.set(qdo.applicationDate, payload.getApplicationDate());
        }
        // 币种 udc[SYSTEM_BASIC:CURRENCY]
        if (payload.getCurrCode() != null) {
            update.set(qdo.currCode, payload.getCurrCode());
        }
        // 金额
        if (payload.getAmt() != null) {
            update.set(qdo.amt, payload.getAmt());
        }
        // 税率
        if (payload.getTaxRate() != null) {
            update.set(qdo.taxRate, payload.getTaxRate());
        }
        // 税额
        if (payload.getTaxAmt() != null) {
            update.set(qdo.taxAmt, payload.getTaxAmt());
        }
        // 合同状态
        if (payload.getContractStatus() != null) {
            update.set(qdo.contractStatus, payload.getContractStatus());
        }
        // 审批状态
        if (payload.getApprStatus() != null) {
            update.set(qdo.apprStatus, payload.getApprStatus());
        }
        // 关联销售合同
        if (payload.getRelatedSalesContract() != null) {
            update.set(qdo.relatedSalesContract, payload.getRelatedSalesContract());
        }
        // 关联协议
        if (payload.getRelatedAgreement() != null) {
            update.set(qdo.relatedAgreement, payload.getRelatedAgreement());
        }
        // 比价资料
        if (payload.getPricecompereFileCodes() != null) {
            update.set(qdo.pricecompereFileCodes, payload.getPricecompereFileCodes());
        }
        // 合同附件
        if (payload.getContractFileCodes() != null) {
            update.set(qdo.contractFileCodes, payload.getContractFileCodes());
        }
        // 盖章附件
        if (payload.getStampFileCodes() != null) {
            update.set(qdo.stampFileCodes, payload.getStampFileCodes());
        }
        // 项目ID
        if (payload.getRelatedProjectId() != null) {
            update.set(qdo.relatedProjectId, payload.getRelatedProjectId());
        }
        // 项目名称
        if (payload.getRelatedProjectName() != null) {
            update.set(qdo.relatedProjectName, payload.getRelatedProjectName());
        }
        // 任务包
        if (payload.getRelatedTask() != null) {
            update.set(qdo.relatedTask, payload.getRelatedTask());
        }
        // 开票方
        if (payload.getInvoice() != null) {
            update.set(qdo.invoice, payload.getInvoice());
        }
        // 付款方式
        if (payload.getPayMethod() != null) {
            update.set(qdo.payMethod, payload.getPayMethod());
        }
        // 采购来源
        if (payload.getContractSource() != null) {
            update.set(qdo.contractSource, payload.getContractSource());
        }
        // 采购来源编号
        if (payload.getContractSourceNo() != null) {
            update.set(qdo.contractSourceNo, payload.getContractSourceNo());
        }
        // 激活时间
        if (payload.getActivateDate() != null) {
            update.set(qdo.activateDate, payload.getActivateDate());
        }
        // 终止原因
        if (payload.getOverWhy() != null) {
            update.set(qdo.overWhy, payload.getOverWhy());
        }
        // 终止时间
        if (payload.getOverDate() != null) {
            update.set(qdo.overDate, payload.getOverDate());
        }
        // 合同归档标志，0：未归档，1：归档
        if (payload.getFilingFlag() != null) {
            update.set(qdo.filingFlag, payload.getFilingFlag());
        }
        if (payload.getCloseReason() != null) {
            update.set(qdo.closeReason, payload.getCloseReason());
        }
        if (payload.getCloseDate() != null) {
            update.set(qdo.closeDate, payload.getCloseDate());
        }
        if (payload.getDeleteFlag() != null) {
            update.set(qdo.deleteFlag, payload.getDeleteFlag());
        }
        if (payload.getProcInstId() != null) {
            update.set(qdo.procInstId, payload.getProcInstId());
        }
        if (payload.getProcInstStatus() != null) {
            update.set(qdo.procInstStatus, payload.getProcInstStatus());
        }
        if (payload.getSubmitTime() != null) {
            update.set(qdo.submitTime, payload.getSubmitTime());
        }
        if (payload.getApprovedTime() != null) {
            update.set(qdo.approvedTime, payload.getApprovedTime());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 采购合同编号
            if (nullFields.contains("purchaseContractNo")) {
                update.setNull(qdo.purchaseContractNo);
            }
            // 采购合同名称
            if (nullFields.contains("purchaseContractName")) {
                update.setNull(qdo.purchaseContractName);
            }
            // 平台合同类型 udc[salecon:plat_type]
            if (nullFields.contains("platType")) {
                update.setNull(qdo.platType);
            }
            // 采购合同类型 udc[purchase:con_type]
            if (nullFields.contains("purchaseContractType")) {
                update.setNull(qdo.purchaseContractType);
            }
            // 业务类型 udc[purchase:business_type]
            if (nullFields.contains("businessType")) {
                update.setNull(qdo.businessType);
            }
            // 验收方式 udc[purchase:check]
            if (nullFields.contains("acceptanceType")) {
                update.setNull(qdo.acceptanceType);
            }
            // 采购公司BookId
            if (nullFields.contains("purchaseCompanyBookId")) {
                update.setNull(qdo.purchaseCompanyBookId);
            }
            // 供应商BookID
            if (nullFields.contains("supplierBookId")) {
                update.setNull(qdo.supplierBookId);
            }
            // 采购BU
            if (nullFields.contains("purchaseBuId")) {
                update.setNull(qdo.purchaseBuId);
            }
            // 采购负责人
            if (nullFields.contains("purchaseInchargeResId")) {
                update.setNull(qdo.purchaseInchargeResId);
            }
            // 签约日期
            if (nullFields.contains("signDate")) {
                update.setNull(qdo.signDate);
            }
            // 申请日期
            if (nullFields.contains("applicationDate")) {
                update.setNull(qdo.applicationDate);
            }
            // 币种 udc[SYSTEM_BASIC:CURRENCY]
            if (nullFields.contains("currCode")) {
                update.setNull(qdo.currCode);
            }
            // 金额
            if (nullFields.contains("amt")) {
                update.setNull(qdo.amt);
            }
            // 税率
            if (nullFields.contains("taxRate")) {
                update.setNull(qdo.taxRate);
            }
            // 税额
            if (nullFields.contains("taxAmt")) {
                update.setNull(qdo.taxAmt);
            }
            // 合同状态
            if (nullFields.contains("contractStatus")) {
                update.setNull(qdo.contractStatus);
            }
            // 审批状态
            if (nullFields.contains("apprStatus")) {
                update.setNull(qdo.apprStatus);
            }
            // 关联销售合同
            if (nullFields.contains("relatedSalesContract")) {
                update.setNull(qdo.relatedSalesContract);
            }
            // 关联协议
            if (nullFields.contains("relatedAgreement")) {
                update.setNull(qdo.relatedAgreement);
            }
            // 需求编号
            if (nullFields.contains("demandNo")) {
                update.setNull(qdo.demandNo);
            }
            // 比价资料
            if (nullFields.contains("pricecompereFileCodes")) {
                update.setNull(qdo.pricecompereFileCodes);
            }
            // 合同附件
            if (nullFields.contains("contractFileCodes")) {
                update.setNull(qdo.contractFileCodes);
            }
            // 盖章附件
            if (nullFields.contains("stampFileCodes")) {
                update.setNull(qdo.stampFileCodes);
            }
            // 项目ID
            if (nullFields.contains("relatedProjectId")) {
                update.setNull(qdo.relatedProjectId);
            }
            // 项目名称
            if (nullFields.contains("relatedProjectName")) {
                update.setNull(qdo.relatedProjectName);
            }
            // 任务包
            if (nullFields.contains("relatedTask")) {
                update.setNull(qdo.relatedTask);
            }
            // 开票方
            if (nullFields.contains("invoice")) {
                update.setNull(qdo.invoice);
            }
            // 付款方式
            if (nullFields.contains("payMethod")) {
                update.setNull(qdo.payMethod);
            }
            // 采购来源
            if (nullFields.contains("contractSource")) {
                update.setNull(qdo.contractSource);
            }
            // 采购来源编号
            if (nullFields.contains("contractSourceNo")) {
                update.setNull(qdo.contractSourceNo);
            }
            // 激活时间
            if (nullFields.contains("activateDate")) {
                update.setNull(qdo.activateDate);
            }
            // 终止原因
            if (nullFields.contains("overWhy")) {
                update.setNull(qdo.overWhy);
            }
            // 终止时间
            if (nullFields.contains("overDate")) {
                update.setNull(qdo.overDate);
            }
            // 合同归档标志，0：未归档，1：归档
            if (nullFields.contains("filingFlag")) {
                update.setNull(qdo.filingFlag);
            }
            if (nullFields.contains("closeReason")) {
                update.setNull(qdo.closeReason);
            }
            if (nullFields.contains("closeDate")) {
                update.setNull(qdo.closeDate);
            }
            if (nullFields.contains("deleteFlag")) {
                update.setNull(qdo.deleteFlag);
            }
            // 流程实例ID
            if (nullFields.contains("procInstId")) {
                update.setNull(qdo.procInstId);
            }
            // 流程审批状态
            if (nullFields.contains("procInstStatus")) {
                update.setNull(qdo.procInstStatus);
            }
            // 审批时间
            if (nullFields.contains("approvedTime")) {
                update.setNull(qdo.approvedTime);
            }
            if (nullFields.contains("submitTime")) {
                update.setNull(qdo.submitTime);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }
    /**
     * 修改状态
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long updateStatusByKeys(List<Long> keys, String status) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.contractStatus, status)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 修改状态
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long active(List<Long> keys, String status, LocalDateTime activateDate) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.contractStatus, status)
                .set(qdo.activateDate, activateDate)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 合同归档
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long updateFilingFlag(List<Long> keys, Integer filingFlag) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.filingFlag, filingFlag)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 修改状态
     *
     * @param id
     * @return 删除的行数
     */
    public long updateContractStatus(Long id, String contractStatus) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.contractStatus, contractStatus)
                .where(qdo.id.eq(id));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 修改状态
     *
     * @param ids
     * @return
     */
    public long closeContract(List<Long> ids, String closeReason) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.closeReason, closeReason)
                .set(qdo.closeDate, LocalDate.now())
                .set(qdo.contractStatus, PurchaseContractStatusEnum.CLOSE.getCode())
                .where(qdo.id.in(ids));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

    /**
     * 查询采购合同前置数据
     * @param docNo 采购合同编号
     * @return 采购合同前置数据
     */
    public PurchaseContractDocumentView findByPurchaseContractNo(String docNo) {
        JPAQuery<PurchaseContractDocumentView> jpaQuery = jpaQueryFactory.select(Projections.bean(PurchaseContractDocumentView.class,
                        qdo.id,
                        // 采购合同编号
                        qdo.purchaseContractNo,
                        // 采购合同名称
                        qdo.purchaseContractName,
                        // 类型
                        qdo.purchaseContractType,
                        // 验收方式
                        qdo.acceptanceType,
                        // 采购公司BookId
                        qdo.purchaseCompanyBookId,
                        // 供应商BookId
                        qdo.supplierBookId,
                        // 币种
                        qdo.currCode,
                        // 关联销售合同Id
                        qdo.relatedSalesContract,
                        // 销售合同编号
                        saleConContractDO.code.as("saleConNo"),
                        // 关联项目ID
                        qdo.relatedProjectId,
                        // 项目编号
                        projectDO.projNo,
                        // 任务包
                        qdo.relatedTask,
                        // 付款公司编码
                        qdo.demandNo,
                        // 付款方式
                        qdo.payMethod,
                        // 收款人/单位BookId（供应商BookId）
                        qdo.supplierBookId.as("receivingUnitBookId")
                )).from(qdo)
                .leftJoin(saleConContractDO).on(qdo.relatedSalesContract.eq(saleConContractDO.id))
                .leftJoin(projectDO).on(qdo.relatedProjectId.eq(projectDO.id))
                .where(qdo.purchaseContractNo.eq(docNo));
        return jpaQuery.fetchFirst();
    }

    /**
     * 通过合同编号查询合同Id
     * @param docNo 合同编号
     * @return 合同Id
     */
    public Long findIdByNo(String docNo) {
        return jpaQueryFactory.select(qdo.id).from(qdo).where(qdo.deleteFlag.eq(0)
                .and(qdo.purchaseContractNo.eq(docNo))).fetchFirst();
    }
    /**
     * 通过合同编号查询合同详情
     * @param docNo 合同编号
     * @return 合同Id
     */
    public PurchaseContractManagerVO queryByNo(String docNo) {
        return jpaQueryFactory.
                select(Projections.bean(PurchaseContractManagerVO.class,qdo.id,qdo.purchaseContractName,qdo.purchaseContractNo,qdo.amt,qdo.taxRate,qdo.relatedSalesContract,qdo.purchaseContractType)).
                from(qdo).where(qdo.deleteFlag.eq(0)).
                where(qdo.purchaseContractNo.eq(docNo)).
                fetchFirst();
    }

    /**
     * 合同归档
     * @param purConId 归档合同Id
     * @param stampFileCodes 盖章附件
     */
    public void purContractArchiving(Long purConId, String stampFileCodes) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.stampFileCodes, stampFileCodes)
                .where(qdo.id.eq(purConId));
        update.execute();
    }

    /**
     * 根据销售合同id查看采购合同列表
     *
     * @param saleConId 销售合同Id
     * @return 采购合同列表
     */
    public List<PurchaseContractManagerVO> queryBySaleConId(Long saleConId) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = jpaQueryFactory.select(Projections.bean(PurchaseContractManagerVO.class,
                        qdo.purchaseContractNo,
                        qdo.purchaseContractName,
                        qPurchaseContractDetailsDO.taxAmt.as("amt"),
//                        qdo.amt,
                        qdo.taxRate,
                        qdo.purchaseBuId,
                        qdo.supplierBookId,
//                        qConPurchaseDemandDDO.productClass,
                        qPurchaseContractDetailsDO.classId.as("detailProductClassId"),
                        classDO.className.as("detailProductClassName"),
//                        qConPurchaseDemandDDO.productSubClass,
                        qPurchaseContractDetailsDO.subClassId.as("detailSubProductClassId"),
                        subClassDO.className.as("detailSubProductClassName"),
                        qdo.platType,
//                        qConPurchaseDemandDDO.productId,
                        qPurchaseContractDetailsDO.relatedProductId.as("productId"),
                        // 采购产品
                        prdProductDO.prodName.as("productName"),
//                        qConPurchaseDemandDDO.demandSaid,
                        qPurchaseContractDetailsDO.note.as("demandSaid"),
                        qdo.signDate,
                        qdo.id
                )).from(qdo)
//                .leftJoin(qConPurchaseDemandDDO).on(qdo.id.eq(qConPurchaseDemandDDO.purContractId).and(qConPurchaseDemandDDO.deleteFlag.eq(0)))
//                .leftJoin(prdProductDO).on(qConPurchaseDemandDDO.productId.eq(prdProductDO.id))
                // 4.0是这么写的，从而导致数据存在问题，这边5.0这么写
                .leftJoin(qPurchaseContractDetailsDO).on(qdo.id.eq(qPurchaseContractDetailsDO.purchaseContractId).and(qPurchaseContractDetailsDO.deleteFlag.eq(0)))
                .leftJoin(prdProductDO).on(qPurchaseContractDetailsDO.relatedProductId.eq(prdProductDO.id))
                .leftJoin(classDO).on(qPurchaseContractDetailsDO.classId.eq(classDO.id))
                .leftJoin(subClassDO).on(qPurchaseContractDetailsDO.subClassId.eq(subClassDO.id))
                .where(qdo.relatedSalesContract.eq(saleConId)
                        .and(qdo.deleteFlag.eq(0))
                        .and(qdo.purchaseContractType.in(Arrays.asList(PurchaseContractEnum.PurchaseConType.PURCHASING_AGENT.getCode(),PurchaseContractEnum.PurchaseConType.AGENCY_FEE.getCode(),PurchaseContractEnum.PurchaseConType.OUTSOURCING.getCode())))
                );
        return jpaQuery.fetch();
    }



    /**
     * 查询采购合同详情 （已使用金额）
     * @param query
     * @return 合同列表
     */
    public List<PurchaseContractManagerVO> queryPurContractInfo(PurchaseContractManagerQuery query) {
        JPAQuery<PurchaseContractManagerVO> jpaQuery = jpaQueryFactory.
                select(Projections.bean(PurchaseContractManagerVO.class, qdo.id, qdo.purchaseContractName, qdo.purchaseContractNo, qdo.amt, qdo.taxRate, qdo.relatedSalesContract, qdo.relatedProjectId, qdo.purchaseContractType, qdo.businessType, qPaymentSlipDO.paymentAmt)).
                from(qdo).leftJoin(qPaymentSlipDO).on(qdo.purchaseContractNo.eq(qPaymentSlipDO.docNo).
                        and(qPaymentSlipDO.docType.eq(PurchasePaymentEnum.PaymentDocType.CONTRACT.getCode())).
                        and(qPaymentSlipDO.state.eq(PurchasePaymentEnum.PaymentSlipStatus.PAID.getCode())).
                        and(qPaymentSlipDO.deleteFlag.eq(0)));
        jpaQuery.where(where(query));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetch();
    }
}

