package com.elitesland.tw.tw5.server.prd.file.controller;

import com.elitesland.tw.tw5.api.prd.file.query.PrdFileQuery;
import com.elitesland.tw.tw5.api.prd.file.service.PrdFileService;
import com.elitesland.tw.tw5.api.prd.file.vo.PrdFileVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.TwOutputUtil;
import com.elitesland.tw.tw5.server.common.util.FileUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.io.PathResource;
import org.springframework.core.io.Resource;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import java.io.IOException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;

/**
 * 附件-文件
 *
 * @author duwh
 * @date 2022/09/22
 */
@RestController
@RequiredArgsConstructor
@RequestMapping("/api/file")
@Slf4j
public class PrdFileController {

    private final PrdFileService prdFileService;

    /**
     * 上传文件
     *
     * @param name 名字
     * @param file 文件
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/file/upload")
    public TwOutputUtil upload(@RequestParam(required = false) String name, @RequestParam(required = false) Long folderId, @RequestParam("file") MultipartFile file) {
        final PrdFileVO upload = prdFileService.upload(name, folderId, file);
        return TwOutputUtil.ok(upload);
    }

    /**
     * 上传照片
     *
     * @param folderId 文件夹id
     * @param file     文件
     * @return {@link TwOutputUtil}
     */
    @PostMapping("/file/pictures")
    public TwOutputUtil uploadPicture(@RequestParam(value = "folderId") Long folderId, @RequestParam MultipartFile file) {
        // 判断文件是否为图片
        String suffix = FileUtil.getExtensionName(file.getOriginalFilename());
        if (!FileUtil.IMAGE.equals(FileUtil.getFileType(suffix))) {
            throw TwException.error("0801", "只能上传图片");
        }
        final PrdFileVO prdFileVO = prdFileService.upload(null, folderId, file);
        return TwOutputUtil.ok(prdFileVO);
    }

    /**
     * 更新
     *
     * @param payload 有效载荷
     * @return {@link TwOutputUtil}
     */
    /*@ApiOperation("修改")
    @PutMapping("/file")
    public TwOutputUtil update(PrdFilePayload payload) {
        return TwOutputUtil.ok(prdFileService.update(payload));
    }*/

    /**
     * 主键查询
     *
     * @param key 关键
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/file/{key}")
    public TwOutputUtil queryOneByKey(@PathVariable Long key) {
        PrdFileVO prdFileVO = prdFileService.queryByKey(key);
        return TwOutputUtil.ok(prdFileVO);
    }

    /**
     * 分页
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/file/paging")
    public TwOutputUtil paging(PrdFileQuery query) {
        PagingVO<PrdFileVO> paging = prdFileService.paging(query);
        return TwOutputUtil.ok(paging);
    }

    /**
     * 查询列表
     *
     * @param query 查询
     * @return {@link TwOutputUtil}
     */
    @GetMapping("/file/list")
    public TwOutputUtil queryList(PrdFileQuery query) {
        List<PrdFileVO> vos = prdFileService.queryList(query);
        return TwOutputUtil.ok(vos);
    }

    /**
     * 删除
     *
     * @param keys 主键id
     * @return result
     */
    @DeleteMapping("/file/deleteSoft")
    public TwOutputUtil deleteSoft(Long[] keys) {
        prdFileService.deleteSoft(Arrays.asList(keys));
        return TwOutputUtil.ok();
    }

    /**
     * 预览
     *
     * @param key 关键
     * @return {@link ResponseEntity}
     */
    @GetMapping("/file/preview/{key}")
    public ResponseEntity preview(@PathVariable Long key) {
        final PrdFileVO prdFileVO = prdFileService.queryByKey(key);
        ResponseEntity responseEntity = ResponseEntity.ok()
            .header(HttpHeaders.CONTENT_TYPE, prdFileVO.getFileType())
            .header(HttpHeaders.CONTENT_LENGTH, String.valueOf(prdFileVO.getFileSize()))
            .header(HttpHeaders.CONTENT_DISPOSITION,
                "filename*=" + StandardCharsets.UTF_8 + "''" + prdFileVO.getName())
            .body(new PathResource(prdFileVO.getServerPath()));
        return responseEntity;
    }

    /**
     * 下载
     *
     * @param key 关键
     * @return {@link ResponseEntity}<{@link Resource}>
     * @throws IOException ioexception
     */
    @GetMapping("/file/download/{key}")
    @ApiOperation("附件下载")
    public ResponseEntity<Resource> download(@PathVariable Long key) throws IOException {
        final PrdFileVO prdFileVO = prdFileService.queryByKey(key);
        ResponseEntity responseEntity = ResponseEntity.ok()
            .header(HttpHeaders.CONTENT_TYPE, prdFileVO.getFileType())
            .header(HttpHeaders.CONTENT_LENGTH, String.valueOf(prdFileVO.getFileSize()))
            .header(HttpHeaders.CONTENT_DISPOSITION,
                "attachment; filename=" + URLEncoder.encode(prdFileVO.getName(),StandardCharsets.UTF_8.name()))
            .body(new PathResource(prdFileVO.getServerPath()));
        return responseEntity;
    }



}
