package com.elitesland.tw.tw5.server.prd.acc.dao;

import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.acc.payload.AccReimPayBatchPayload;
import com.elitesland.tw.tw5.api.prd.acc.query.AccReimPayBatchQuery;
import com.elitesland.tw.tw5.api.prd.acc.vo.AccReimPayBatchVO;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitesland.tw.tw5.server.prd.acc.entity.AccReimPayBatchDO;
import com.elitesland.tw.tw5.server.prd.acc.entity.QAccReimPayBatchDO;
import com.elitesland.tw.tw5.server.prd.acc.repo.AccReimPayBatchRepo;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 费用报销记账/付款导出管理
 *
 * @author sunxw
 * @date 2023-12-01
 */
@Repository
@RequiredArgsConstructor
public class AccReimPayBatchDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final AccReimPayBatchRepo repo;
    private final QAccReimPayBatchDO qdo = QAccReimPayBatchDO.accReimPayBatchDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimPayBatchVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(AccReimPayBatchVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 审批批次编号
                qdo.batchNo,
                // 审批批次名称
                qdo.batchName,
                // 批次生成时间
                qdo.batchTime,
                // 批状态，0正常，1作废
                qdo.batchStatus,
                // 批类型 account-财务记账导出，cashier-财务付款导出
                qdo.batchType,
                // 数据最后一次导出时间
                qdo.lastExportTime,
                // 导出次数
                qdo.exportCount,
                // 本批次报销总额
                qdo.batchAmt,
                // 报销批明细条数量
                qdo.batchQty
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<AccReimPayBatchVO> getJpaQueryWhere(AccReimPayBatchQuery query) {
        JPAQuery<AccReimPayBatchVO> jpaQuery = getJpaQuerySelect();
        // 条件封装
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        // 动态排序
        jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
        return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(AccReimPayBatchQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
                .select(qdo.count())
                .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(AccReimPayBatchQuery query) {
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 记录唯一ID集合 精确 */
        if (!ObjectUtils.isEmpty(query.getIds())) {
            list.add(qdo.id.in(query.getIds()));
        }
        /** 审批批次编号 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchNo())) {
            list.add(qdo.batchNo.eq(query.getBatchNo()));
        }
        /** 审批批次名称 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchName())) {
            list.add(qdo.batchName.eq(query.getBatchName()));
        }
        /** 批次生成时间 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchTime())) {
            list.add(qdo.batchTime.eq(query.getBatchTime()));
        }
        /** 批状态，0正常，1作废 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchStatus())) {
            list.add(qdo.batchStatus.eq(query.getBatchStatus()));
        }
        /** 批类型 account-财务记账导出，cashier-财务付款导出 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchType())) {
            list.add(qdo.batchType.eq(query.getBatchType()));
        }
        /** 数据最后一次导出时间 精确 */
        if (!ObjectUtils.isEmpty(query.getLastExportTime())) {
            list.add(qdo.lastExportTime.eq(query.getLastExportTime()));
        }
        /** 导出次数 精确 */
        if (!ObjectUtils.isEmpty(query.getExportCount())) {
            list.add(qdo.exportCount.eq(query.getExportCount()));
        }
        /** 本批次报销总额 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchAmt())) {
            list.add(qdo.batchAmt.eq(query.getBatchAmt()));
        }
        /** 报销批明细条数量 精确 */
        if (!ObjectUtils.isEmpty(query.getBatchQty())) {
            list.add(qdo.batchQty.eq(query.getBatchQty()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public AccReimPayBatchVO queryByKey(Long id) {
        JPAQuery<AccReimPayBatchVO> jpaQuery = getJpaQuerySelect();
        jpaQuery.where(qdo.id.eq(id));
        jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<AccReimPayBatchVO> queryListDynamic(AccReimPayBatchQuery query) {
        JPAQuery<AccReimPayBatchVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<AccReimPayBatchVO> queryPaging(AccReimPayBatchQuery query) {
        long total = count(query);
        if (total == 0) {
            return PagingVO.empty();
        }
        JPAQuery<AccReimPayBatchVO> jpaQuery = getJpaQueryWhere(query);
        List<AccReimPayBatchVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<AccReimPayBatchVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public AccReimPayBatchDO save(AccReimPayBatchDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<AccReimPayBatchDO> saveAll(List<AccReimPayBatchDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(AccReimPayBatchPayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 审批批次编号
        if (payload.getBatchNo() != null) {
            update.set(qdo.batchNo, payload.getBatchNo());
        }
        // 审批批次名称
        if (payload.getBatchName() != null) {
            update.set(qdo.batchName, payload.getBatchName());
        }
        // 批次生成时间
        if (payload.getBatchTime() != null) {
            update.set(qdo.batchTime, payload.getBatchTime());
        }
        // 批状态，0正常，1作废
        if (payload.getBatchStatus() != null) {
            update.set(qdo.batchStatus, payload.getBatchStatus());
        }
        // 批类型 account-财务记账导出，cashier-财务付款导出
        if (payload.getBatchType() != null) {
            update.set(qdo.batchType, payload.getBatchType());
        }
        // 数据最后一次导出时间
        if (payload.getLastExportTime() != null) {
            update.set(qdo.lastExportTime, payload.getLastExportTime());
        }
        // 导出次数
        if (payload.getExportCount() != null) {
            update.set(qdo.exportCount, payload.getExportCount());
        }
        // 本批次报销总额
        if (payload.getBatchAmt() != null) {
            update.set(qdo.batchAmt, payload.getBatchAmt());
        }
        // 报销批明细条数量
        if (payload.getBatchQty() != null) {
            update.set(qdo.batchQty, payload.getBatchQty());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 审批批次编号
            if (nullFields.contains("batchNo")) {
                update.setNull(qdo.batchNo);
            }
            // 审批批次名称
            if (nullFields.contains("batchName")) {
                update.setNull(qdo.batchName);
            }
            // 批次生成时间
            if (nullFields.contains("batchTime")) {
                update.setNull(qdo.batchTime);
            }
            // 批状态，0正常，1作废
            if (nullFields.contains("batchStatus")) {
                update.setNull(qdo.batchStatus);
            }
            // 批类型 account-财务记账导出，cashier-财务付款导出
            if (nullFields.contains("batchType")) {
                update.setNull(qdo.batchType);
            }
            // 数据最后一次导出时间
            if (nullFields.contains("lastExportTime")) {
                update.setNull(qdo.lastExportTime);
            }
            // 导出次数
            if (nullFields.contains("exportCount")) {
                update.setNull(qdo.exportCount);
            }
            // 本批次报销总额
            if (nullFields.contains("batchAmt")) {
                update.setNull(qdo.batchAmt);
            }
            // 报销批明细条数量
            if (nullFields.contains("batchQty")) {
                update.setNull(qdo.batchQty);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
                .set(qdo.deleteFlag, 1)
                .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }

}

