package com.elitesland.tw.tw5.server.prd.borrow.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.borrow.payload.BorrowMoneyPayload;
import com.elitesland.tw.tw5.api.prd.borrow.payload.TransferApplyPayload;
import com.elitesland.tw.tw5.api.prd.borrow.query.BorrowWriteOffQuery;
import com.elitesland.tw.tw5.api.prd.borrow.query.TransferApplyQuery;
import com.elitesland.tw.tw5.api.prd.borrow.service.BorrowMoneyService;
import com.elitesland.tw.tw5.api.prd.borrow.service.BorrowWriteOffService;
import com.elitesland.tw.tw5.api.prd.borrow.service.TransferApplyService;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowMoneyVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.BorrowWriteOffVO;
import com.elitesland.tw.tw5.api.prd.borrow.vo.TransferApplyVO;
import com.elitesland.tw.tw5.api.prd.org.vo.PrdOrgOrganizationRefVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.common.service.TransactionUtilService;
import com.elitesland.tw.tw5.server.common.workFlow.ProcDefKey;
import com.elitesland.tw.tw5.server.prd.borrow.constant.BorrowStatusEnum;
import com.elitesland.tw.tw5.server.prd.borrow.convert.TransferApplyConvert;
import com.elitesland.tw.tw5.server.prd.borrow.dao.TransferApplyDAO;
import com.elitesland.tw.tw5.server.prd.borrow.entity.TransferApplyDO;
import com.elitesland.tw.tw5.server.prd.borrow.repo.TransferApplyRepo;
import com.elitesland.tw.tw5.server.prd.common.CacheUtil;
import com.elitesland.tw.tw5.server.prd.common.WorkflowUtil;
import com.elitesland.workflow.ProcessInfo;
import com.elitesland.workflow.enums.ProcInstStatus;
import com.elitesland.workflow.payload.StartProcessPayload;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.time.LocalDate;
import java.util.HashMap;
import java.util.List;

/**
 * 出让申请
 *
 * @author wangly
 * @date 2024-07-15
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class TransferApplyServiceImpl extends BaseServiceImpl implements TransferApplyService {

    private final TransferApplyRepo transferApplyRepo;
    private final TransferApplyDAO transferApplyDAO;
    @Autowired
    @Lazy
    private BorrowMoneyService borrowMoneyService;
    private final BorrowWriteOffService borrowWriteOffService;
    private final WorkflowUtil workflowUtil;
    private final TransactionUtilService transactionUtilService;

    private final CacheUtil cacheUtil;
    @Override
    public PagingVO<TransferApplyVO> queryPaging(TransferApplyQuery query){
        return transferApplyDAO.queryPaging(query);
    }

    @Override
    public List<TransferApplyVO> queryListDynamic(TransferApplyQuery query){
        return transferApplyDAO.queryListDynamic(query);
    }

    @Override
    public TransferApplyVO queryByKey(Long key) {
        TransferApplyDO entity = transferApplyRepo.findById(key).orElseGet(TransferApplyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        TransferApplyVO transferApplyVO = TransferApplyConvert.INSTANCE.toVo(entity);
        // 查询借款信息
        BorrowMoneyVO borrowMoneyVO = borrowMoneyService.queryByKey(entity.getBorrowId());
        transferApplyVO.setBorrowMoneyVO(borrowMoneyVO);
        // 查询核销的数据
        BorrowWriteOffQuery borrowWriteOffQuery = new BorrowWriteOffQuery();
        borrowWriteOffQuery.setBorrowId(entity.getBorrowId());
        List<BorrowWriteOffVO> borrowWriteOffVOS = borrowWriteOffService.queryListDynamic(borrowWriteOffQuery);
        transferApplyVO.setBorrowWriteOffList(borrowWriteOffVOS);
        return transferApplyVO;
    }

    @Override
    public TransferApplyVO insert(TransferApplyPayload payload) {
        TransferApplyDO transferApplyDO = transactionUtilService.executeWithCallable(() -> transferApplyRepo.save(processData(payload)));
        TransferApplyVO transferApplyVO = TransferApplyConvert.INSTANCE.toVo(transferApplyDO);
        // 发起流程
        activeTransferApplyProc(transferApplyVO);
        return transferApplyVO;
    }

    /**
     * 处理数据
     *
     * @param payload payload
     * @return result
     */
    private TransferApplyDO processData(TransferApplyPayload payload) {
        TransferApplyDO entityDo = TransferApplyConvert.INSTANCE.toDo(payload);
        BorrowMoneyVO borrowMoneyVO = borrowMoneyService.queryByKey(payload.getBorrowId());
        if (borrowMoneyVO == null || !(BorrowStatusEnum.PAID.getCode().equals(borrowMoneyVO.getApprStatus()) || BorrowStatusEnum.WRITE_OFF_PART.getCode().equals(borrowMoneyVO.getApprStatus()))) {
            throw TwException.error("", "只有已付款/部分核销的单据才能发起出让流程！");
        }
        // 受让金额
        BigDecimal amt = borrowMoneyVO.getBorrowAmt() == null ? BigDecimal.ZERO : borrowMoneyVO.getBorrowAmt();
        // 出让人 + 借款金额 + "出让" + 受让人 + "申请单"
        String transferName = cacheUtil.getUserName(borrowMoneyVO.getApplicantUserId()) + amt +
                "出让" + cacheUtil.getUserName(payload.getReceiveUserId()) + "申请单";
        entityDo.setTransferName(transferName);
        entityDo.setTransferUserId(borrowMoneyVO.getRepaymentUserId());
        entityDo.setTransferDate(LocalDate.now());
        entityDo.setTransferBuId(cacheUtil.getDefaultOrgIdByUserId(borrowMoneyVO.getRepaymentUserId()));
        return entityDo;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public TransferApplyVO update(TransferApplyPayload payload) {
        TransferApplyDO entity = transferApplyRepo.findById(payload.getId()).orElseGet(TransferApplyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        TransferApplyDO entityDo = this.processData(payload);
        entity.copy(entityDo);
        return TransferApplyConvert.INSTANCE.toVo(transferApplyRepo.save(entity));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public long updateByKeyDynamic(TransferApplyPayload payload) {
        TransferApplyDO entity = transferApplyRepo.findById(payload.getId()).orElseGet(TransferApplyDO::new);
        Assert.notNull(entity.getId(), "不存在");
        return transferApplyDAO.updateByKeyDynamic(payload);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            transferApplyDAO.deleteSoft(keys);
        }
    }
    private void activeTransferApplyProc(TransferApplyVO transferApplyVO){
        HashMap<String, Object> variables = new HashMap<>();
        String procInstName = transferApplyVO.getTransferName() + "-出让流程";
        // 收让人所属一级BU
        PrdOrgOrganizationRefVO firstBuByBuId = cacheUtil.getBULevel1ByOrgId(transferApplyVO.getReceiveBuId());
        if (firstBuByBuId != null) {
            transferApplyVO.setReceiveFirstBuId(firstBuByBuId.getOrgId());
        }
        // 发起流程
        ProcessInfo processInfo = workflowUtil.simpleStartProcess(StartProcessPayload.of(
                ProcDefKey.DIB_BORROW_MONEY_TRANSFER.name(),
                procInstName,
                transferApplyVO.getId() + "",
                variables), transferApplyVO);
        if (ProcInstStatus.APPROVING.getDesc().equals(processInfo.getProcInstStatus().getDesc())) {
            transactionUtilService.executeWithRunnable(() -> {
                TransferApplyPayload payload = new TransferApplyPayload();
                payload.setId(transferApplyVO.getId());
                BorrowMoneyPayload borrowMoneyPayload = new BorrowMoneyPayload();
                borrowMoneyPayload.setId(transferApplyVO.getBorrowId());
                borrowMoneyPayload.setTransferFlag(true);
                transferApplyDAO.updateByKeyDynamic(payload);
                borrowMoneyService.updateByKeyDynamic(borrowMoneyPayload);
            });
        }


    }
}
