package com.elitesland.tw.tw5.server.prd.cal.service;

import com.elitescloud.boot.core.base.BaseServiceImpl;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalEqvaIncomeLogPayload;
import com.elitesland.tw.tw5.api.prd.cal.payload.CalEqvaIncomePayload;
import com.elitesland.tw.tw5.api.prd.cal.query.CalEqvaIncomeQuery;
import com.elitesland.tw.tw5.api.prd.cal.service.CalEqvaIncomeLogService;
import com.elitesland.tw.tw5.api.prd.cal.service.CalEqvaIncomeService;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalEqvaIncomeLogVO;
import com.elitesland.tw.tw5.api.prd.cal.vo.CalEqvaIncomeVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.cal.convert.CalEqvaIncomeConvert;
import com.elitesland.tw.tw5.server.prd.cal.dao.CalEqvaIncomeDAO;
import com.elitesland.tw.tw5.server.prd.cal.entity.CalEqvaIncomeDO;
import com.elitesland.tw.tw5.server.prd.cal.repo.CalEqvaIncomeRepo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.time.LocalDate;
import java.util.List;

/**
 * 当量收入配置管理
 *
 * @author carl
 * @date 2023-11-07
 */
@Service
@RequiredArgsConstructor
@Slf4j
public class CalEqvaIncomeServiceImpl extends BaseServiceImpl implements CalEqvaIncomeService {

    private final CalEqvaIncomeRepo calEqvaIncomeRepo;
    private final CalEqvaIncomeDAO calEqvaIncomeDAO;
    private final CalEqvaIncomeLogService calEqvaIncomeLogService;

    @Override
    public PagingVO<CalEqvaIncomeVO> queryPaging(CalEqvaIncomeQuery query) {
        PagingVO<CalEqvaIncomeVO> CalEqvaIncomeVOPagingVO = calEqvaIncomeDAO.queryPaging(query);
        if (CalEqvaIncomeVOPagingVO.getTotal() > 0) {
            CalEqvaIncomeVOPagingVO.getRecords().forEach(this::translateData);
        }
        return CalEqvaIncomeVOPagingVO;
    }

    @Override
    public List<CalEqvaIncomeVO> queryListDynamic(CalEqvaIncomeQuery query) {
        return calEqvaIncomeDAO.queryListDynamic(query);
    }

    @Override
    public CalEqvaIncomeVO queryByKey(Long key) {
        CalEqvaIncomeVO calEqvaIncomeVO = calEqvaIncomeDAO.queryByKey(key);
        translateData(calEqvaIncomeVO);
        return calEqvaIncomeVO;
    }

    /**
     * 数据翻译
     *
     * @param CalEqvaIncomeVO
     */
    void translateData(CalEqvaIncomeVO CalEqvaIncomeVO) {
        if (CalEqvaIncomeVO.getFinYear() != null && CalEqvaIncomeVO.getFinPeriod() != null) {
            String finPeriodName = CalEqvaIncomeVO.getFinYear() + "-" + (CalEqvaIncomeVO.getFinPeriod() > 9 ? ("" + CalEqvaIncomeVO.getFinPeriod()) : ("0" + CalEqvaIncomeVO.getFinPeriod()));
            CalEqvaIncomeVO.setFinPeriodName(finPeriodName);
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CalEqvaIncomeVO insertOrUpdate(CalEqvaIncomePayload payload) {
        //数据校验
        checkDate(payload);
        payload.setLineStatus("1");
        payload.setResType("1");
        CalEqvaIncomeDO entityDo = CalEqvaIncomeConvert.INSTANCE.toDo(payload);
        return CalEqvaIncomeConvert.INSTANCE.toVo(calEqvaIncomeRepo.save(entityDo));
    }

    /**
     * 数据校验
     *
     * @param payload
     */
    void checkDate(CalEqvaIncomePayload payload) {
        if (payload.getFinPeriod() != null && payload.getFinYear() == null) {
            throw TwException.error("", "期间存在，年度不可为空");
        }
        if (StringUtils.hasText(payload.getJobType2()) && !StringUtils.hasText(payload.getJobType1())) {
            throw TwException.error("", "工种子类存在，工种不可为空");
        }
        CalEqvaIncomeQuery CalEqvaIncomeQuery = CalEqvaIncomeConvert.INSTANCE.toQuery(payload);
        List<Long> longs = calEqvaIncomeDAO.queryCheckData(CalEqvaIncomeQuery);
        if (!ObjectUtils.isEmpty(longs)) {
            if (payload.getId() == null || longs.size() > 1 || !longs.get(0).equals(payload.getId())) {
                throw TwException.error("", "存在相同收入配置");
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public void deleteSoft(List<Long> keys) {
        if (!keys.isEmpty()) {
            //   calEqvaIncomeDAO.deleteSoft(keys);
        }
    }

    @Override
    public CalEqvaIncomeVO getEqvaSalary(CalEqvaIncomeQuery query) {

        return calEqvaIncomeDAO.getEqvaIncomePlus(query);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public CalEqvaIncomeVO queryExternalEqvaIncome(Long userId) {
        CalEqvaIncomeQuery query = new CalEqvaIncomeQuery();
        query.setResId(userId);
        query.setResType("0");
        query.setLineStatus("1");
        List<CalEqvaIncomeVO> calEqvaIncomeVOS = calEqvaIncomeDAO.queryListDynamic(query);
        if (!ObjectUtils.isEmpty(calEqvaIncomeVOS)) {
            return calEqvaIncomeVOS.get(0);
        }
        return null;
    }

    @Override
    public CalEqvaIncomeVO saveExterEqvaIncome(CalEqvaIncomePayload payload) {
        if (payload.getResId() == null) {
            throw TwException.error("", "外部资源id不可为空");
        }
        CalEqvaIncomeVO calEqvaIncomeVO = null;
        //如果是空的话 先查询资源存不存在 存在的话更新 不存在的话新增 并且向记录表插入一条数据
        if (payload.getId() == null) {
            CalEqvaIncomeQuery query = new CalEqvaIncomeQuery();
            query.setResId(payload.getResId());
            query.setResType("0");
            List<CalEqvaIncomeVO> calEqvaIncomeVOS = calEqvaIncomeDAO.queryListDynamic(query);
            if (!CollectionUtils.isEmpty(calEqvaIncomeVOS)) {
                if (calEqvaIncomeVOS.size() > 1) {
                    throw TwException.error("", "外部资源数据错误，请联系管理员");
                }
                CalEqvaIncomeVO oldCalEqvaIncomeVO = calEqvaIncomeVOS.get(0);
                payload.setId(oldCalEqvaIncomeVO.getId());
                calEqvaIncomeVO = updateExterEqvaIncome(payload);

            } else {
                // 新增
                CalEqvaIncomeDO entityDo = CalEqvaIncomeConvert.INSTANCE.toDo(payload);
                entityDo.setVersion(1);
                entityDo.setResType("0");
                calEqvaIncomeVO = CalEqvaIncomeConvert.INSTANCE.toVo(calEqvaIncomeRepo.save(entityDo));
                // 保存新增变更记录
                savecalEqvaIncomeLog(calEqvaIncomeVO);
                return calEqvaIncomeVO;
            }
        } else {
            //更新 如果是“单位当量收入”，“相关项目”，“结算方式”,"状态修改" 那么要更新记录表数据 并将最新一条插入
            if (payload.getProjId() != null || StringUtils.hasText(payload.getSettleType())
                    || StringUtils.hasText(payload.getLineStatus()) || payload.getPreeqvaAmt() != null) {
                calEqvaIncomeVO = updateExterEqvaIncome(payload);
            } else {
                // 如果只更新了备注/起始时间 更新数据即可 不需要新增变更记录
                calEqvaIncomeDAO.updateByKeyDynamic(payload);
                calEqvaIncomeVO = calEqvaIncomeDAO.queryByKey(payload.getId());
            }
        }
        return calEqvaIncomeVO;
    }

    //更新外部资源当量收入配置
    public CalEqvaIncomeVO updateExterEqvaIncome(CalEqvaIncomePayload payload) {
        CalEqvaIncomeDO entity = calEqvaIncomeRepo.findById(payload.getId()).orElseGet(CalEqvaIncomeDO::new);
        Assert.notNull(entity.getId(), "不存在");
        CalEqvaIncomeDO entityDo = CalEqvaIncomeConvert.INSTANCE.toDo(payload);
        entity.copy(entityDo);
        if (entity.getVersion() == null) {
            entity.setVersion(1);
        } else {
            entity.setVersion(entity.getVersion() + 1);
        }
        boolean updateFlag = true;
        // 如果有效变无效 要更新失效时间 且更新变更记录的失效时间  如果无效变有效 要更新当量收入表 并且在记录表新增一条数据
        if(StringUtils.hasText(payload.getLineStatus())){
            // 变为有效的时候 更新失效时间为空
            if("1".equals(payload.getLineStatus())){
                updateFlag = false;
                entity.setEndDate(null);
            }else{
                // 失效的时候要处理失效时间
                if(payload.getEndDate()!=null){
                    entity.setEndDate(payload.getEndDate());
                }else{
                    entity.setEndDate(LocalDate.now());
                }
            }
        }
        CalEqvaIncomeVO calEqvaIncomeVO = CalEqvaIncomeConvert.INSTANCE.toVo(calEqvaIncomeRepo.save(entity));
        //查询然后更新最近的一条变更记录的失效时间
        CalEqvaIncomeLogVO calEqvaIncomeLogVO = calEqvaIncomeLogService.queryFirstByRelateId(calEqvaIncomeVO.getId());
        if (calEqvaIncomeLogVO != null && updateFlag) {
            CalEqvaIncomeLogPayload calEqvaIncomeLogUpdatePayload = new CalEqvaIncomeLogPayload();
            calEqvaIncomeLogUpdatePayload.setId(calEqvaIncomeLogVO.getId());
            //失效日期为新增的起始日期的前一天 如果起始日期是空 或者 起始日期等于当天 那么失效日期也是当天 或者 状态由有效改为失效
            calEqvaIncomeLogUpdatePayload.setEndDate(LocalDate.now());
            if (payload.getStartDate() != null && !payload.getStartDate().equals(LocalDate.now()) && !updateFlag ) {
                calEqvaIncomeLogUpdatePayload.setEndDate(payload.getStartDate().minusDays(1));
            }
            calEqvaIncomeLogService.update(calEqvaIncomeLogUpdatePayload);
        }
        //保存新增的变更记录 有效变无效的时候不新增数据
        if(!StringUtils.hasText(payload.getLineStatus()) || !updateFlag ){
            savecalEqvaIncomeLog(calEqvaIncomeVO);
        }
        return calEqvaIncomeVO;
    }

    // 保存新增的变更记录
    public void savecalEqvaIncomeLog(CalEqvaIncomeVO calEqvaIncomeVO) {
        CalEqvaIncomeLogPayload calEqvaIncomeLogPayload = new CalEqvaIncomeLogPayload();
        calEqvaIncomeLogPayload.setRemark(calEqvaIncomeVO.getRemark());
        calEqvaIncomeLogPayload.setPreeqvaAmt(calEqvaIncomeVO.getPreeqvaAmt());
        calEqvaIncomeLogPayload.setLineStatus(calEqvaIncomeVO.getLineStatus());
        calEqvaIncomeLogPayload.setProjId(calEqvaIncomeVO.getProjId());
        calEqvaIncomeLogPayload.setProjName(calEqvaIncomeVO.getProjName());
        calEqvaIncomeLogPayload.setResId(calEqvaIncomeVO.getResId());
        calEqvaIncomeLogPayload.setVersion(calEqvaIncomeVO.getVersion());
        calEqvaIncomeLogPayload.setStartDate(calEqvaIncomeVO.getStartDate());
        calEqvaIncomeLogPayload.setSettleType(calEqvaIncomeVO.getSettleType());
        calEqvaIncomeLogPayload.setRelateId(calEqvaIncomeVO.getId());
        calEqvaIncomeLogService.insert(calEqvaIncomeLogPayload);
    }
}
