package com.elitesland.tw.tw5.server.prd.salecon.dao;

import com.elitesland.tw.tw5.api.prd.salecon.payload.ConServicePricePayload;
import com.elitesland.tw.tw5.api.prd.salecon.query.ConServicePriceQuery;
import com.elitesland.tw.tw5.api.prd.salecon.vo.ConServicePriceVO;
import com.elitesland.tw.tw5.server.common.TwException;
import com.elitesland.tw.tw5.server.prd.salecon.entity.ConServicePriceDO;
import com.elitesland.tw.tw5.server.prd.salecon.entity.QConServicePriceDO;
import com.elitesland.tw.tw5.server.prd.salecon.repo.ConServicePriceRepo;
import com.elitesland.tw.tw5.server.common.util.SqlUtil;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.querydsl.core.types.ExpressionUtils;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.RequiredArgsConstructor;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 销售合同-T&M-服务价格
 *
 * @author sam.li
 * @date 2024-05-07
 */
@Repository
@RequiredArgsConstructor
public class ConServicePriceDAO {

    private final JPAQueryFactory jpaQueryFactory;
    private final ConServicePriceRepo repo;
    private final QConServicePriceDO qdo = QConServicePriceDO.conServicePriceDO;

    /**
     * 拼装查询字段
     *
     * @return jpaQuery对象
     */
    private JPAQuery<ConServicePriceVO> getJpaQuerySelect() {
        return jpaQueryFactory.select(Projections.bean(ConServicePriceVO.class,
                qdo.id,
                qdo.remark,
                qdo.createUserId,
                qdo.creator,
                qdo.createTime,
                qdo.modifyUserId,
                qdo.updater,
                qdo.modifyTime,
                // 合同ID
                qdo.saleConId,
                // 结算周期
                qdo.settlementCycle,
                // 结算方式
                qdo.settlementMethod,
                // 角色
                qdo.role,
                // 级别
                qdo.level,
                // 人天单价
                qdo.priceDay
        )).from(qdo);
    }

    /**
     * 拼装查询条件
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    private JPAQuery<ConServicePriceVO> getJpaQueryWhere(ConServicePriceQuery query) {
        JPAQuery<ConServicePriceVO> jpaQuery = getJpaQuerySelect();
            // 条件封装
            jpaQuery.where(where(query));
            // 常用基础查询条件拼装
            SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
            // 动态排序
            jpaQuery.orderBy(SqlUtil.getSortedColumn(qdo, query.getOrders()));
            return jpaQuery;
    }

    /**
     * 统计
     *
     * @param query 查询参数
     * @return jpaQuery对象
     */
    public long count(ConServicePriceQuery query) {
        JPAQuery<Long> jpaQuery = jpaQueryFactory
        .select(qdo.count())
        .from(qdo);
        jpaQuery.where(where(query));
        // 常用基础查询条件拼装
        SqlUtil.handleCommonJpaQuery(jpaQuery, qdo._super, query);
        long total = jpaQuery.fetchOne();
        return total;
    }

    /**
     * 查询条件封装
     *
     * @param query 条件
     * @return {@link Predicate}
     */
    private Predicate where(ConServicePriceQuery query){
        List<Predicate> list = new ArrayList<>();
        /** 记录唯一ID 精确 */
        if (!ObjectUtils.isEmpty(query.getId())) {
            list.add(qdo.id.eq(query.getId()));
        }
        /** 合同ID 精确 */
        if (!ObjectUtils.isEmpty(query.getSaleConId())) {
            list.add(qdo.saleConId.eq(query.getSaleConId()));
        }
        /** 结算周期 精确 */
        if (!ObjectUtils.isEmpty(query.getSettlementCycle())) {
            list.add(qdo.settlementCycle.eq(query.getSettlementCycle()));
        }
        /** 结算方式 精确 */
        if (!ObjectUtils.isEmpty(query.getSettlementMethod())) {
            list.add(qdo.settlementMethod.eq(query.getSettlementMethod()));
        }
        /** 角色 精确 */
        if (!ObjectUtils.isEmpty(query.getRole())) {
            list.add(qdo.role.eq(query.getRole()));
        }
        /** 级别 精确 */
        if (!ObjectUtils.isEmpty(query.getLevel())) {
            list.add(qdo.level.eq(query.getLevel()));
        }
        /** 人天单价 精确 */
        if (!ObjectUtils.isEmpty(query.getPriceDay())) {
            list.add(qdo.priceDay.eq(query.getPriceDay()));
        }
        return ExpressionUtils.allOf(list);
    }

    /**
     * 根据主键查询
     *
     * @param id 主键
     * @return 结果
     */
    public ConServicePriceVO queryByKey(Long id) {
        JPAQuery<ConServicePriceVO> jpaQuery = getJpaQuerySelect();
            jpaQuery.where(qdo.id.eq(id));
            jpaQuery.where(qdo.deleteFlag.eq(0));
        return jpaQuery.fetchFirst();
    }

    /**
     * 动态查询集合
     *
     * @param query 查询参数
     * @return 结果集合
     */
    public List<ConServicePriceVO> queryListDynamic(ConServicePriceQuery query) {
        JPAQuery<ConServicePriceVO> jpaQuery = getJpaQueryWhere(query);
        return jpaQuery.fetch();
    }

    /**
     * 分页查询
     *
     * @param query 查询参数
     * @return 分页结果
     */
    public PagingVO<ConServicePriceVO> queryPaging(ConServicePriceQuery query) {
        long total = count(query);
        if (total == 0) {
             return PagingVO.empty();
        }
        JPAQuery<ConServicePriceVO> jpaQuery = getJpaQueryWhere(query);
        List<ConServicePriceVO> result = jpaQuery
                .offset(query.getPageRequest().getOffset())
                .limit(query.getPageRequest().getPageSize())
                .fetch();
        return PagingVO.<ConServicePriceVO>builder().records(result).total(total).build();
    }

    /**
     * 调用jpa的保存
     *
     * @param ado do对象
     * @return 保存后的对象
     */
    public ConServicePriceDO save(ConServicePriceDO ado) {
        return repo.save(ado);
    }

    /**
     * 调用jpa的保存所有
     *
     * @param dos 多个do对象
     * @return 保存后的对象集合
     */
    public List<ConServicePriceDO> saveAll(List<ConServicePriceDO> dos) {
        return repo.saveAll(dos);
    }

    /**
     * 按主键动态修改（只修非null字段，如果需要将某些字段修改为null，请添加nullFields）
     *
     * @param payload 要修改的对象
     * @return 修改的行数
     */
    @Transactional
    public long updateByKeyDynamic(ConServicePricePayload payload) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
        .where(qdo.id.eq(payload.getId()));
        // 记录唯一ID
        if (payload.getId() != null) {
            update.set(qdo.id, payload.getId());
        }
        // 合同ID
        if (payload.getSaleConId() != null) {
            update.set(qdo.saleConId, payload.getSaleConId());
        }
        // 结算周期
        if (payload.getSettlementCycle() != null) {
            update.set(qdo.settlementCycle, payload.getSettlementCycle());
        }
        // 结算方式
        if (payload.getSettlementMethod() != null) {
            update.set(qdo.settlementMethod, payload.getSettlementMethod());
        }
        // 角色
        if (payload.getRole() != null) {
            update.set(qdo.role, payload.getRole());
        }
        // 级别
        if (payload.getLevel() != null) {
            update.set(qdo.level, payload.getLevel());
        }
        // 人天单价
        if (payload.getPriceDay() != null) {
            update.set(qdo.priceDay, payload.getPriceDay());
        }
        // 处理要设置成空的字段
        List<String> nullFields = payload.getNullFields();
        if (nullFields != null && nullFields.size() > 0) {
            // 记录唯一ID
            if (nullFields.contains("id")) {
                update.setNull(qdo.id);
            }
            // 合同ID
            if (nullFields.contains("saleConId")) {
                update.setNull(qdo.saleConId);
            }
            // 结算周期
            if (nullFields.contains("settlementCycle")) {
                update.setNull(qdo.settlementCycle);
            }
            // 结算方式
            if (nullFields.contains("settlementMethod")) {
                update.setNull(qdo.settlementMethod);
            }
            // 角色
            if (nullFields.contains("role")) {
                update.setNull(qdo.role);
            }
            // 级别
            if (nullFields.contains("level")) {
                update.setNull(qdo.level);
            }
            // 人天单价
            if (nullFields.contains("priceDay")) {
                update.setNull(qdo.priceDay);
            }
        }
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        // 执行修改
        return update.execute();
    }

    /**
     * 逻辑删除
     *
     * @param keys 主集合
     * @return 删除的行数
     */
    public long deleteSoft(List<Long> keys) {
        JPAUpdateClause update = jpaQueryFactory.update(qdo)
            .set(qdo.deleteFlag, 1)
            .where(qdo.id.in(keys));
        //拼装更新
        SqlUtil.updateCommonJpaQuery(update, qdo._super);
        return update.execute();
    }


    /**
     * 通过合同id更新结算周期和方式
     *
     */

    public void updateBySaleConId(Long saleConId, String settlementCycle, String settlementMethod){
        JPAUpdateClause update = jpaQueryFactory.update(qdo);
        if(StringUtils.hasText(settlementCycle)){
            update.set(qdo.settlementCycle, settlementCycle);
        }
        if(StringUtils.hasText(settlementMethod)) {
            update.set(qdo.settlementMethod, settlementMethod);
        }
        update.where(qdo.saleConId.eq(saleConId));
        update.execute();
    }
}

